#!/usr/bin/python
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# This program is base on yumdownloader
#

import sys
import os
import yum
import shutil
import rpmUtils

from yum.packages import parsePackages
from yum.Errors import *
from mic.imgcreate.fs import *
from mic.imgcreate import kickstart


def _best_convert_pkg2srcpkgs(self, pkg):
    if pkg.arch == 'src':
        return [pkg]

    (n,v,r,e,a) = rpmUtils.miscutils.splitFilename(pkg.sourcerpm)
    src = self.pkgSack.searchNevra(name=n, ver=v, rel=r, arch='src')
    if src == []:
        self.logger.error('No source RPM found for %s' % str(pkg))

    return src

class SrcDownload(yum.YumBase):

    def __init__(self):
        yum.YumBase.__init__(self)
        self.destdir = './'
        self.cachedir = '/var/tmp/cache'

    def config(self, ks, cachedir, repo_urls = {}):

        self.cachedir = cachedir

        for repo in kickstart.get_repos(ks, repo_urls):

            (name, baseurl, mirrorlist, inc, exc, proxy, proxy_username, proxy_password, debuginfo, source, gpgkey, disable) = repo

            self.addRepository(name, baseurl, mirrorlist, proxy, proxy_username, proxy_password)

        self.setupSourceRepos()


    def addRepository(self, name, url = None, mirrorlist = None, proxy = None, proxy_username = None, proxy_password = None):
        def _varSubstitute(option):
            # takes a variable and substitutes like yum configs do
            option = option.replace("$basearch", rpmUtils.arch.getBaseArch())
            option = option.replace("$arch", rpmUtils.arch.getCanonArch())
            return option

        repo = yum.yumRepo.YumRepository(name)

        """Set proxy"""
        repo.proxy = proxy
        repo.proxy_username = proxy_username
        repo.proxy_password = proxy_password

        if url:
            repo.baseurl.append(_varSubstitute(url))

        if mirrorlist:
            repo.mirrorlist = _varSubstitute(mirrorlist)
        conf = yum.config.RepoConf()
        for k, v in conf.iteritems():
            if v or not hasattr(repo, k):
                repo.setAttribute(k, v)

        repo.basecachedir = self.cachedir
        repo.failovermethod = "priority"
        repo.metadata_expire = 0
        repo.gpgcheck = 0
        repo.enable()
        repo.setCallback(TextProgress())
        repo.setup(0)
        self.repos.add(repo)
        return repo

    def setupSourceRepos(self):
        # enable the -source repos for enabled primary repos
        archlist = rpmUtils.arch.getArchList() + ['src']
        # Ok, we have src and bin repos. What we want to do here is:
        #
        # 1. _enable_ source repos for which the bin repos are enabled.
        # 2. _disable_ the _other_ src repos.

        # Get all src repos.
        src_repos = {}
        for repo in self.repos.findRepos('*-source'):
            src_repos[repo.id] = False

        #  Find the enabled bin repos, and mark their respective *-source repo.
        # as good.
        for repo in self.repos.listEnabled():
            if repo.id not in src_repos:
                srcrepo = '%s-source' % repo.id
                if srcrepo in src_repos:
                    src_repos[srcrepo] = True

        # Toggle src repos that are set the wrong way
        for repo in self.repos.findRepos('*-source'):
            if  repo.isEnabled() and src_repos[repo.id]:
                repo.setup(0)
                try:
                    # Setup pkgSack with 'src' in the archlist
                    self._getSacks(archlist=archlist, thisrepo=repo.id)
                except yum.Errors.YumBaseError, msg:
                    self.logger.critical(str(msg))
                    sys.exit(1)
        if len(src_repos) == 0:
            raise ConfigError('No source repo found')

    def get(self, packages, instroot, destdir = None):

        if not destdir:
            self.destdir = instroot + '/usr/src/SRPMS'
        else:
            self.destdir = instroot + destdir

        toDownload = []

        for pkgnames in packages:
            toActOn = []
            pos = self.pkgSack.returnPackages(patterns=[pkgnames])
            exactmatch, matched, unmatched = parsePackages(pos, [pkgnames])
            installable = yum.misc.unique(exactmatch + matched)
            if not installable: # doing one at a time, apart from groups
                self.logger.error('No Match for argument %s' % pkgnames)
                continue
            for newpkg in installable:
                toActOn.extend(_best_convert_pkg2srcpkgs(self, newpkg))
            if toActOn:
                pkgGroups = self._groupPackages(toActOn)
                for group in pkgGroups:
                    pkgs = pkgGroups[group]

                    toDownload.extend(self.bestPackagesFromList(pkgs, 'src'))

        if len(toDownload) == 0:
            self.logger.error('Nothing to download')
            sys.exit(1)

        for pkg in toDownload:
            n,a,e,v,r = pkg.pkgtup
            packages =  self.pkgSack.searchNevra(n,e,v,r,a)
            for download in packages:
                repo = self.repos.getRepo(download.repoid)
                remote = download.returnSimple('relativepath')
                local = os.path.basename(remote)
                if not os.path.exists(self.destdir):
                    os.makedirs(self.destdir)
                local = os.path.join(self.destdir, local)
                if (os.path.exists(local) and
                    os.path.getsize(local) == int(download.returnSimple('packagesize'))):
                    self.logger.error("%s already exists and appears to be complete" % local)
                    continue
                # Disable cache otherwise things won't download
                repo.cache = 0
                download.localpath = local # Hack: to set the localpath we want.
                try:
                    checkfunc = (self.verifyPkg, (download, 1), {})
                    path = repo.getPackage(download, checkfunc=checkfunc)
                except IOError, e:
                    self.logger.error("Cannot write to file %s. Error was: %s" % (local, e))
                    continue
                except RepoError, e:
                    self.logger.error("Could not download/verify pkg %s: %s" % (download, e))
                    continue

                if not os.path.exists(local) or not os.path.samefile(path, local):
                    progress = TextMeter()
                    progress.start(basename=os.path.basename(local),
                                   size=os.stat(path).st_size)
                    shutil.copy2(path, local)
                    progress.end(progress.size)

    def _groupPackages(self,pkglist):
        pkgGroups = {}
        for po in pkglist:
            na = '%s.%s' % (po.name,po.arch)
            if not na in pkgGroups:
                pkgGroups[na] = [po]
            else:
                pkgGroups[na].append(po)
        return pkgGroups
