# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is based on the library named PyPlot, originaly developped
#     by Gordon Williams and Jeff Grimmett. Thank you!

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Spectrum calculations.

# load libs
import math


# ----
def getIndex(points, xPos):
    """ Get index for selected point. (by Norton Kitagawa)"""

    lo = 0
    hi = len(points)
    while lo < hi:
        mid = (lo + hi) / 2
        if xPos < points[mid][0]:
            hi = mid
        else:
            lo = mid + 1

    return lo
# ----


# ----
def getPoint(points, xPos):
    """ Get interpolated Y position from X (Interpolated simply from line). """

    if points and xPos:
        pointsLen = len(points)

        # get index of nearest higher point
        index = getIndex(points, xPos)

        # interpolate between two points
        x1 = points[index-1][0]
        y1 = points[index-1][1]
        x2 = points[index][0]
        y2 = points[index][1]
        yPos = y1 + ((xPos - x1) * (y2 - y1)/(x2 - x1))

        # get point values and coordinations
        return [round(xPos, 6), round(yPos, 6), '', 0]

    else:
        return False
# ----


# ----
def getCentroid(points, selection, height):
    """ Get peak centroid at given height. """

    if points and selection and height:
        pointsLen = len(points)
        minX = selection[0]
        maxX = selection[2]

        # get indexes of points in selection
        i1 = getIndex(points, minX)
        i2 = getIndex(points, maxX)
        exactSelection = points[i1:i2+1]

        # get maximum
        i = 0
        val = exactSelection[0][1]
        for x, point in enumerate(exactSelection):
            if point[1] > val:
                val = point[1]
                i = x
        peakMaximum = exactSelection[i]

        # get new selection according to % of maximum
        minY = float(peakMaximum[1]) * float(height)/100
        i1 = 0
        i2 = len(exactSelection)
        for x, point in enumerate(exactSelection):
            if point[0] < peakMaximum[0] and point[1] < minY:
                i1 = x
            elif point[0] > peakMaximum[0] and point[1] < minY:
                i2 = x
                break

        # check selection
        if i1 == 0 or i2 == len(exactSelection):
            return False

        # get mass at half of heightSelection
        leftX = interpolateLine(exactSelection[i1], exactSelection[i1+1], y=minY)
        rightX = interpolateLine(exactSelection[i2-1], exactSelection[i2], y=minY)
        xPos = (leftX + rightX)/2

        # get index of nearest higher point
        i = 0
        for x, point in enumerate(exactSelection):
            if point[0] > xPos:
                i = x
                break

        # interpolate between two points
        yPos = interpolateLine(exactSelection[x-1], exactSelection[x], x=xPos)

        # return point
        return [round(xPos, 6), round(yPos, 6), '', 1, rightX-leftX, minY]

    else:
        return False
# ----


# ----
def interpolateLine(point1, point2, x=None, y=None):
    """ Get interpolated X or Y from line defined by two points (y = mx + b). """

    # get equation
    m = (point2[1] - point1[1])/(point2[0] - point1[0])
    b = point1[1] - m * point1[0]

    # get point
    if x != None:
        return m * x + b
    elif y != None:
        return (y - b) / m
# ----
