# -------------------------------------------------------------------------
#     Copyright (C) 2005-2010 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------


# load libs
import sys
import os
import xml.dom.minidom
from copy import deepcopy


# SET CONFING FOLDER
# ------------------

if sys.platform == 'darwin':
    confdir = 'configs'
    support = os.path.expanduser("~/Library/Application Support/")
    userconf = os.path.join(support,'mMass')
    if os.path.exists(support) and not os.path.exists(userconf):
        try: os.mkdir(userconf)
        except: pass
    if os.path.exists(userconf):
        confdir = userconf

else:
    confdir = os.path.sep
    for folder in os.path.dirname(os.path.realpath(__file__)).split(os.path.sep)[:-1]:
        path = os.path.join(confdir, folder)
        if os.path.isdir(path):
            confdir = path
        if os.path.isfile(path):
            break
    confdir = os.path.join(confdir, 'configs')
    if not os.path.exists(confdir):
        try: os.mkdir(confdir)
        except: pass

if not os.path.exists(confdir):
    raise IOError, "Configuration folder cannot be found!"


# LOAD mspy MODULE
# ----------------

import blocks

if not os.path.exists(os.path.join(confdir,'elements.xml')):
    blocks.saveElements(os.path.join(confdir,'elements.xml'))
if not os.path.exists(os.path.join(confdir,'aminoacids.xml')):
    blocks.saveAminoacids(os.path.join(confdir,'aminoacids.xml'))
if not os.path.exists(os.path.join(confdir,'enzymes.xml')):
    blocks.saveEnzymes(os.path.join(confdir,'enzymes.xml'))
if not os.path.exists(os.path.join(confdir,'fragments.xml')):
    blocks.saveFragments(os.path.join(confdir,'fragments.xml'))
if not os.path.exists(os.path.join(confdir,'modifications.xml')):
    blocks.saveModifications(os.path.join(confdir,'modifications.xml'))

import mspy


# DEFAULT CONFIG VALUES
# ---------------------

import libs

version = '3.9.0'
testing = False

main={
    'appWidth': 1000,
    'appHeight': 550,
    'appMaximized': 0,
    'unlockGUI': 0,
    'mzDigits': 4,
    'intDigits': 0,
    'ppmDigits': 1,
    'lastDir': '',
    'errorUnits': 'Da',
    'printQuality': 5,
    'macListCtrlGeneric': 1,
    'donate': '',
    'updatesEnabled': 1,
    'updatesChecked': '',
    'updatesCurrent': version,
    'updatesAvailable': version,
    'compassMode': 'Profile',
    'compassFormat': 'mzData',
    'compassDeleteFile': 0,
}

recent=[]

colours=[
    [16,71,185],
    [50,140,0],
    [241,144,0],
    [76,199,197],
    [143,143,21],
    [38,122,255],
    [38,143,73],
    [237,187,0],
    [120,109,255],
    [179,78,0],
    [128,191,189],
    [137,136,68],
    [200,136,18],
    [197,202,61],
    [123,182,255],
    [69,67,138],
    [24,129,131],
    [131,129,131],
    [69,126,198],
    [189,193,123],
    [127,34,0],
    [76,78,76],
    [31,74,145],
    [15,78,75],
    [79,26,81],
]

export={
    'imageWidth': 750,
    'imageHeight': 500,
    'imageUnits': 'px',
    'imageResolution': 72,
    'imageFormat': 'PNG',
    'peaklistColumns': ['mz','int'],
    'peaklistFormat': 'ASCII',
    'peaklistSeparator': 'tab',
    'spectrumSeparator': 'tab',
}

spectrum={
    'xLabel': 'm/z',
    'yLabel': 'a.i.',
    'showGrid': 1,
    'showLegend': 1,
    'showPosBar': 1,
    'showIntBar': 1,
    'showGel': 1,
    'showGelLegend': 1,
    'showTracker': 1,
    'showNotationMarks': 1,
    'showLabels': 1,
    'showAllLabels': 1,
    'showTicks': 1,
    'showDataPoints': 1,
    'showCursorImage': 1,
    'barHeight': 7,
    'gelHeight': 19,
    'autoscale': 1,
    'normalize': 0,
    'overlapLabels': 0,
    'checkLimits': 1,
    'mzDigits': 4,
    'intDigits': 1,
    'labelAngle': 90,
    'labelCharge': 1,
    'labelBgr': 1,
    'labelFontSize': 10,
    'axisFontSize': 10,
    'tickColour': [255,75,75],
    'tmpSpectrumColour': [255,0,0],
    'notationMarksColour': [0,255,0],
}

match={
    'tolerance': 0.2,
    'units': 'Da',
    'ignoreCharge': 0,
}

processing={
    'math':{
        'operation': 'add',
        'spectrumB': '',
        'multiplier': 1,
    },
    'crop':{
        'lowMass': 500,
        'highMass': 5000,
    },
    'baseline':{
        'segments': 15,
        'offset': 0,
        'smooth': 1,
    },
    'smoothing':{
        'method': 'SG',
        'windowSize': 0.3,
        'cycles': 2,
    },
    'peakpicking':{
        'snThreshold': 3,
        'absIntThreshold': 0,
        'relIntThreshold': 0,
        'adaptiveNoise': 0,
        'pickingHeight': 0.75,
        'smoothing': 1,
        'deisotoping': 1,
        'monoisotopic': 0,
        'removeShoulders': 0,
    },
    'deisotoping':{
        'maxCharge': 1,
        'massTolerance': 0.1,
        'intTolerance': 0.5,
        'removeIsotopes': 1,
        'removeUnknown': 1,
    },
}

calibration={
    'fitting': 'quadratic',
    'tolerance': 50,
    'units': 'ppm',
    'statCutOff': 800,
}

sequence={
    'digest':{
        'maxMods': 1,
        'maxCharge': 1,
        'massType': 0,
        'enzyme': 'Trypsin',
        'miscl': 1,
        'lowMass': 500,
        'highMass': 5000,
        'retainPos': 0,
        'allowMods': 0,
        'listFormat': 'b.S.a [m]',
        'matchFormat': '[r] b.S.a [m]',
    },
    'fragment':{
        'maxMods': 1,
        'maxCharge': 1,
        'massType': 1,
        'fragments': ['a','b','y','-NH3','-H2O'],
        'filterFragments': 1,
        'listFormat': 'b.S.a [m]',
        'matchFormat': '[f] [r] b.S.a [m]',
    },
    'search':{
        'mass': 0,
        'maxMods': 1,
        'charge': 1,
        'massType': 0,
        'enzyme': 'Trypsin',
        'tolerance': 0.2,
        'units': 'Da',
        'retainPos': 0,
        'listFormat': 'b.S.a [m]',
    },
}

masscalc={
    'ionseriesAgent': 'H',
    'ionseriesAgentCharge': 1,
    'ionseriesPolarity': 1,
    'patternFwhm': 0.1,
    'patternIntensity': 100,
    'patternBaseline': 0,
    'patternShift': 0,
    'patternThreshold': 0.001,
}

compounds={
    'massType': 0,
    'maxCharge': 1,
    'radicals': 0,
    'adducts':  ['Na','K'],
}

differences={
    'aminoacids': 1,
    'dipeptides': 1,
    'massType': 0,
    'tolerance': 0.1,
}

compare={
    'tolerance': 0.2,
    'units': 'Da',
    'ignoreCharge': 0,
    'compare': 'peaklists',
}

mascot={
    'common':{
        'title':'',
        'userName':'',
        'userEmail':'',
        'server': 'Matrix Science',
        'searchType': 'pmf',
    },
    'pmf':{
        'database': 'SwissProt',
        'taxonomy': 'All entries',
        'enzyme': 'Trypsin',
        'miscleavages': 1,
        'fixedMods': [],
        'variableMods': [],
        'hiddenMods': 0,
        'proteinMass': '',
        'peptideTol': 0.1,
        'peptideTolUnits': 'Da',
        'massType': 'Monoisotopic',
        'charge': '1+',
        'decoy': 0,
        'report': 'AUTO',
    },
    'sq':{
        'database': 'SwissProt',
        'taxonomy': 'All entries',
        'enzyme': 'Trypsin',
        'miscleavages': 1,
        'fixedMods': [],
        'variableMods': [],
        'hiddenMods': 0,
        'peptideTol': 0.1,
        'peptideTolUnits': 'Da',
        'msmsTol': 0.2,
        'msmsTolUnits': 'Da',
        'massType': 'Average',
        'charge': '1+',
        'instrument': 'Default',
        'quantitation': 'None',
        'decoy': 0,
        'report': 'AUTO',
    },
    'mis':{
        'database': 'SwissProt',
        'taxonomy': 'All entries',
        'enzyme': 'Trypsin',
        'miscleavages': 1,
        'fixedMods': [],
        'variableMods': [],
        'hiddenMods': 0,
        'peptideMass': '',
        'peptideTol': 0.1,
        'peptideTolUnits': 'Da',
        'msmsTol': 0.2,
        'msmsTolUnits': 'Da',
        'massType': 'Average',
        'charge': '1+',
        'instrument': 'Default',
        'quantitation': 'None',
        'decoy': 0,
        'errorTolerant': 0,
        'report': 'AUTO',
    },
}

profound={
    'script': 'http://prowl.rockefeller.edu/prowl-cgi/profound.exe',
    'title':'',
    'database': 'NCBI nr',
    'taxonomy': 'All taxa',
    'enzyme': 'Trypsin',
    'miscleavages': 1,
    'fixedMods': [],
    'variableMods': [],
    'proteinMassLow': 0,
    'proteinMassHigh': 300,
    'proteinPILow': 0,
    'proteinPIHigh': 14,
    'peptideTol': 0.1,
    'peptideTolUnits': 'Da',
    'massType': 'Monoisotopic',
    'charge': 'MH+',
    'ranking': 'expect',
    'expectation': 1,
    'candidates': 10,
}

prospector={
    'common':{
        'title':'',
        'script': 'http://prospector.ucsf.edu/prospector/cgi-bin/mssearch.cgi',
        'searchType': 'msfit',
    },
    'msfit':{
        'database': 'SwissProt',
        'taxonomy': 'All',
        'enzyme': 'Trypsin',
        'miscleavages': 1,
        'fixedMods': [],
        'variableMods': [],
        'proteinMassLow': 0,
        'proteinMassHigh': 300,
        'proteinPILow': 0,
        'proteinPIHigh': 14,
        'peptideTol': 0.1,
        'peptideTolUnits': 'Da',
        'massType': 'Monoisotopic',
        'instrument': 'MALDI-TOFTOF',
        'minMatches': 4,
        'maxMods': 1,
        'report': 5,
        'pfactor': 0.4,
    },
    'mstag':{
        'database': 'SwissProt',
        'taxonomy': 'All',
        'enzyme': 'Trypsin',
        'miscleavages': 1,
        'fixedMods': [],
        'variableMods': [],
        'peptideMass': '',
        'peptideTol': 0.1,
        'peptideTolUnits': 'Da',
        'peptideCharge': '1',
        'msmsTol': 0.2,
        'msmsTolUnits': 'Da',
        'massType': 'Monoisotopic',
        'instrument': 'MALDI-TOFTOF',
        'maxMods': 1,
        'report': 5,
    },
}

links={
    'mMassHomepage': 'http://www.mmass.org/',
    'mMassForum': 'http://forum.mmass.org/',
    'mMassCite': 'http://www.mmass.org/home/donate.php',
    'mMassDonate': 'http://www.mmass.org/home/donate.php',
    'mMassDownload': 'http://www.mmass.org/download/',
    
    'biomedmstools': 'http://ms.biomed.cas.cz/MSTools/',
    'blast': 'http://www.ebi.ac.uk/Tools/blastall/',
    'clustalw': 'http://www.ebi.ac.uk/Tools/clustalw/',
    'deltamass': 'http://www.abrf.org/index.cfm/dm.home',
    'emblebi': 'http://www.ebi.ac.uk/services/',
    'expasy': 'http://www.expasy.org/',
    'fasta': 'http://www.ebi.ac.uk/Tools/fasta33/',
    'matrixscience': 'http://www.matrixscience.com/',
    'muscle': 'http://phylogenomics.berkeley.edu/cgi-bin/muscle/input_muscle.py',
    'ncbi': 'http://www.ncbi.nlm.nih.gov/Entrez/',
    'pdb': 'http://www.rcsb.org/pdb/',
    'pir': 'http://pir.georgetown.edu/',
    'profound': 'http://prowl.rockefeller.edu/prowl-cgi/profound.exe',
    'prospector': 'http://prospector.ucsf.edu/',
    'unimod': 'http://www.unimod.org/',
    'uniprot': 'http://www.uniprot.org/',
}

replacements={
    'LipidMaps.org':{
        'pattern': 'LM[A-Z]{2}[0-9]{4}[0-9A-Z]{2}[0-9]{2}',
        'url': 'http://www.lipidmaps.org/data/LMSDRecord.php?LMID=%s',
    }
}


# LOAD FUNCTIONS
# --------------

def loadConfig(path=os.path.join(confdir,'config.xml')):
    """Parse config XML and get data."""
    
    # parse XML
    document = xml.dom.minidom.parse(path)
    
    # main
    mainTags = document.getElementsByTagName('main')
    if mainTags:
        _getParams(mainTags[0], main)
    
    # recent files
    recentTags = document.getElementsByTagName('recent')
    if recentTags:
        pathTags = recentTags[0].getElementsByTagName('path')
        if pathTags:
            del recent[:]
            for pathTag in pathTags:
                recent.append(pathTag.getAttribute('value'))
    
    # colours
    coloursTags = document.getElementsByTagName('colours')
    if coloursTags:
        colourTags = coloursTags[0].getElementsByTagName('colour')
        if colourTags:
            del colours[:]
            for colourTag in colourTags:
                col = colourTag.getAttribute('value')
                colours.append([int(c, 16) for c in (col[0:2], col[2:4], col[4:6])])
    
    # export
    exportTags = document.getElementsByTagName('export')
    if exportTags:
        _getParams(exportTags[0], export)
        
        if type(export['peaklistColumns']) != list:
            export['peaklistColumns'] = export['peaklistColumns'].split(';')
    
    # spectrum
    spectrumTags = document.getElementsByTagName('spectrum')
    if spectrumTags:
        _getParams(spectrumTags[0], spectrum)
        
        if type(spectrum['tickColour']) != list:
            col = spectrum['tickColour']
            spectrum['tickColour'] = [int(c, 16) for c in (col[0:2], col[2:4], col[4:6])]
        
        if type(spectrum['tmpSpectrumColour']) != list:
            col = spectrum['tmpSpectrumColour']
            spectrum['tmpSpectrumColour'] = [int(c, 16) for c in (col[0:2], col[2:4], col[4:6])]
        
        if type(spectrum['notationMarksColour']) != list:
            col = spectrum['notationMarksColour']
            spectrum['notationMarksColour'] = [int(c, 16) for c in (col[0:2], col[2:4], col[4:6])]
    
    # match
    matchTags = document.getElementsByTagName('match')
    if matchTags:
        _getParams(matchTags[0], match)
    
    # processing
    processingTags = document.getElementsByTagName('processing')
    if processingTags:
        
        cropTags = processingTags[0].getElementsByTagName('crop')
        if cropTags:
            _getParams(cropTags[0], processing['crop'])
        
        baselineTags = processingTags[0].getElementsByTagName('baseline')
        if baselineTags:
            _getParams(baselineTags[0], processing['baseline'])
        
        smoothingTags = processingTags[0].getElementsByTagName('smoothing')
        if smoothingTags:
            _getParams(smoothingTags[0], processing['smoothing'])
        
        peakpickingTags = processingTags[0].getElementsByTagName('peakpicking')
        if peakpickingTags:
            _getParams(peakpickingTags[0], processing['peakpicking'])
        
        deisotopingTags = processingTags[0].getElementsByTagName('deisotoping')
        if deisotopingTags:
            _getParams(deisotopingTags[0], processing['deisotoping'])
    
    # calibration
    calibrationTags = document.getElementsByTagName('calibration')
    if calibrationTags:
        _getParams(calibrationTags[0], calibration)
    
    # sequence
    sequenceTags = document.getElementsByTagName('sequence')
    if sequenceTags:
        
        digestTags = sequenceTags[0].getElementsByTagName('digest')
        if digestTags:
            _getParams(digestTags[0], sequence['digest'])
        
        fragmentTags = sequenceTags[0].getElementsByTagName('fragment')
        if fragmentTags:
            _getParams(fragmentTags[0], sequence['fragment'])
        
        searchTags = sequenceTags[0].getElementsByTagName('search')
        if searchTags:
            _getParams(searchTags[0], sequence['search'])
        
        if type(sequence['fragment']['fragments']) != list:
            sequence['fragment']['fragments'] = sequence['fragment']['fragments'].split(';')
    
    # masscalc
    masscalcTags = document.getElementsByTagName('masscalc')
    if masscalcTags:
        _getParams(masscalcTags[0], masscalc)
    
    # compounds
    compoundsTags = document.getElementsByTagName('compounds')
    if compoundsTags:
        _getParams(compoundsTags[0], compounds)
        
        if type(compounds['adducts']) != list:
            compounds['adducts'] = compounds['adducts'].split(';')
    
    # differences
    differencesTags = document.getElementsByTagName('differences')
    if differencesTags:
        _getParams(differencesTags[0], differences)
    
    # compare
    compareTags = document.getElementsByTagName('compare')
    if compareTags:
        _getParams(compareTags[0], compare)
    
    # mascot
    mascotTags = document.getElementsByTagName('mascot')
    if mascotTags:
        
        commonTags = mascotTags[0].getElementsByTagName('common')
        if commonTags:
            _getParams(commonTags[0], mascot['common'])
        
        pmfTags = mascotTags[0].getElementsByTagName('pmf')
        if pmfTags:
            _getParams(pmfTags[0], mascot['pmf'])
        
        sqTags = mascotTags[0].getElementsByTagName('sq')
        if sqTags:
            _getParams(sqTags[0], mascot['sq'])
        
        misTags = mascotTags[0].getElementsByTagName('mis')
        if misTags:
            _getParams(misTags[0], mascot['mis'])
        
        for key in ('pmf', 'sq', 'mis'):
            if type(mascot[key]['fixedMods']) != list:
                mascot[key]['fixedMods'] = mascot[key]['fixedMods'].split(';')
            if type(mascot[key]['variableMods']) != list:
                mascot[key]['variableMods'] = mascot[key]['variableMods'].split(';')
    
    # profound
    profoundTags = document.getElementsByTagName('profound')
    if profoundTags:
        _getParams(profoundTags[0], profound)
        
        if type(profound['fixedMods']) != list:
            profound['fixedMods'] = profound['fixedMods'].split(';')
        if type(profound['variableMods']) != list:
            profound['variableMods'] = profound['variableMods'].split(';')
    
    # prospector
    prospectorTags = document.getElementsByTagName('prospector')
    if prospectorTags:
        
        commonTags = prospectorTags[0].getElementsByTagName('common')
        if commonTags:
            _getParams(commonTags[0], prospector['common'])
        
        msfitTags = prospectorTags[0].getElementsByTagName('msfit')
        if msfitTags:
            _getParams(msfitTags[0], prospector['msfit'])
        
        mstagTags = prospectorTags[0].getElementsByTagName('mstag')
        if mstagTags:
            _getParams(mstagTags[0], prospector['mstag'])
        
        for key in ('msfit', 'mstag'):
            if type(prospector[key]['fixedMods']) != list:
                prospector[key]['fixedMods'] = prospector[key]['fixedMods'].split(';')
            if type(prospector[key]['variableMods']) != list:
                prospector[key]['variableMods'] = prospector[key]['variableMods'].split(';')
    
    # links
    linksTags = document.getElementsByTagName('links')
    if linksTags:
        linkTags = linksTags[0].getElementsByTagName('link')
        for linkTag in linkTags:
            name = linkTag.getAttribute('name')
            value = linkTag.getAttribute('value')
            if name not in ('mMassHomepage', 'mMassCite', 'mMassDonate', 'mMassDownload'):
                links[name] = value
# ----


def loadPresets(path=os.path.join(confdir,'presets.xml')):
    """Parse processing presets XML and get data."""
    
    # parse XML
    document = xml.dom.minidom.parse(path)
    
    # get operator presets
    operatorTags = document.getElementsByTagName('operator')
    if operatorTags:
        presetsTags = operatorTags[0].getElementsByTagName('presets')
        if presetsTags:
            libs.presets['operator'].clear()
            
            for presetsTag in presetsTags:
                name = presetsTag.getAttribute('name')
                libs.presets['operator'][name] = {'operator':'', 'contact':'', 'institution':'', 'instrument':''}
                _getParams(presetsTag, libs.presets['operator'][name])
    
    # get processing presets
    processingTags = document.getElementsByTagName('processing')
    if processingTags:
        presetsTags = processingTags[0].getElementsByTagName('presets')
        if presetsTags:
            libs.presets['processing'].clear()
            
            for presetsTag in presetsTags:
                name = presetsTag.getAttribute('name')
                libs.presets['processing'][name] = deepcopy(processing)
                
                cropTags = presetsTag.getElementsByTagName('crop')
                if cropTags:
                    _getParams(cropTags[0], libs.presets['processing'][name]['crop'])
                
                baselineTags = presetsTag.getElementsByTagName('baseline')
                if baselineTags:
                    _getParams(baselineTags[0], libs.presets['processing'][name]['baseline'])
                
                smoothingTags = presetsTag.getElementsByTagName('smoothing')
                if smoothingTags:
                    _getParams(smoothingTags[0], libs.presets['processing'][name]['smoothing'])
                
                peakpickingTags = presetsTag.getElementsByTagName('peakpicking')
                if peakpickingTags:
                    _getParams(peakpickingTags[0], libs.presets['processing'][name]['peakpicking'])
                
                deisotopingTags = presetsTag.getElementsByTagName('deisotoping')
                if deisotopingTags:
                    _getParams(deisotopingTags[0], libs.presets['processing'][name]['deisotoping'])
    
    # get modifications presets
    modificationsTags = document.getElementsByTagName('modifications')
    if modificationsTags:
        presetsTags = modificationsTags[0].getElementsByTagName('presets')
        if presetsTags:
            libs.presets['modifications'].clear()
            
            for presetsTag in presetsTags:
                name = presetsTag.getAttribute('name')
                
                modifications = []
                modificationTags = presetsTag.getElementsByTagName('modification')
                for modificationTag in modificationTags:
                    modName = modificationTag.getAttribute('name')
                    modPosition = modificationTag.getAttribute('position')
                    modType = modificationTag.getAttribute('type')
                    modifications.append([modName, modPosition, modType])
                
                libs.presets['modifications'][name] = modifications
# ----


def loadReferences(path=os.path.join(confdir,'references.xml')):
    """Parse calibration references XML and get data."""
    
    # parse XML
    document = xml.dom.minidom.parse(path)
    
    # get references
    groupTags = document.getElementsByTagName('group')
    if groupTags:
        libs.references.clear()
        for groupTag in groupTags:
            groupName = groupTag.getAttribute('name')
            libs.references[groupName] = []
            
            referenceTags = groupTag.getElementsByTagName('reference')
            if referenceTags:
                for referenceTag in referenceTags:
                    name = referenceTag.getAttribute('name')
                    mass = referenceTag.getAttribute('mass')
                    libs.references[groupName].append((name, float(mass)))
# ----


def loadCompounds(path=os.path.join(confdir,'compounds.xml')):
    """Parse compounds XML and get data."""
    
    # parse XML
    document = xml.dom.minidom.parse(path)
    
    # get references
    groupTags = document.getElementsByTagName('group')
    if groupTags:
        libs.compounds.clear()
        for groupTag in groupTags:
            groupName = groupTag.getAttribute('name')
            libs.compounds[groupName] = {}
            
            compoundTags = groupTag.getElementsByTagName('compound')
            if compoundTags:
                for compoundTag in compoundTags:
                    try:
                        name = compoundTag.getAttribute('name')
                        compound = mspy.compound(compoundTag.getAttribute('formula'))
                        compound.description = _getNodeText(compoundTag)
                        libs.compounds[groupName][name] = compound
                    except:
                        pass
# ----


def loadMascot(path=os.path.join(confdir,'mascot.xml')):
    """Parse mascot servers XML and get data."""
    
    # parse XML
    document = xml.dom.minidom.parse(path)
    
    # get references
    serverTags = document.getElementsByTagName('server')
    if serverTags:
        libs.mascot.clear()
        for serverTag in serverTags:
            name = serverTag.getAttribute('name')
            libs.mascot[name] = {}
            
            libs.mascot[name] = {
                'host': '',
                'path': '/',
                'search': 'cgi/nph-mascot.exe',
                'results': 'cgi/master_results.pl',
                'export': 'cgi/export_dat_2.pl',
                'params': 'cgi/get_params.pl',
            }
            _getParams(serverTag, libs.mascot[name])
# ----


def _getParams(sectionTag, section):
    """Get params from nodes."""
    
    if sectionTag:
        paramTags = sectionTag.getElementsByTagName('param')
        if paramTags:
            if paramTags:
                for paramTag in paramTags:
                    name = paramTag.getAttribute('name')
                    value = paramTag.getAttribute('value')
                    valueType = paramTag.getAttribute('type')
                    if name in section:
                        if valueType in ('unicode', 'str', 'float', 'int'):
                            try:
                                section[name] = eval(valueType+'(value)')
                            except:
                                pass
# ----


def _getNodeText(node):
    """Get text from node list."""
    
    buff = ''
    for node in node.childNodes:
        if node.nodeType == node.TEXT_NODE:
            buff += node.data
    
    return buff
# ----



# SAVE FUNCTIONS
# --------------

def saveConfig(path=os.path.join(confdir,'config.xml')):
    """Make and save config XML."""
    
    data = makeConfigXML()
    try:
        save = file(path, 'w')
        save.write(data.encode("utf-8"))
        save.close()
        return True
    except:
        return False
# ----


def savePresets(path=os.path.join(confdir,'presets.xml')):
    """Make and save presets XML."""
    
    data = makePresetsXML()
    try:
        save = file(path, 'w')
        save.write(data.encode("utf-8"))
        save.close()
        return True
    except:
        return False
# ----


def saveReferences(path=os.path.join(confdir,'references.xml')):
    """Make and save calibration references XML."""
    
    data = makeReferencesXML()
    try:
        save = file(path, 'w')
        save.write(data.encode("utf-8"))
        save.close()
        return True
    except:
        return False
# ----


def saveCompounds(path=os.path.join(confdir,'compounds.xml')):
    """Make and save compounds XML."""
    
    data = makeCompoundsXML()
    try:
        save = file(path, 'w')
        save.write(data.encode("utf-8"))
        save.close()
        return True
    except:
        return False
# ----


def saveMascot(path=os.path.join(confdir,'mascot.xml')):
    """Make and save mascot servers XML."""
    
    data = makeMascotXML()
    try:
        save = file(path, 'w')
        save.write(data.encode("utf-8"))
        save.close()
        return True
    except:
        return False
# ----



# XML FORMATING FUNCTIONS
# -----------------------

def makeConfigXML():
    """Format config XML."""
    
    buff = '<?xml version="1.0" encoding="utf-8" ?>\n'
    buff += '<mMassConfig version="1.0">\n\n'
    
    # main
    buff += '  <main>\n'
    buff += '    <param name="appWidth" value="%d" type="int" />\n' % (main['appWidth'])
    buff += '    <param name="appHeight" value="%d" type="int" />\n' % (main['appHeight'])
    buff += '    <param name="appMaximized" value="%d" type="int" />\n' % (bool(main['appMaximized']))
    buff += '    <param name="unlockGUI" value="%d" type="int" />\n' % (bool(main['unlockGUI']))
    buff += '    <param name="macListCtrlGeneric" value="%d" type="int" />\n' % (bool(main['macListCtrlGeneric']))
    buff += '    <param name="mzDigits" value="%d" type="int" />\n' % (main['mzDigits'])
    buff += '    <param name="intDigits" value="%d" type="int" />\n' % (main['intDigits'])
    buff += '    <param name="ppmDigits" value="%d" type="int" />\n' % (main['ppmDigits'])
    buff += '    <param name="lastDir" value="%s" type="unicode" />\n' % (_escape(main['lastDir']))
    buff += '    <param name="errorUnits" value="%s" type="str" />\n' % (main['errorUnits'])
    buff += '    <param name="printQuality" value="%d" type="int" />\n' % (main['printQuality'])
    buff += '    <param name="donate" value="%s" type="str" />\n' % (main['donate'])
    buff += '    <param name="updatesEnabled" value="%d" type="int" />\n' % (bool(main['updatesEnabled']))
    buff += '    <param name="updatesChecked" value="%s" type="str" />\n' % (main['updatesChecked'])
    buff += '    <param name="updatesCurrent" value="%s" type="str" />\n' % (main['updatesCurrent'])
    buff += '    <param name="updatesAvailable" value="%s" type="str" />\n' % (main['updatesAvailable'])
    buff += '    <param name="compassMode" value="%s" type="str" />\n' % (main['compassMode'])
    buff += '    <param name="compassFormat" value="%s" type="str" />\n' % (main['compassFormat'])
    buff += '    <param name="compassDeleteFile" value="%d" type="int" />\n' % (bool(main['compassDeleteFile']))
    buff += '  </main>\n\n'
    
    # recent files
    buff += '  <recent>\n'
    for item in recent:
        buff += '    <path value="%s" />\n' % (_escape(item))
    buff += '  </recent>\n\n'
    
    # colours
    buff += '  <colours>\n'
    for item in colours:
        buff += '    <colour value="%02x%02x%02x" />\n' % tuple(item)
    buff += '  </colours>\n\n'
    
    # export
    buff += '  <export>\n'
    buff += '    <param name="imageWidth" value="%.1f" type="float" />\n' % (export['imageWidth'])
    buff += '    <param name="imageHeight" value="%.1f" type="float" />\n' % (export['imageHeight'])
    buff += '    <param name="imageUnits" value="%s" type="str" />\n' % (export['imageUnits'])
    buff += '    <param name="imageResolution" value="%d" type="int" />\n' % (export['imageResolution'])
    buff += '    <param name="imageFormat" value="%s" type="str" />\n' % (export['imageFormat'])
    buff += '    <param name="peaklistColumns" value="%s" type="str" />\n' % (';'.join(export['peaklistColumns']))
    buff += '    <param name="peaklistFormat" value="%s" type="str" />\n' % (export['peaklistFormat'])
    buff += '    <param name="peaklistSeparator" value="%s" type="str" />\n' % (export['peaklistSeparator'])
    buff += '    <param name="spectrumSeparator" value="%s" type="str" />\n' % (export['spectrumSeparator'])
    buff += '  </export>\n\n'
    
    # spectrum
    buff += '  <spectrum>\n'
    buff += '    <param name="xLabel" value="%s" type="unicode" />\n' % (_escape(spectrum['xLabel']))
    buff += '    <param name="yLabel" value="%s" type="unicode" />\n' % (_escape(spectrum['yLabel']))
    buff += '    <param name="showGrid" value="%d" type="int" />\n' % (bool(spectrum['showGrid']))
    buff += '    <param name="showLegend" value="%d" type="int" />\n' % (bool(spectrum['showLegend']))
    buff += '    <param name="showPosBar" value="%d" type="int" />\n' % (bool(spectrum['showPosBar']))
    buff += '    <param name="showIntBar" value="%d" type="int" />\n' % (bool(spectrum['showIntBar']))
    buff += '    <param name="showGel" value="%d" type="int" />\n' % (bool(spectrum['showGel']))
    buff += '    <param name="showGelLegend" value="%d" type="int" />\n' % (bool(spectrum['showGelLegend']))
    buff += '    <param name="showTracker" value="%d" type="int" />\n' % (bool(spectrum['showTracker']))
    buff += '    <param name="showNotationMarks" value="%d" type="int" />\n' % (bool(spectrum['showNotationMarks']))
    buff += '    <param name="showDataPoints" value="%d" type="int" />\n' % (bool(spectrum['showDataPoints']))
    buff += '    <param name="showLabels" value="%d" type="int" />\n' % (bool(spectrum['showLabels']))
    buff += '    <param name="showAllLabels" value="%d" type="int" />\n' % (bool(spectrum['showAllLabels']))
    buff += '    <param name="showTicks" value="%d" type="int" />\n' % (bool(spectrum['showTicks']))
    buff += '    <param name="showCursorImage" value="%d" type="int" />\n' % (bool(spectrum['showCursorImage']))
    buff += '    <param name="barHeight" value="%d" type="int" />\n' % (spectrum['barHeight'])
    buff += '    <param name="gelHeight" value="%d" type="int" />\n' % (spectrum['gelHeight'])
    buff += '    <param name="autoscale" value="%d" type="int" />\n' % (bool(spectrum['autoscale']))
    buff += '    <param name="overlapLabels" value="%d" type="int" />\n' % (bool(spectrum['overlapLabels']))
    buff += '    <param name="checkLimits" value="%d" type="int" />\n' % (bool(spectrum['checkLimits']))
    buff += '    <param name="labelAngle" value="%d" type="int" />\n' % (spectrum['labelAngle'])
    buff += '    <param name="labelCharge" value="%d" type="int" />\n' % (bool(spectrum['labelCharge']))
    buff += '    <param name="labelBgr" value="%d" type="int" />\n' % (bool(spectrum['labelBgr']))
    buff += '    <param name="labelFontSize" value="%d" type="int" />\n' % (spectrum['labelFontSize'])
    buff += '    <param name="axisFontSize" value="%d" type="int" />\n' % (spectrum['axisFontSize'])
    buff += '    <param name="tickColour" value="%02x%02x%02x" type="str" />\n' % tuple(spectrum['tickColour'])
    buff += '    <param name="tmpSpectrumColour" value="%02x%02x%02x" type="str" />\n' % tuple(spectrum['tmpSpectrumColour'])
    buff += '    <param name="notationMarksColour" value="%02x%02x%02x" type="str" />\n' % tuple(spectrum['notationMarksColour'])
    buff += '  </spectrum>\n\n'
    
    # match
    buff += '  <match>\n'
    buff += '    <param name="tolerance" value="%f" type="float" />\n' % (match['tolerance'])
    buff += '    <param name="units" value="%s" type="str" />\n' % (match['units'])
    buff += '    <param name="ignoreCharge" value="%d" type="int" />\n' % (bool(match['ignoreCharge']))
    buff += '  </match>\n\n'
    
    # processing
    buff += '  <processing>\n'
    buff += '    <crop>\n'
    buff += '      <param name="lowMass" value="%d" type="int" />\n' % (processing['crop']['lowMass'])
    buff += '      <param name="highMass" value="%d" type="int" />\n' % (processing['crop']['highMass'])
    buff += '    </crop>\n'
    buff += '    <baseline>\n'
    buff += '      <param name="segments" value="%d" type="int" />\n' % (processing['baseline']['segments'])
    buff += '      <param name="offset" value="%f" type="float" />\n' % (processing['baseline']['offset'])
    buff += '      <param name="smooth" value="%d" type="int" />\n' % (bool(processing['baseline']['smooth']))
    buff += '    </baseline>\n'
    buff += '    <smoothing>\n'
    buff += '      <param name="method" value="%s" type="str" />\n' % (processing['smoothing']['method'])
    buff += '      <param name="windowSize" value="%f" type="float" />\n' % (processing['smoothing']['windowSize'])
    buff += '      <param name="cycles" value="%d" type="int" />\n' % (processing['smoothing']['cycles'])
    buff += '    </smoothing>\n'
    buff += '    <peakpicking>\n'
    buff += '      <param name="snThreshold" value="%f" type="float" />\n' % (processing['peakpicking']['snThreshold'])
    buff += '      <param name="absIntThreshold" value="%f" type="float" />\n' % (processing['peakpicking']['absIntThreshold'])
    buff += '      <param name="relIntThreshold" value="%f" type="float" />\n' % (processing['peakpicking']['relIntThreshold'])
    buff += '      <param name="adaptiveNoise" value="%d" type="int" />\n' % (bool(processing['peakpicking']['adaptiveNoise']))
    buff += '      <param name="pickingHeight" value="%f" type="float" />\n' % (processing['peakpicking']['pickingHeight'])
    buff += '      <param name="smoothing" value="%d" type="int" />\n' % (bool(processing['peakpicking']['smoothing']))
    buff += '      <param name="deisotoping" value="%d" type="int" />\n' % (bool(processing['peakpicking']['deisotoping']))
    buff += '      <param name="removeShoulders" value="%d" type="int" />\n' % (bool(processing['peakpicking']['removeShoulders']))
    buff += '      <param name="monoisotopic" value="%d" type="int" />\n' % (bool(processing['peakpicking']['monoisotopic']))
    buff += '    </peakpicking>\n'
    buff += '    <deisotoping>\n'
    buff += '      <param name="maxCharge" value="%d" type="int" />\n' % (processing['deisotoping']['maxCharge'])
    buff += '      <param name="massTolerance" value="%f" type="float" />\n' % (processing['deisotoping']['massTolerance'])
    buff += '      <param name="intTolerance" value="%f" type="float" />\n' % (processing['deisotoping']['intTolerance'])
    buff += '      <param name="removeIsotopes" value="%d" type="int" />\n' % (bool(processing['deisotoping']['removeIsotopes']))
    buff += '      <param name="removeUnknown" value="%d" type="int" />\n' % (bool(processing['deisotoping']['removeUnknown']))
    buff += '    </deisotoping>\n'
    buff += '  </processing>\n\n'
    
    # calibration
    buff += '  <calibration>\n'
    buff += '    <param name="fitting" value="%s" type="str" />\n' % (calibration['fitting'])
    buff += '    <param name="tolerance" value="%f" type="float" />\n' % (calibration['tolerance'])
    buff += '    <param name="units" value="%s" type="str" />\n' % (calibration['units'])
    buff += '    <param name="statCutOff" value="%d" type="int" />\n' % (calibration['statCutOff'])
    buff += '  </calibration>\n\n'
    
    # sequence
    buff += '  <sequence>\n'
    buff += '    <digest>\n'
    buff += '      <param name="maxMods" value="%d" type="int" />\n' % (sequence['digest']['maxMods'])
    buff += '      <param name="maxCharge" value="%d" type="int" />\n' % (sequence['digest']['maxCharge'])
    buff += '      <param name="massType" value="%d" type="int" />\n' % (sequence['digest']['massType'])
    buff += '      <param name="enzyme" value="%s" type="unicode" />\n' % (_escape(sequence['digest']['enzyme']))
    buff += '      <param name="miscl" value="%d" type="int" />\n' % (sequence['digest']['miscl'])
    buff += '      <param name="lowMass" value="%d" type="int" />\n' % (sequence['digest']['lowMass'])
    buff += '      <param name="highMass" value="%d" type="int" />\n' % (sequence['digest']['highMass'])
    buff += '      <param name="retainPos" value="%d" type="int" />\n' % (bool(sequence['digest']['retainPos']))
    buff += '      <param name="allowMods" value="%d" type="int" />\n' % (bool(sequence['digest']['allowMods']))
    buff += '      <param name="listFormat" value="%s" type="str" />\n' % (sequence['digest']['listFormat'])
    buff += '      <param name="matchFormat" value="%s" type="str" />\n' % (sequence['digest']['matchFormat'])
    buff += '    </digest>\n'
    buff += '    <fragment>\n'
    buff += '      <param name="maxMods" value="%d" type="int" />\n' % (sequence['fragment']['maxMods'])
    buff += '      <param name="maxCharge" value="%d" type="int" />\n' % (sequence['fragment']['maxCharge'])
    buff += '      <param name="massType" value="%d" type="int" />\n' % (sequence['fragment']['massType'])
    buff += '      <param name="fragments" value="%s" type="str" />\n' % (';'.join(sequence['fragment']['fragments']))
    buff += '      <param name="filterFragments" value="%d" type="int" />\n' % (bool(sequence['fragment']['filterFragments']))
    buff += '      <param name="listFormat" value="%s" type="str" />\n' % (sequence['fragment']['listFormat'])
    buff += '      <param name="matchFormat" value="%s" type="str" />\n' % (sequence['fragment']['matchFormat'])
    buff += '    </fragment>\n'
    buff += '    <search>\n'
    buff += '      <param name="maxMods" value="%d" type="int" />\n' % (sequence['search']['maxMods'])
    buff += '      <param name="charge" value="%d" type="int" />\n' % (sequence['search']['charge'])
    buff += '      <param name="massType" value="%d" type="int" />\n' % (sequence['search']['massType'])
    buff += '      <param name="enzyme" value="%s" type="unicode" />\n' % (_escape(sequence['search']['enzyme']))
    buff += '      <param name="tolerance" value="%f" type="float" />\n' % (sequence['search']['tolerance'])
    buff += '      <param name="units" value="%s" type="str" />\n' % (sequence['search']['units'])
    buff += '      <param name="retainPos" value="%d" type="int" />\n' % (bool(sequence['search']['retainPos']))
    buff += '      <param name="listFormat" value="%s" type="str" />\n' % (sequence['search']['listFormat'])
    buff += '    </search>\n'
    buff += '  </sequence>\n\n'
    
    # masscalc
    buff += '  <masscalc>\n'
    buff += '    <param name="ionseriesAgent" value="%s" type="str" />\n' % (masscalc['ionseriesAgent'])
    buff += '    <param name="ionseriesAgentCharge" value="%d" type="int" />\n' % (masscalc['ionseriesAgentCharge'])
    buff += '    <param name="ionseriesPolarity" value="%d" type="int" />\n' % (masscalc['ionseriesPolarity'])
    buff += '    <param name="patternFwhm" value="%f" type="float" />\n' % (masscalc['patternFwhm'])
    buff += '    <param name="patternThreshold" value="%f" type="float" />\n' % (masscalc['patternThreshold'])
    buff += '  </masscalc>\n\n'
    
    # compounds
    buff += '  <compounds>\n'
    buff += '    <param name="massType" value="%d" type="int" />\n' % (compounds['massType'])
    buff += '    <param name="maxCharge" value="%d" type="int" />\n' % (compounds['maxCharge'])
    buff += '    <param name="radicals" value="%d" type="int" />\n' % (bool(compounds['radicals']))
    buff += '    <param name="adducts" value="%s" type="str" />\n' % (';'.join(compounds['adducts']))
    buff += '  </compounds>\n\n'
    
    # differences
    buff += '  <differences>\n'
    buff += '    <param name="aminoacids" value="%d" type="int" />\n' % (bool(differences['aminoacids']))
    buff += '    <param name="dipeptides" value="%d" type="int" />\n' % (bool(differences['dipeptides']))
    buff += '    <param name="tolerance" value="%f" type="float" />\n' % (differences['tolerance'])
    buff += '    <param name="massType" value="%d" type="int" />\n' % (differences['massType'])
    buff += '  </differences>\n\n'
    
    # compare
    buff += '  <compare>\n'
    buff += '    <param name="tolerance" value="%f" type="float" />\n' % (compare['tolerance'])
    buff += '    <param name="units" value="%s" type="str" />\n' % (compare['units'])
    buff += '    <param name="ignoreCharge" value="%d" type="int" />\n' % (bool(compare['ignoreCharge']))
    buff += '  </compare>\n\n'
    
    # mascot
    buff += '  <mascot>\n'
    buff += '    <common>\n'
    buff += '      <param name="server" value="%s" type="unicode" />\n' % (_escape(mascot['common']['server']))
    buff += '      <param name="searchType" value="%s" type="str" />\n' % (mascot['common']['searchType'])
    buff += '      <param name="userName" value="%s" type="unicode" />\n' % (_escape(mascot['common']['userName']))
    buff += '      <param name="userEmail" value="%s" type="unicode" />\n' % (_escape(mascot['common']['userEmail']))
    buff += '    </common>\n'
    buff += '    <pmf>\n'
    buff += '      <param name="database" value="%s" type="unicode" />\n' % (mascot['pmf']['database'])
    buff += '      <param name="taxonomy" value="%s" type="unicode" />\n' % (mascot['pmf']['taxonomy'])
    buff += '      <param name="enzyme" value="%s" type="unicode" />\n' % (mascot['pmf']['enzyme'])
    buff += '      <param name="miscleavages" value="%s" type="unicode" />\n' % (mascot['pmf']['miscleavages'])
    buff += '      <param name="fixedMods" value="%s" type="unicode" />\n' % (';'.join(mascot['pmf']['fixedMods']))
    buff += '      <param name="variableMods" value="%s" type="unicode" />\n' % (';'.join(mascot['pmf']['variableMods']))
    buff += '      <param name="hiddenMods" value="%d" type="int" />\n' % (bool(mascot['pmf']['hiddenMods']))
    buff += '      <param name="proteinMass" value="%s" type="unicode" />\n' % (mascot['pmf']['proteinMass'])
    buff += '      <param name="peptideTol" value="%s" type="unicode" />\n' % (mascot['pmf']['peptideTol'])
    buff += '      <param name="peptideTolUnits" value="%s" type="unicode" />\n' % (mascot['pmf']['peptideTolUnits'])
    buff += '      <param name="massType" value="%s" type="unicode" />\n' % (mascot['pmf']['massType'])
    buff += '      <param name="charge" value="%s" type="unicode" />\n' % (mascot['pmf']['charge'])
    buff += '      <param name="decoy" value="%d" type="int" />\n' % (bool(mascot['pmf']['decoy']))
    buff += '      <param name="report" value="%s" type="unicode" />\n' % (mascot['pmf']['report'])
    buff += '    </pmf>\n'
    buff += '    <sq>\n'
    buff += '      <param name="database" value="%s" type="unicode" />\n' % (mascot['sq']['database'])
    buff += '      <param name="taxonomy" value="%s" type="unicode" />\n' % (mascot['sq']['taxonomy'])
    buff += '      <param name="enzyme" value="%s" type="unicode" />\n' % (mascot['sq']['enzyme'])
    buff += '      <param name="miscleavages" value="%s" type="unicode" />\n' % (mascot['sq']['miscleavages'])
    buff += '      <param name="fixedMods" value="%s" type="unicode" />\n' % (';'.join(mascot['sq']['fixedMods']))
    buff += '      <param name="variableMods" value="%s" type="unicode" />\n' % (';'.join(mascot['sq']['variableMods']))
    buff += '      <param name="hiddenMods" value="%d" type="int" />\n' % (bool(mascot['sq']['hiddenMods']))
    buff += '      <param name="peptideTol" value="%s" type="unicode" />\n' % (mascot['sq']['peptideTol'])
    buff += '      <param name="peptideTolUnits" value="%s" type="unicode" />\n' % (mascot['sq']['peptideTolUnits'])
    buff += '      <param name="msmsTol" value="%s" type="unicode" />\n' % (mascot['sq']['msmsTol'])
    buff += '      <param name="msmsTolUnits" value="%s" type="unicode" />\n' % (mascot['sq']['msmsTolUnits'])
    buff += '      <param name="massType" value="%s" type="unicode" />\n' % (mascot['sq']['massType'])
    buff += '      <param name="charge" value="%s" type="unicode" />\n' % (mascot['sq']['charge'])
    buff += '      <param name="instrument" value="%s" type="unicode" />\n' % (mascot['sq']['instrument'])
    buff += '      <param name="quantitation" value="%s" type="unicode" />\n' % (mascot['sq']['quantitation'])
    buff += '      <param name="decoy" value="%d" type="int" />\n' % (bool(mascot['sq']['decoy']))
    buff += '      <param name="report" value="%s" type="unicode" />\n' % (mascot['sq']['report'])
    buff += '    </sq>\n'
    buff += '    <mis>\n'
    buff += '      <param name="database" value="%s" type="unicode" />\n' % (mascot['mis']['database'])
    buff += '      <param name="taxonomy" value="%s" type="unicode" />\n' % (mascot['mis']['taxonomy'])
    buff += '      <param name="enzyme" value="%s" type="unicode" />\n' % (mascot['mis']['enzyme'])
    buff += '      <param name="miscleavages" value="%s" type="unicode" />\n' % (mascot['mis']['miscleavages'])
    buff += '      <param name="fixedMods" value="%s" type="unicode" />\n' % (';'.join(mascot['mis']['fixedMods']))
    buff += '      <param name="variableMods" value="%s" type="unicode" />\n' % (';'.join(mascot['mis']['variableMods']))
    buff += '      <param name="hiddenMods" value="%d" type="int" />\n' % (bool(mascot['mis']['hiddenMods']))
    buff += '      <param name="peptideTol" value="%s" type="unicode" />\n' % (mascot['mis']['peptideTol'])
    buff += '      <param name="peptideTolUnits" value="%s" type="unicode" />\n' % (mascot['mis']['peptideTolUnits'])
    buff += '      <param name="msmsTol" value="%s" type="unicode" />\n' % (mascot['mis']['msmsTol'])
    buff += '      <param name="msmsTolUnits" value="%s" type="unicode" />\n' % (mascot['mis']['msmsTolUnits'])
    buff += '      <param name="massType" value="%s" type="unicode" />\n' % (mascot['mis']['massType'])
    buff += '      <param name="charge" value="%s" type="unicode" />\n' % (mascot['mis']['charge'])
    buff += '      <param name="instrument" value="%s" type="unicode" />\n' % (mascot['mis']['instrument'])
    buff += '      <param name="quantitation" value="%s" type="unicode" />\n' % (mascot['mis']['quantitation'])
    buff += '      <param name="errorTolerant" value="%d" type="int" />\n' % (bool(mascot['mis']['errorTolerant']))
    buff += '      <param name="decoy" value="%d" type="int" />\n' % (bool(mascot['mis']['decoy']))
    buff += '      <param name="report" value="%s" type="unicode" />\n' % (mascot['mis']['report'])
    buff += '    </mis>\n'
    buff += '  </mascot>\n\n'
    
    # profound
    buff += '  <profound>\n'
    buff += '    <param name="script" value="%s" type="unicode" />\n' % (_escape(profound['script']))
    buff += '    <param name="database" value="%s" type="unicode" />\n' % (profound['database'])
    buff += '    <param name="taxonomy" value="%s" type="unicode" />\n' % (profound['taxonomy'])
    buff += '    <param name="enzyme" value="%s" type="unicode" />\n' % (profound['enzyme'])
    buff += '    <param name="miscleavages" value="%s" type="unicode" />\n' % (profound['miscleavages'])
    buff += '    <param name="fixedMods" value="%s" type="unicode" />\n' % (';'.join(profound['fixedMods']))
    buff += '    <param name="variableMods" value="%s" type="unicode" />\n' % (';'.join(profound['variableMods']))
    buff += '    <param name="proteinMassLow" value="%f" type="float" />\n' % (profound['proteinMassLow'])
    buff += '    <param name="proteinMassHigh" value="%f" type="float" />\n' % (profound['proteinMassHigh'])
    buff += '    <param name="proteinPILow" value="%d" type="int" />\n' % (profound['proteinPILow'])
    buff += '    <param name="proteinPIHigh" value="%d" type="int" />\n' % (profound['proteinPIHigh'])
    buff += '    <param name="peptideTol" value="%f" type="float" />\n' % (profound['peptideTol'])
    buff += '    <param name="peptideTolUnits" value="%s" type="unicode" />\n' % (profound['peptideTolUnits'])
    buff += '    <param name="massType" value="%s" type="unicode" />\n' % (profound['massType'])
    buff += '    <param name="charge" value="%s" type="unicode" />\n' % (profound['charge'])
    buff += '    <param name="ranking" value="%s" type="unicode" />\n' % (profound['ranking'])
    buff += '    <param name="expectation" value="%f" type="float" />\n' % (profound['expectation'])
    buff += '    <param name="candidates" value="%d" type="int" />\n' % (profound['candidates'])
    buff += '  </profound>\n\n'
    
    # protein prospector
    buff += '  <prospector>\n'
    buff += '    <common>\n'
    buff += '      <param name="script" value="%s" type="unicode" />\n' % (_escape(prospector['common']['script']))
    buff += '      <param name="searchType" value="%s" type="str" />\n' % (prospector['common']['searchType'])
    buff += '    </common>\n'
    buff += '    <msfit>\n'
    buff += '      <param name="database" value="%s" type="unicode" />\n' % (prospector['msfit']['database'])
    buff += '      <param name="taxonomy" value="%s" type="unicode" />\n' % (prospector['msfit']['taxonomy'])
    buff += '      <param name="enzyme" value="%s" type="unicode" />\n' % (prospector['msfit']['enzyme'])
    buff += '      <param name="miscleavages" value="%s" type="unicode" />\n' % (prospector['msfit']['miscleavages'])
    buff += '      <param name="fixedMods" value="%s" type="unicode" />\n' % (';'.join(prospector['msfit']['fixedMods']))
    buff += '      <param name="variableMods" value="%s" type="unicode" />\n' % (';'.join(prospector['msfit']['variableMods']))
    buff += '      <param name="proteinMassLow" value="%s" type="unicode" />\n' % (prospector['msfit']['proteinMassLow'])
    buff += '      <param name="proteinMassHigh" value="%s" type="unicode" />\n' % (prospector['msfit']['proteinMassHigh'])
    buff += '      <param name="proteinPILow" value="%s" type="unicode" />\n' % (prospector['msfit']['proteinPILow'])
    buff += '      <param name="proteinPIHigh" value="%s" type="unicode" />\n' % (prospector['msfit']['proteinPIHigh'])
    buff += '      <param name="peptideTol" value="%s" type="unicode" />\n' % (prospector['msfit']['peptideTol'])
    buff += '      <param name="peptideTolUnits" value="%s" type="unicode" />\n' % (prospector['msfit']['peptideTolUnits'])
    buff += '      <param name="massType" value="%s" type="unicode" />\n' % (prospector['msfit']['massType'])
    buff += '      <param name="instrument" value="%s" type="unicode" />\n' % (prospector['msfit']['instrument'])
    buff += '      <param name="minMatches" value="%s" type="unicode" />\n' % (prospector['msfit']['minMatches'])
    buff += '      <param name="maxMods" value="%s" type="unicode" />\n' % (prospector['msfit']['maxMods'])
    buff += '      <param name="report" value="%s" type="unicode" />\n' % (prospector['msfit']['report'])
    buff += '      <param name="pfactor" value="%s" type="unicode" />\n' % (prospector['msfit']['pfactor'])
    buff += '    </msfit>\n'
    buff += '    <mstag>\n'
    buff += '      <param name="database" value="%s" type="unicode" />\n' % (prospector['mstag']['database'])
    buff += '      <param name="taxonomy" value="%s" type="unicode" />\n' % (prospector['mstag']['taxonomy'])
    buff += '      <param name="enzyme" value="%s" type="unicode" />\n' % (prospector['mstag']['enzyme'])
    buff += '      <param name="miscleavages" value="%s" type="unicode" />\n' % (prospector['mstag']['miscleavages'])
    buff += '      <param name="fixedMods" value="%s" type="unicode" />\n' % (';'.join(prospector['mstag']['fixedMods']))
    buff += '      <param name="variableMods" value="%s" type="unicode" />\n' % (';'.join(prospector['mstag']['variableMods']))
    buff += '      <param name="peptideTol" value="%s" type="unicode" />\n' % (prospector['mstag']['peptideTol'])
    buff += '      <param name="peptideTolUnits" value="%s" type="unicode" />\n' % (prospector['mstag']['peptideTolUnits'])
    buff += '      <param name="peptideCharge" value="%s" type="unicode" />\n' % (prospector['mstag']['peptideCharge'])
    buff += '      <param name="msmsTol" value="%s" type="unicode" />\n' % (prospector['mstag']['msmsTol'])
    buff += '      <param name="msmsTolUnits" value="%s" type="unicode" />\n' % (prospector['mstag']['msmsTolUnits'])
    buff += '      <param name="massType" value="%s" type="unicode" />\n' % (prospector['mstag']['massType'])
    buff += '      <param name="instrument" value="%s" type="unicode" />\n' % (prospector['mstag']['instrument'])
    buff += '      <param name="maxMods" value="%s" type="unicode" />\n' % (prospector['mstag']['maxMods'])
    buff += '      <param name="report" value="%s" type="unicode" />\n' % (prospector['mstag']['report'])
    buff += '    </mstag>\n'
    buff += '  </prospector>\n\n'
    
    # links
    buff += '  <links>\n'
    for name in links:
        if name not in ('mMassHomepage', 'mMassCite', 'mMassDonate', 'mMassDownload'):
            buff += '    <link name="%s" value="%s" />\n' % (_escape(name), _escape(links[name]))
    buff += '  </links>\n\n'
    
    buff += '</mMassConfig>'
    return buff
# ----


def makePresetsXML():
    """Format presets XML."""
    
    buff = '<?xml version="1.0" encoding="utf-8" ?>\n'
    buff += '<mMassPresets version="1.0">\n\n'
    
    # operator presets
    buff += '  <operator>\n\n'
    for name in sorted(libs.presets['operator'].keys()):
        item = libs.presets['operator'][name]
        buff += '    <presets name="%s">\n' % (_escape(name))
        buff += '      <param name="operator" value="%s" type="unicode" />\n' % (_escape(item['operator']))
        buff += '      <param name="contact" value="%s" type="unicode" />\n' % (_escape(item['contact']))
        buff += '      <param name="institution" value="%s" type="unicode" />\n' % (_escape(item['institution']))
        buff += '      <param name="instrument" value="%s" type="unicode" />\n' % (_escape(item['instrument']))
        buff += '    </presets>\n\n'
    buff += '  </operator>\n\n'
    
    # processing presets
    buff += '  <processing>\n\n'
    for name in sorted(libs.presets['processing'].keys()):
        item = libs.presets['processing'][name]
        buff += '    <presets name="%s">\n' % (_escape(name))
        buff += '      <crop>\n'
        buff += '        <param name="lowMass" value="%d" type="int" />\n' % (item['crop']['lowMass'])
        buff += '        <param name="highMass" value="%d" type="int" />\n' % (item['crop']['highMass'])
        buff += '      </crop>\n'
        buff += '      <baseline>\n'
        buff += '        <param name="segments" value="%d" type="int" />\n' % (item['baseline']['segments'])
        buff += '        <param name="offset" value="%f" type="float" />\n' % (item['baseline']['offset'])
        buff += '        <param name="smooth" value="%d" type="int" />\n' % (bool(item['baseline']['smooth']))
        buff += '      </baseline>\n'
        buff += '      <smoothing>\n'
        buff += '        <param name="method" value="%s" type="str" />\n' % (item['smoothing']['method'])
        buff += '        <param name="windowSize" value="%f" type="float" />\n' % (item['smoothing']['windowSize'])
        buff += '        <param name="cycles" value="%d" type="int" />\n' % (item['smoothing']['cycles'])
        buff += '      </smoothing>\n'
        buff += '      <peakpicking>\n'
        buff += '        <param name="snThreshold" value="%f" type="float" />\n' % (item['peakpicking']['snThreshold'])
        buff += '        <param name="absIntThreshold" value="%f" type="float" />\n' % (item['peakpicking']['absIntThreshold'])
        buff += '        <param name="relIntThreshold" value="%f" type="float" />\n' % (item['peakpicking']['relIntThreshold'])
        buff += '        <param name="adaptiveNoise" value="%d" type="int" />\n' % (bool(item['peakpicking']['adaptiveNoise']))
        buff += '        <param name="pickingHeight" value="%f" type="float" />\n' % (item['peakpicking']['pickingHeight'])
        buff += '        <param name="smoothing" value="%d" type="int" />\n' % (bool(item['peakpicking']['smoothing']))
        buff += '        <param name="deisotoping" value="%d" type="int" />\n' % (bool(item['peakpicking']['deisotoping']))
        buff += '        <param name="removeShoulders" value="%d" type="int" />\n' % (bool(item['peakpicking']['removeShoulders']))
        buff += '        <param name="monoisotopic" value="%d" type="int" />\n' % (bool(item['peakpicking']['monoisotopic']))
        buff += '      </peakpicking>\n'
        buff += '      <deisotoping>\n'
        buff += '        <param name="maxCharge" value="%d" type="int" />\n' % (item['deisotoping']['maxCharge'])
        buff += '        <param name="massTolerance" value="%f" type="float" />\n' % (item['deisotoping']['massTolerance'])
        buff += '        <param name="intTolerance" value="%f" type="float" />\n' % (item['deisotoping']['intTolerance'])
        buff += '        <param name="removeIsotopes" value="%d" type="int" />\n' % (bool(item['deisotoping']['removeIsotopes']))
        buff += '        <param name="removeUnknown" value="%d" type="int" />\n' % (bool(item['deisotoping']['removeUnknown']))
        buff += '      </deisotoping>\n'
        buff += '    </presets>\n\n'
    buff += '  </processing>\n\n'
    
    # modifications presets
    buff += '  <modifications>\n\n'
    for name in sorted(libs.presets['modifications'].keys()):
        buff += '    <presets name="%s">\n' % (_escape(name))
        for mod in libs.presets['modifications'][name]:
            buff += '      <modification name="%s" position="%s" type="%s" />\n' % (mod[0], mod[1], mod[2])
        buff += '    </presets>\n\n'
    buff += '  </modifications>\n\n'
    
    buff += '</mMassPresets>'
    return buff
# ----


def makeReferencesXML():
    """Format calibration references XML."""
    
    buff = '<?xml version="1.0" encoding="utf-8" ?>\n'
    buff += '<mMassReferenceMasses version="1.0">\n\n'
    
    for group in sorted(libs.references.keys()):
        buff += '  <group name="%s">\n' % (_escape(group))
        for ref in libs.references[group]:
            buff += '    <reference name="%s" mass="%f" />\n' % (_escape(ref[0]), ref[1])
        buff += '  </group>\n\n'
    
    buff += '</mMassReferenceMasses>'
    return buff
# ----


def makeCompoundsXML():
    """Format compounds XML."""
    
    buff = '<?xml version="1.0" encoding="utf-8" ?>\n'
    buff += '<mMassCompounds version="1.0">\n\n'
    
    for group in sorted(libs.compounds.keys()):
        buff += '  <group name="%s">\n' % (_escape(group))
        for name, compound in sorted(libs.compounds[group].items()):
            buff += '    <compound name="%s" formula="%s">%s</compound>\n' % (_escape(name), compound.rawFormula, _escape(compound.description))
        buff += '  </group>\n\n'
    
    buff += '</mMassCompounds>'
    return buff
# ----


def makeMascotXML():
    """Format mascot servers XML."""
    
    buff = '<?xml version="1.0" encoding="utf-8" ?>\n'
    buff += '<mMassMascot version="1.0">\n\n'
    
    for name in sorted(libs.mascot.keys()):
        buff += '   <server name="%s">\n' % (_escape(name))
        buff += '     <param name="host" value="%s" type="unicode" />\n' % (_escape(libs.mascot[name]['host']))
        buff += '     <param name="path" value="%s" type="unicode" />\n' % (_escape(libs.mascot[name]['path']))
        buff += '     <param name="search" value="%s" type="unicode" />\n' % (_escape(libs.mascot[name]['search']))
        buff += '     <param name="results" value="%s" type="unicode" />\n' % (_escape(libs.mascot[name]['results']))
        buff += '     <param name="export" value="%s" type="unicode" />\n' % (_escape(libs.mascot[name]['export']))
        buff += '     <param name="params" value="%s" type="unicode" />\n' % (_escape(libs.mascot[name]['params']))
        buff += '   </server>\n\n'
    
    buff += '</mMassMascot>'
    return buff
# ----


def _escape(text):
    """Clear special characters such as <> etc."""
    
    text = text.strip()
    search = ('&', '"', "'", '<', '>')
    replace = ('&amp;', '&quot;', '&apos;', '&lt;', '&gt;')
    for x, item in enumerate(search):
        text = text.replace(item, replace[x])
        
    return text
# ----



# LOAD CONFIGS
# ------------

try: loadConfig()
except: saveConfig()
try: loadPresets()
except: savePresets()
try: loadReferences()
except: saveReferences()
try: loadCompounds()
except: saveCompounds()
try: loadMascot()
except: saveMascot()
