/* Copyright (C) 2000-2009 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>
#include <assert.h>

#ifdef WIN32
#include <process.h>
#endif

#ifdef HAVE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef CHASEN
#include <chasen.h>
#endif

#ifdef MECAB
#include <mecab.h>
#endif

#include "udmsearch.h"
#include "udm_sqldbms.h"
#include "udm_uniconv.h"
#include "udm_parsexml.h"
#include "udm_http.h"

#ifdef WIN32
#include "udm_winutils.h"
#endif
/* This should be last include */
#ifdef DMALLOC
#include "dmalloc.h"
#endif

       unsigned int seconds =0; /* To sleep between documents    */
       int flags    =0; /* For indexer            */
       int total_threads=0; /* Total threads number         */
       int sleep_threads=0; /* Number of sleepping threads      */
       int max_index_time=-1;
       int cur_url_number=0;
static int log2stderr=1;
static int have_loglevel= 0;
static int loglevel= UDM_LOG_INFO;
static char cname[1024]="";
static int add_servers=UDM_FLAG_ADD_SERV;
static int add_server_urls = UDM_FLAG_ADD_SERVURL;
static int load_langmaps=UDM_FLAG_LOAD_LANGMAP;
static int load_spells=UDM_FLAG_SPELL;
static int load_for_dump= 0;
static int warnings=1;
       int maxthreads=1;
       int indexing = 0;

static UDM_ENV Conf;

UDM_AGENT *ThreadIndexers;

#ifdef HAVE_PTHREAD
#ifdef WIN32
HANDLE *threads;
typedef HANDLE udm_thread_t;
#ifdef CONSOLE

#include <time.h>
static struct tm*
localtime_r(const time_t *clock, struct tm *result)
{
  *result= *localtime(clock);
  return result;
}

#define strptime(x, y, z) (0)
#endif
#else
#define CONSOLE
pthread_t *threads;
typedef pthread_t udm_thread_t;
#endif
#endif

/* CallBack function for log information */
#ifdef WIN32
#ifdef CONSOLE
#define UDM_DBTYPE "odbc"
void UdmShowInfo(UDM_AGENT* A, const char *state, const char* str)
{
  printf("%d %s %s\n", A ? A->handle : 0,state,str);
}
#else
extern __C_LINK void __UDMCALL UdmShowInfo(UDM_AGENT* A, const char *state, const char* str);
#endif
#endif


static int UdmDisplaySQLQuery(UDM_SQLMON_PARAM *prm, UDM_SQLRES *sqlres)
{
  int       res = UDM_OK;
#ifdef HAVE_SQL     
  size_t         i,j;

  if (prm->flags & UDM_SQLMON_DISPLAY_FIELDS)
  {
    for (i=0;i<sqlres->nCols;i++)
    {
      if (i>0)
        fprintf(prm->outfile,"\t");
      fprintf(prm->outfile,"%s", sqlres->Fields ?
                                 sqlres->Fields[i].sqlname : "<NONAME>");
      if (i+1==sqlres->nCols)
        fprintf(prm->outfile,"\n");
    }
  }
  
  for (i=0;i<sqlres->nRows;i++)
  {
    for (j=0;j<sqlres->nCols;j++)
    {
      const char *v=UdmSQLValue(sqlres,i,j);
      if (j>0)
        fprintf(prm->outfile,"\t");
      if (j < 10 && (prm->colflags[j] & 1))
      {
        const char* s, *e;
        fprintf(prm->outfile,"0x");
                 
        for (s=v, e= s+UdmSQLLen(sqlres,i,j); s < e; s++)
          fprintf(prm->outfile,"%02X",(int)(unsigned char)s[0]);
      }
      else
      {
        fprintf(prm->outfile,"%s",v?v:"NULL");
      }
      if (j+1==sqlres->nCols)fprintf(prm->outfile,"\n");
    }
  }
     
#endif
     return res;
}

static char* sqlmongets(UDM_SQLMON_PARAM *prm, char *str, size_t size)
{
#ifdef HAVE_READLINE
  if ((prm->infile == stdin) && isatty(0))
  {
     char prompt[]="SQL>";
     char *line= readline(prompt);
     if (!line)
       return 0;
     
     if (*line) add_history(line);
     strncpy(str, line, size);
  }
  else
#endif
  {
    prm->prompt(prm, UDM_SQLMON_MSG_PROMPT, "SQL>");
    if (!fgets(str, size, prm->infile))
      return 0;
  }
  return str;
}

static int sqlmonprompt(UDM_SQLMON_PARAM *prm, int msqtype, const char *msg)
{
  fprintf(prm->outfile,"%s",msg);
  return UDM_OK;
}

__C_LINK const char* __UDMCALL UdmIndCmdStr(enum udm_indcmd cmd)
{
  switch(cmd)
  {
    case UDM_IND_CREATE: return "create";
    case UDM_IND_DROP:   return "drop";
    default: return "";
  }
  return "unknown_cmd";
}



static int CreateOrDrop(UDM_AGENT *A, enum udm_indcmd cmd)
{
  size_t i;
  char fname[1024];
  const char *sdir=UdmVarListFindStr(&Conf.Vars,"ShareDir",UDM_SHARE_DIR);
  UDM_DBLIST *L= &A->Conf->dbl;
  UDM_SQLMON_PARAM prm;
  size_t num= UdmVarListFindInt(&A->Conf->Vars, "DBLimit", 0);
  
  for (i=0; i<L->nitems; i++)
  {
    FILE *infile;
    UDM_DB *db= &L->db[i];
    if (num != 0 && num != i + 1)
      continue;
    udm_snprintf(fname,sizeof(fname),"%s%s%s%s%s.%s.sql",
      sdir,UDMSLASHSTR, UdmDBTypeToStr(db->DBType),UDMSLASHSTR,
      UdmIndCmdStr(cmd),UdmDBModeToStr(db->DBMode));
    printf("'%s' dbtype=%d dbmode=%d\n",fname,db->DBType,db->DBMode);
    if (!(infile= fopen(fname,"r")))
    {
      sprintf(A->Conf->errstr,"Can't open file '%s'",fname);
      return UDM_ERROR;
    }
    L->currdbnum= i;
    bzero((void*)&prm,sizeof(prm));
    prm.infile= infile;
    prm.outfile= stdout;
    prm.flags= UDM_SQLMON_DISPLAY_FIELDS;
    prm.gets= sqlmongets;
    prm.display= UdmDisplaySQLQuery;
    prm.prompt= sqlmonprompt;
    UdmSQLMonitor(A, A->Conf,&prm);
    printf("%d queries sent, %d succeeded, %d failed\n",
      prm.nqueries, prm.ngood, prm.nbad);
    fclose(infile);
  }
  return UDM_OK;
}

static int ShowStatistics(UDM_AGENT *Indexer)
{
  int       res;
  struct tm tm;
  const char *stat_time;
  char sbuf[32];
  UDM_STATLIST   Stats;
  size_t         snum;
  UDM_STAT  Total;
     
  bzero((void*)&Total, sizeof(Total));
  Stats.time = time(NULL);
  stat_time = UdmVarListFindStr(&Conf.Vars, "stat_time", "0");
  bzero(&tm, sizeof(tm));

#ifndef WIN32
  if (stat_time &&
      strlen(stat_time) >= 7 &&
      stat_time[4] == '-' &&
      (stat_time[7] == '-' || !stat_time[7]) &&
      (strptime(stat_time, "%Y-%m-%d %H:%M:%S", &tm) ||
       strptime(stat_time, "%Y-%m-%d %H:%M", &tm) ||
       strptime(stat_time, "%Y-%m-%d %H:%M", &tm) ||
       strptime(stat_time, "%Y-%m-%d %H", &tm) ||
       strptime(stat_time, "%Y-%m-%d", &tm) ||
       strptime(stat_time, "%Y-%m", &tm)))
  {
    Stats.time = mktime(&tm);
  }
  else if (stat_time && (Stats.time = Udm_dp2time_t(stat_time)) >= 0)
  {
    Stats.time += time(NULL);
    localtime_r(&Stats.time, &tm);
  }
  else
  {
    Stats.time = time(NULL);
    localtime_r(&Stats.time, &tm);
  }
#else
  {
	struct tm *tm1;
    Stats.time= time(NULL);
    tm1= localtime(&Stats.time);
  }
#endif
  strftime(sbuf, sizeof(sbuf), "%Y-%m-%d %H:%M:%S", &tm);
  res=UdmStatAction(Indexer,&Stats);

  printf("\n          Database statistics [%s]\n\n", sbuf);
  printf("%10s %10s %10s\n","Status","Expired","Total");
  printf("   -----------------------------\n");
  for(snum=0;snum<Stats.nstats;snum++)
  {
    UDM_STAT  *S=&Stats.Stat[snum];
    printf("%10d %10d %10d %s\n",S->status,S->expired,S->total,UdmHTTPErrMsg(S->status));
    Total.expired+=S->expired;
    Total.total+=S->total;
  }
  printf("   -----------------------------\n");
  printf("%10s %10d %10d\n","Total",Total.expired,Total.total);
  printf("\n");
  UDM_FREE(Stats.Stat);
  return(res);
}

/* CallBack Func for Referers*/
static void UdmRefProc(int code, const char *url, const char * ref)
{
     printf("%d %s %s\n",code,url,ref);
}

__C_LINK static int __UDMCALL ShowReferers(UDM_AGENT * Indexer)
{
  int res;
  printf("\n          URLs and referers \n\n");
  res = UdmURLAction(Indexer, NULL, UDM_URL_ACTION_REFERERS);
  return(res);
}


#undef THINFO_TEST
#ifdef THINFO_TEST
/* CallBack function for Thread information */
void UdmThreadInfo(int handle,char *state, char* str)
{
  printf("%d %s %s\n",handle,state,str);
}
#endif


static int cmpgrp(const void *v1, const void *v2)
{
  int res;
  const UDM_CHARSET *c1=v1;
  const UDM_CHARSET *c2=v2;
  if ((res = strcasecmp(UdmCsGroup(c1), UdmCsGroup(c2))))
    return res;
  return strcasecmp(c1->name,c2->name);
}

static void display_charsets(FILE *file)
{
  UDM_CHARSET *cs=NULL;
  UDM_CHARSET c[100];
  size_t i=0;
  size_t n=0;
  int family=-1;
     
  for(cs=UdmGetCharSetByID(0) ; cs && cs->name ; cs++)
  {
    /* Skip not compiled charsets */
    if (cs->family != UDM_CHARSET_UNKNOWN)
      c[n++]=*cs;
  }
  fprintf(file,"\n%d charsets available:\n",n);

  UdmSort(c,n,sizeof(UDM_CHARSET),&cmpgrp);
  for(i=0;i<n;i++)
  {
    if (family!=c[i].family)
    {
      fprintf(file, "\n%19s : ", UdmCsGroup(&c[i]));
      family=c[i].family;
    }
    fprintf(file,"%s ",c[i].name);
  }
  fprintf(file,"\n");
}

static void UdmFeatures(UDM_VARLIST *V)
{
#ifdef HAVE_PTHREAD
  UdmVarListReplaceStr(V,"HAVE_PTHREAD","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_PTHREAD","no");
#endif
#ifdef USE_HTTPS
  UdmVarListReplaceStr(V,"USE_HTTPS","yes");
#else
  UdmVarListReplaceStr(V,"USE_HTTPS","no");
#endif
#ifdef DMALLOC
  UdmVarListReplaceStr(V,"DMALLOC","yes");
#else
  UdmVarListReplaceStr(V,"DMALLOC","no");
#endif
#ifdef EFENCE
  UdmVarListReplaceStr(V,"EFENCE","yes");
#else
  UdmVarListReplaceStr(V,"EFENCE","no");
#endif
#ifdef CHASEN
  UdmVarListReplaceStr(V,"CHASEN","yes");
#else
  UdmVarListReplaceStr(V,"CHASEN","no");
#endif
#ifdef MECAB
  UdmVarListReplaceStr(V,"MECAB","yes");
#else
  UdmVarListReplaceStr(V,"MECAB","no");
#endif
#ifdef HAVE_ZLIB
  UdmVarListReplaceStr(V,"HAVE_ZLIB","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_ZLIB","no");
#endif
#ifdef USE_SYSLOG
  UdmVarListReplaceStr(V,"USE_SYSLOG","yes");
#else
  UdmVarListReplaceStr(V,"USE_SYSLOG","no");
#endif
#ifdef USE_PARSER
  UdmVarListReplaceStr(V,"USE_PARSER","yes");
#else
  UdmVarListReplaceStr(V,"USE_PARSER","no");
#endif
#ifdef USE_MP3
  UdmVarListReplaceStr(V,"USE_MP3","yes");
#else
  UdmVarListReplaceStr(V,"USE_MP3","no");
#endif
#ifdef USE_FILE
  UdmVarListReplaceStr(V,"USE_FILE","yes");
#else
  UdmVarListReplaceStr(V,"USE_FILE","no");
#endif
#ifdef USE_HTTP
  UdmVarListReplaceStr(V,"USE_HTTP","yes");
#else
  UdmVarListReplaceStr(V,"USE_HTTP","no");
#endif
#ifdef USE_FTP
  UdmVarListReplaceStr(V,"USE_FTP","yes");
#else
  UdmVarListReplaceStr(V,"USE_FTP","no");
#endif
#ifdef USE_NEWS
  UdmVarListReplaceStr(V,"USE_NEWS","yes");
#else
  UdmVarListReplaceStr(V,"USE_NEWS","no");
#endif
#ifdef HAVE_MYSQL
  UdmVarListReplaceStr(V,"HAVE_MYSQL","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_MYSQL","no");
#endif
#ifdef HAVE_PGSQL
  UdmVarListReplaceStr(V,"HAVE_PGSQL","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_PGSQL","no");
#endif
#ifdef HAVE_IODBC
  UdmVarListReplaceStr(V,"HAVE_IODBC","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_IODBC","no");
#endif
#ifdef HAVE_UNIXODBC
  UdmVarListReplaceStr(V,"HAVE_UNIXODBC","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_UNIXODBC","no");
#endif
#ifdef HAVE_DB2
  UdmVarListReplaceStr(V,"HAVE_DB2","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_DB2","no");
#endif
#ifdef HAVE_SOLID
  UdmVarListReplaceStr(V,"HAVE_SOLID","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_SOLID","no");
#endif
#ifdef HAVE_VIRT
  UdmVarListReplaceStr(V,"HAVE_VIRT","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_VIRT","no");
#endif
#ifdef HAVE_EASYSOFT
  UdmVarListReplaceStr(V,"HAVE_EASYSOFT","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_EASYSOFT","no");
#endif
#ifdef HAVE_SAPDB
  UdmVarListReplaceStr(V,"HAVE_SAPDB","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_SAPDB","no");
#endif
#ifdef HAVE_IBASE
  UdmVarListReplaceStr(V,"HAVE_IBASE","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_IBASE","no");
#endif
#ifdef HAVE_CTLIB
  UdmVarListReplaceStr(V,"HAVE_CTLIB","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_CTLIB","no");
#endif
#ifdef HAVE_ORACLE8
  UdmVarListReplaceStr(V,"HAVE_ORACLE8","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_ORACLE8","no");
#endif
#ifdef HAVE_CHARSET_big5
  UdmVarListReplaceStr(V,"HAVE_CHARSET_big5","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_CHARSET_big5","no");
#endif
#ifdef HAVE_CHARSET_euc_kr
  UdmVarListReplaceStr(V,"HAVE_CHARSET_euc_kr","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_CHARSET_euc_kr","no");
#endif
#ifdef HAVE_CHARSET_gb2312
  UdmVarListReplaceStr(V,"HAVE_CHARSET_gb2312","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_CHARSET_gb2312","no");
#endif
#ifdef HAVE_CHARSET_japanese
  UdmVarListReplaceStr(V,"HAVE_CHARSET_japanese","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_CHARSET_japanese","no");
#endif
#ifdef HAVE_CHARSET_gbk
  UdmVarListReplaceStr(V,"HAVE_CHARSET_gbk","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_CHARSET_gbk","no");
#endif
#ifdef HAVE_CHARSET_gujarati
  UdmVarListReplaceStr(V,"HAVE_CHARSET_gujarati","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_CHARSET_gujarati","no");
#endif
#ifdef HAVE_CHARSET_tscii
  UdmVarListReplaceStr(V,"HAVE_CHARSET_tscii","yes");
#else
  UdmVarListReplaceStr(V,"HAVE_CHARSET_tscii","no");
#endif
}

#ifdef CONSOLE

static int usage(int level)
{
  FILE *file= stdout;
  fprintf(file, "\n");
  fprintf(file, "indexer from %s-%s-%s\n", PACKAGE, VERSION, UDM_DBTYPE);
  fprintf(file, "http://www.mnogosearch.org/ (C)1998-2009, LavTech Corp.\n");
  fprintf(file, "\n");
  fprintf(file, "Usage: indexer [OPTIONS]  [configfile]\n");
  fprintf(file, "\n");
  fprintf(file, "Indexing options:\n");

#ifdef HAVE_SQL
  fprintf(file, "  -a              reindex all documents even if not expired (may be\n");
  fprintf(file, "                  limited using -t, -u, -s, -c, -y and -f options)\n");
  fprintf(file, "  -m              reindex expired documents even if not modified (may\n");
  fprintf(file, "                  be limited using -t, -u, -c, -s, -y and -f options)\n");
  fprintf(file, "  -e              index 'most expired' (oldest) documents first\n");
  fprintf(file, "  -o              index documents with less depth (hops value) first\n");
  fprintf(file, "  -r              do not try to reduce remote servers load by randomising\n");
  fprintf(file, "                  url fetch list before indexing (-r recommended for very \n");
  fprintf(file, "                  big number of URLs)\n");
  fprintf(file, "  -n n            index only n documents and exit\n");
  fprintf(file, "  -c n            index only n seconds and exit\n");
  fprintf(file, "  -q              quick startup (do not add Server URLs)\n");
#endif

  fprintf(file, "  -b              block starting more than one indexer instances\n");
  fprintf(file, "  -i              insert new URLs (URLs to insert must be given using -u or -f)\n");
  fprintf(file, "  -p n            sleep n seconds after each URL\n");
  fprintf(file, "  -w              do not ask for confirmation when clearing documents from the database\n");

#ifdef HAVE_PTHREAD
  fprintf(file, "  -N n            run N threads\n");
#endif

  fprintf(file, "\n");

#ifdef HAVE_SQL
  fprintf(file, "Subsection control options (may be combined):\n");
  fprintf(file, "  -s status       limit indexer to documents matching status (HTTP Status code)\n");
  fprintf(file, "  -t tag          limit indexer to documents matching tag\n");
  fprintf(file, "  -g category     limit indexer to documents matching category\n");
  fprintf(file, "  -y content-type limit indexer to documents matching content-type\n");
  fprintf(file, "  -L language     limit indexer to documents matching language\n");
  fprintf(file, "  -u pattern      limit indexer to documents with URLs matching pattern\n");
  fprintf(file, "                  (supports SQL LIKE wildcard '%%')\n");
  fprintf(file, "  -D n            work with the n-th database only (i.e. with the n-th DBAddr)\n");
  fprintf(file, "  -f filename     read URLs to be indexed/inserted/cleared from file (with -a\n");
  fprintf(file, "                  or -C option, supports SQL LIKE wildcard '%%'; has no effect\n");
  fprintf(file, "                  when combined with -m option)\n");
  fprintf(file, "  -f -            Use STDIN instead of file as URL list\n");
#else
  fprintf(file, "URL options:\n");
  fprintf(file, "  -u URL          insert URL at startup\n");
  fprintf(file, "  -f filename     read URLs to be inserted from file\n");
#endif

  fprintf(file, "\n");
  fprintf(file, "Logging options:\n");

#ifdef LOG_PERROR
  fprintf(file, "  -l              do not log to stdout/stderr\n");
#endif

  fprintf(file, "  -v n            verbose level, 0-5\n");
  fprintf(file, "\n");
  fprintf(file, "Misc. options:\n");

#ifdef HAVE_SQL
  fprintf(file, "  -C              clear database and exit\n");
  fprintf(file, "  -S              print statistics and exit\n");
  fprintf(file, "  -j t            set current time for statistic (use with -S),\n");
  fprintf(file, "                  YYYY-MM[-DD[ HH[:MM[:SS]]]]\n");
  fprintf(file, "                  or time offset, e.g. 1d12h (see Period in indexer.conf)\n");
  fprintf(file, "  -I              print referers and exit\n");
  fprintf(file, "  -R              calculate popularity rank\n");
  fprintf(file, "  -Ecreate        create SQL table structure and exit\n");
  fprintf(file, "  -Edrop          drop SQL table structure and exit\n");
  fprintf(file, "  -Eblob          create fast search index\n");
  fprintf(file, "  -Ewordstat      create statistics for misspelled word suggestions\n");
#endif
  
  fprintf(file, "  -F pattern      print compile configuration and exit, e.g. -F '*'\n");
  fprintf(file, "  -h,-?           print help page and exit\n");
  fprintf(file, "  -hh             print more help and exit\n");
  fprintf(file, "  -d configfile   use given configfile instead of default one.\n");
  fprintf(file, "                  This option is usefull when running indexer as an\n");
  fprintf(file, "                  interpreter, e.g.: #!/usr/local/sbin/indexer -d\n");
  fprintf(file, "\n");
  fprintf(file, "\n");
  fprintf(file, "Please post bug reports and suggestions at http://www.mnogosearch.org/bugs/\n");

  if (level>1)display_charsets(file);
  return(0);
}

#endif /* ifndef CONSOLE */

UDM_AGENT Main;

/*
  Load indexer.conf and check if any DBAddr were given
*/
static int UdmIndexerEnvLoad(UDM_AGENT *Indexer, const char *fname,int lflags)
{
  int rc;
  if (UDM_OK == (rc= UdmEnvLoad(Indexer, fname, lflags)))
  {
    if (Indexer->Conf->dbl.nitems == 0)
    {
      sprintf(Indexer->Conf->errstr, "Error: '%s': No required DBAddr commands were specified", fname);
      rc= UDM_ERROR;
    }
  }
  return rc;
}


struct udm_indcmd2str_st
{
  const char *name;
  enum udm_indcmd cmd;
};

static struct udm_indcmd2str_st indcmd[]=
{
  {"index",         UDM_IND_INDEX},
  {"statistics",    UDM_IND_STAT},
  {"create",        UDM_IND_CREATE},
  {"drop",          UDM_IND_DROP},
  {"delete",        UDM_IND_DELETE},
  {"referers",      UDM_IND_REFERERS},
  {"sqlmon",        UDM_IND_SQLMON},
  {"checkconf",     UDM_IND_CHECKCONF},
  {"blob",          UDM_IND_MULTI2BLOB},
  {"export",        UDM_IND_EXPORT},
  {"wordstat",      UDM_IND_WRDSTAT},
  {"rewriteurl",    UDM_IND_REWRITEURL},
  {"rewritelimits", UDM_IND_REWRITELIMITS},
  {"hashspell",     UDM_IND_HASHSPELL},
  {"dumpspell",     UDM_IND_DUMPSPELL},
  {"dumpconf",      UDM_IND_DUMPCONF},
  {"",              UDM_IND_INDEX}
};

static enum udm_indcmd UdmIndCmd(const char *cmdname)
{
  struct udm_indcmd2str_st *cmd;
  size_t matches;
  enum udm_indcmd res= UDM_IND_INDEX;
  size_t cmdlen= strlen(cmdname);
  
  for (matches= 0, cmd= indcmd; cmd->name[0]; cmd++)
  {
    if (!strncasecmp(cmd->name, cmdname, cmdlen))
    {
      res= cmd->cmd;
      matches++;
    }
  }
  if (matches == 1)
    return res;
  return matches ? UDM_IND_AMBIGUOUS : UDM_IND_UNKNOWN;
}

/*
  Parse command line
*/
static int UdmARGC;
static char **UdmARGV;
static enum udm_indcmd cmd = UDM_IND_INDEX;
static int insert = 0, expire = 0, pop_rank = 0, mkind = 0, block = 0, help = 0;
static char *url_filename=NULL;
static int thd_errors= 0;

static void UdmParseCmdLine(void)
{
  int ch;

  /* 
    Available new options:
    Capital letters:  B    GH JK   OP   T VXWXYZ
    Small   letters:           k             x z
    Digits:          123456789
  */
  
  while ((ch = getopt(UdmARGC, UdmARGV, "QUCSIRMabheorlmqiw?-:E:F:t:u:s:n:v:L:A:D:p:N:f:c:g:y:j:d:D:")) != -1)
  {
    switch (ch)
    {
      case 'F':
      {
        UDM_VARLIST V,W;
        size_t i;
        UdmVarListInit(&V);
        UdmVarListInit(&W);
        UdmFeatures(&V);
        UdmVarListAddLst(&W,&V,NULL,optarg);
        for(i=0;i<W.nvars;i++)
          printf("%s:%s\n",W.Var[i].name,W.Var[i].val);
        exit(0);
      }
      case 'C': cmd= UDM_IND_DELETE;  add_servers=0;load_langmaps=0;load_spells=0;break;
      case 'S': cmd= UDM_IND_STAT;    add_servers=0;load_langmaps=0;load_spells=0;break;
      case 'I': cmd= UDM_IND_REFERERS;add_servers=0;load_langmaps=0;load_spells=0;break;
      case 'Q': cmd= UDM_IND_SQLMON;  add_servers=0;load_langmaps=0;load_spells=0;break;
      case 'E': cmd= UdmIndCmd(optarg);break;
      case 'R': pop_rank++; break;
      case 'M': mkind=1;break;
      case 'q': add_server_urls = 0; break;
      case 'l': log2stderr=0;break;
      case 'a': expire=1;break;
      case 'b': block++;break;
      case 'e': flags|=UDM_FLAG_SORT_EXPIRED;break;
      case 'o': flags|=UDM_FLAG_SORT_HOPS;break;
      case 'r': flags|=UDM_FLAG_DONTSORT_SEED; break;
      case 'm': flags|=UDM_FLAG_REINDEX;break;
      case 'n': Conf.url_number=atoi(optarg);break;
      case 'c': max_index_time=atoi(optarg);break;
      case 'v': loglevel= atoi(optarg); have_loglevel= 1; break;
      case 'p': seconds=atoi(optarg);break;
      case 't': UdmVarListAddStr(&Conf.Vars,"tag" , optarg);break;
      case 'g': UdmVarListAddStr(&Conf.Vars,"cat" , optarg);break;
      case 's': UdmVarListAddStr(&Conf.Vars, "status", optarg);break;
      case 'y': UdmVarListAddStr(&Conf.Vars,"type", optarg);break;
      case 'L': UdmVarListAddStr(&Conf.Vars,"lang", optarg);break;
      case 'u': UdmVarListAddStr(&Conf.Vars,"u"   , optarg);
        if (insert)
        {
          UDM_HREF Href;
          UdmHrefInit(&Href);
          Href.url=optarg;
          Href.method=UDM_METHOD_GET;
          UdmHrefListAdd(&Conf.Hrefs, &Href);
        }
        break;
      case 'N':
        maxthreads=atoi(optarg);
        UdmVarListReplaceInt(&Conf.Vars, "CrawlerThreads", maxthreads);
        break;
      case 'f': url_filename=optarg;break;
      case 'i': insert=1;break;
      case 'w': warnings=0;break;
      case 'j': UdmVarListAddStr(&Conf.Vars, "stat_time", optarg); break;
      case 'd': strncpy(cname, optarg, sizeof(cname));
        cname[sizeof(cname) - 1] = '\0';
        break;
      case 'D': UdmVarListAddStr(&Conf.Vars,"DBLimit"   , optarg); break;
      case '-':
        {
          char *arg= strchr(optarg, '=');
          if (arg)
          {
            *arg++= '\0';
            UdmVarListAddStr(&Conf.Vars, optarg, arg);
          }
          else
            help++;
        }
        break;
      case '?':
      case 'h':
      default:
        help++;
    }
  }
}



static int
UdmReloadEnv(UDM_AGENT *Indexer)
{
  UDM_ENV   NewConf;
  int  rc;

  UdmLog(Indexer,UDM_LOG_ERROR,"Reloading config '%s'",cname);
  UdmEnvInit(&NewConf);
  UdmSetLockProc(&NewConf,UdmLockProc);
  UdmSetRefProc(&NewConf,UdmRefProc);

  UDM_GETLOCK(Indexer, UDM_LOCK_CONF);
  Indexer->Conf = &NewConf;
  rc = UdmIndexerEnvLoad(Indexer, cname, add_servers + load_langmaps + UDM_FLAG_SPELL);
  Indexer->Conf = &Conf;
               
  if (rc!=UDM_OK)
  {
    UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
    UdmLog(Indexer,UDM_LOG_ERROR,"Can't load config: %s",UdmEnvErrMsg(&NewConf));
    UdmLog(Indexer,UDM_LOG_ERROR,"Continuing with old config");
    UdmEnvFree(&NewConf);
  }
  else
  {
    UdmEnvFree(&Conf);
    Conf=NewConf;
    UdmParseCmdLine();
#ifndef WIN32
    UdmOpenLog("indexer", &Conf, log2stderr);
#endif
    UDM_RELEASELOCK(Indexer, UDM_LOCK_CONF);
  }
  return UDM_OK;
}
  

static int
httpd_client_handler(int client, UDM_AGENT *A)
{
  char request[4096];
  char response[1024];
  char speed_info[128]= "";
  ssize_t nrecv,nsent;
  size_t i, len, total_docs= 0, total_sec= 0;
  udm_uint8 total_bytes= 0;
  time_t now= time(0);
  
  nrecv= recv(client, request, sizeof(request), 0);
  UdmLog(A, UDM_LOG_ERROR, "Received request len=%d", (int) nrecv);
  udm_snprintf(response, sizeof(response) - 1,
               "HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n");
  nsent= UdmSend(client, response, strlen(response), 0);
  len= sprintf(response,
               "<table border=1 cellspacing=1 cellpadding=1>\n"
               "<tr><th>ID</th>"
               "<th>Docs</th>"
               "<th>Size</th>"
               "<th>Task</th>"
               "<th>Time</th>"
               "<th>Param</th>"
               "<th>Extra</th></tr>");
  nsent= UdmSend(client, response, len, 0);
  for (i= 0; i < maxthreads; i++)
  {
    UDM_AGENT *Tmp= &ThreadIndexers[i];
    char mutex_owned_info[64]= "";
    size_t mutex, sec;
    
    for (mutex= 0;
         mutex < Tmp->State.nmutexes && mutex < UDM_AGENT_STATE_MUTEXES;
         mutex++)
    {
      sprintf(UDM_STREND(mutex_owned_info), " #%d",
              Tmp->State.mutex_owned[mutex]);
    }
    
    /*
    size_t mutex_owned= Tmp->State.mutex_owned;
    
    if (mutex_owned)
      sprintf(mutex_owned_info, "Owner for mitex #%d", mutex_owned);
    */
    
    len= sprintf(response,
                 "<tr><td>%d</td>"
                 "<td align=right>%d</td>"
                 "<td align=right>%llu</td>"
                 "<td>%s</td>"
                 "<td align=right>%d</td>"
                 "<td>%s&nbsp;</td>"
                 "<td>%s%s%s&nbsp;</td></tr>\n",
                 Tmp->handle,
                 Tmp->ndocs,
                 Tmp->nbytes,
                 Tmp->State.task,
                 (int) (now - Tmp->State.start_time),
                 UDM_NULL2EMPTY(Tmp->State.param),
                 mutex_owned_info[0] ? "Owner for mutex: " : "",
                 mutex_owned_info,
                 UDM_NULL2EMPTY(Tmp->State.extra));
    nsent= UdmSend(client, response, len, 0);
    total_docs+= Tmp->ndocs;
    total_bytes+= Tmp->nbytes;
    sec= (size_t) (now - Tmp->start_time);
    if (sec > total_sec)
      total_sec= sec;
  }
  if (total_sec)
  {
    udm_snprintf(speed_info, sizeof(speed_info) - 1,
                 "%d seconds, %d docs/sec, %d bytes/sec",
                 total_sec,
                 total_docs / total_sec,
                 (int) (total_bytes / total_sec));
  }
  
  len= sprintf(response,
                 "<tr><td>&nbsp;</td>"
                 "<td align=right>%d</td>"
                 "<td align=right>%llu</td>"
                 "<td>&nbsp;</td>"
                 "<td align=right>&nbsp;</td>"
                 "<td>%s&nbsp;</td>"
                 "<td>&nbsp;</td></tr>\n",
                 total_docs,
                 total_bytes,
                 speed_info);
  nsent= UdmSend(client, response, len, 0);
  len= sprintf(response, "</table>\n");
  nsent= UdmSend(client, response, len, 0);
  return UDM_OK;
}


#ifdef  WIN32
unsigned int __stdcall thread_main_httpd(void *arg)
#else
static void* thread_main_httpd(void *arg)
#endif
{
  UDM_AGENT *A= (UDM_AGENT*) arg;
#ifndef WIN32
  UdmStartHTTPD(A, httpd_client_handler);
#endif
  return 0;
}


#define UDM_NOTARGETS_SLEEP 10

#ifdef  WIN32
unsigned int __stdcall thread_main(void *arg)
{
  char *str_buf;
#else
static void * thread_main(void *arg)
{
#endif
  UDM_AGENT * Indexer = (UDM_AGENT *)arg;
  int res=UDM_OK;
  int done=0;
  int i_sleep=0;
 
  while (!done)
  {
    if (max_index_time>=0)
    {
      time_t now;
      time(&now);
      if ((now-Indexer->start_time)>max_index_time)
        break;
    }

    UDM_GETLOCK(Indexer, UDM_LOCK_THREAD);
    if (have_sighup)
    {
      UdmReloadEnv(Indexer);
      have_sighup=0;
    }
    
    if (have_sigint || have_sigterm)
    {
      int z;
      UdmLog(Indexer, UDM_LOG_ERROR, "%s received. Terminating. Please wait...", (have_sigint) ? "SIGINT" : "SIGTERM");
      for (z = 0 ; z < maxthreads; z++)
      {
        if (ThreadIndexers[z].handle)
          UdmAgentSetAction(&ThreadIndexers[z], UDM_TERMINATED);
      }
      UdmAgentSetAction(&Main, UDM_TERMINATED);
      have_sigint = have_sigterm = 0;
    }

    if (have_sigusr1)
    {
      UdmIncLogLevel(Indexer);
      have_sigusr1 = 0;
    }

    if (have_sigusr2)
    {
      UdmDecLogLevel(Indexer);
      have_sigusr2 = 0;
    }
    UDM_RELEASELOCK(Indexer, UDM_LOCK_THREAD);

    if (done)break;
          
    if (res == UDM_OK || res == UDM_NOTARGET)
    { /* Possible after bad startup */
      res= UdmIndexNextURL(Indexer);
    }
          
    UDM_GETLOCK(Indexer, UDM_LOCK_THREAD);
    cur_url_number++;
    UDM_RELEASELOCK(Indexer, UDM_LOCK_THREAD);

    switch(res)
    {
      case UDM_OK:
        if (i_sleep)
        {
          UDM_GETLOCK(Indexer, UDM_LOCK_THREAD);
          sleep_threads--;
          UDM_RELEASELOCK(Indexer, UDM_LOCK_THREAD);
          i_sleep= 0;
        }
        break;

               
      case UDM_NOTARGET:
#ifdef HAVE_PTHREAD
        /*
         in multi-threaded environment we
         should wait for a moment when every thread
         has nothing to do
        */
        UdmURLAction(Indexer, NULL, UDM_URL_ACTION_FLUSH); /* flush DocCache */

        if (!i_sleep)
        {
          UDM_GETLOCK(Indexer, UDM_LOCK_THREAD);
          sleep_threads++;
          UDM_RELEASELOCK(Indexer, UDM_LOCK_THREAD);
          i_sleep= 1;
        }

        UDM_GETLOCK(Indexer, UDM_LOCK_THREAD);
        done= (sleep_threads>=total_threads);
        UDM_RELEASELOCK(Indexer, UDM_LOCK_THREAD);
        break;
#else
        done= 1;
        break;
#endif
      case UDM_ERROR:
        thd_errors++;
#ifdef WIN32
        str_buf = (char *)UdmMalloc(1024);
        udm_snprintf(str_buf, 1024, "Error: %s",  UdmEnvErrMsg(Indexer->Conf));
        UdmShowInfo(0,NULL, str_buf);
        UDM_FREE(str_buf);
        UdmShowInfo(Indexer, "Error", UdmEnvErrMsg(Indexer->Conf));
#endif
      case UDM_TERMINATED:
#ifdef WIN32
        UdmShowInfo(Indexer, "Aborted", "");
#endif
      default:
#ifdef HAVE_PTHREAD
        /*
         in multi-threaded environment we
         should wait for a moment when every thread
         has nothing to do
        */
        if (!i_sleep)
        {
          if (res == UDM_ERROR)
            UdmLog(Indexer,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Indexer->Conf));
          UDM_GETLOCK(Indexer, UDM_LOCK_THREAD);
          sleep_threads++;
          UDM_RELEASELOCK(Indexer, UDM_LOCK_THREAD);
          i_sleep=1;
        }
        UDM_GETLOCK(Indexer, UDM_LOCK_THREAD);
        done= (sleep_threads>=total_threads);
        UDM_RELEASELOCK(Indexer, UDM_LOCK_THREAD);
        break;
#else
        if (res == UDM_ERROR)
          UdmLog(Indexer,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Indexer->Conf));
        done=1;
#endif
        break;
    }

    if ((seconds)&&(!done))
    {
      UdmAgentSetTask(Indexer, "Sleeping");
      UdmLog(Indexer,UDM_LOG_DEBUG,"Sleeping %d second(s)",seconds);
#ifndef WIN32
      Indexer->nsleepsecs+= seconds - UDMSLEEP(seconds);
#else
      Indexer->nsleepsecs+= seconds;
      UDMSLEEP(seconds);
#endif
    }
  
    if ((i_sleep) && (!done))
    {
      UdmLog(Indexer, UDM_LOG_ERROR, "%s, sleeping %d seconds (%d threads, %d sleeping)",
             (res == UDM_NOTARGET) ? "No targets" : 
             ((res == UDM_TERMINATED) ? "Terminating" : "An error occured"),
             UDM_NOTARGETS_SLEEP, total_threads, sleep_threads);
#ifndef WIN32
      Indexer->nsleepsecs+= UDM_NOTARGETS_SLEEP - UDMSLEEP(UDM_NOTARGETS_SLEEP);
#else
      Indexer->nsleepsecs+= UDM_NOTARGETS_SLEEP;
      UDMSLEEP(UDM_NOTARGETS_SLEEP);
#endif
    }
    UdmAgentSetTask(Indexer, "Unknown");
  }

  if (res!=UDM_ERROR)
  {
    time_t now, sec;
    float M = 0.0, K = 0.0;

    UdmURLAction(Indexer, NULL, UDM_URL_ACTION_FLUSH); /* flush DocCache */
    UdmWordCacheFlush(Indexer);

    time(&now);
    sec= now - Indexer->start_time - Indexer->nsleepsecs;
    if (sec > 0)
    {
      /* Convert to int64 - conversion from uint64 to double doesn't work on windows */
      M= ((udm_timer_t)Indexer->nbytes) / 1048576.0 / sec;
      if (M < 1.0) K = ((udm_timer_t)Indexer->nbytes) / 1024.0 / sec;
    }
    UdmLog(Indexer,UDM_LOG_ERROR,"Done (%d seconds, %u documents, %llu bytes, %5.2f %cbytes/sec.)",
           sec, Indexer->ndocs, Indexer->nbytes, (M < 1.0) ? K : M, (M < 1.0) ? 'K' : 'M' );
#if !defined(WIN32) && defined(HAVE_PTHREAD)
	{
      int z;
      for (z = 0 ; z < maxthreads; z++)
      if (ThreadIndexers[z].handle)
        pthread_kill(threads[z], SIGALRM); /* wake-up sleeping threads */
	}
#endif
  }
  UDM_GETLOCK(Indexer, UDM_LOCK_THREAD);
  total_threads--;
  UDM_RELEASELOCK(Indexer, UDM_LOCK_THREAD);

  UdmAgentFree(Indexer);

#ifdef WIN32
  return(0);
#else
  return(NULL);
#endif
}


static char pidname[1024];
static char time_pid[100];

static void exitproc(void)
{
  unlink(pidname);
}

#ifdef CONSOLE
static char * time_pid_info(void)
{
  struct tm * tim;
  time_t t;
  t= time(NULL);
  tim= localtime(&t);
  strftime(time_pid,sizeof(time_pid),"%a %d %H:%M:%S",tim);
  sprintf(time_pid+strlen(time_pid)," [%d]",(int)getpid());
  return(time_pid);
}
#endif

static void UdmWSAStartup(void)
{
#ifdef WIN32
  WSADATA wsaData;
  if (WSAStartup(0x101,&wsaData)!=0)
  {
    fprintf(stderr,"WSAStartup() error %d\n",WSAGetLastError);
    exit(1);
  }
#endif
}

static void UdmWSACleanup(void)
{
#ifdef WIN32
  WSACleanup();
#endif
  return;
}

static int UdmConfirm(const char *msg)
{
  char str[5];
  printf("%s",msg);
  return (fgets(str,sizeof(str),stdin) && !strncmp(str,"YES",3));
}

static int UdmClear(UDM_AGENT *A, const char *url_fname)
{
  int clear_confirmed=1;
  if (warnings)
  {
    size_t i;
    printf("You are going to delete content from the database(s):\n");
    for (i = 0; i < Conf.dbl.nitems; i++)
    {
      const char *dbaddr;
      dbaddr= UdmVarListFindStr(&Conf.dbl.db[i].Vars,"DBAddr","<noaddr>");
      printf("%s\n", dbaddr);
    }
    clear_confirmed=UdmConfirm("Are you sure?(YES/no)");
  }
     
  if (clear_confirmed)
  {
    if (url_fname)
    {
      if (UDM_OK != UdmURLFile(A,url_fname,UDM_URL_FILE_CLEAR))
      {
        UdmLog(A,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(A->Conf));
      }
    }
    else
    {
      printf("Deleting...");
      if (UDM_OK != UdmClearDatabase(A))
      {
        return UDM_ERROR;
      }
      printf("Done\n");
    }
  }
  else
  {
    printf("Canceled\n");
  }
  return UDM_OK;
}


static int
#ifdef WIN32
UdmThreadCreate(udm_thread_t *thd, unsigned (__stdcall *start_routine)(void *), void *arg)
#else
UdmThreadCreate(udm_thread_t *thd, void *(*start_routine)(void*), void *arg)
#endif
{
  int res;
#ifdef WIN32
  {
    res= _beginthreadex(NULL, 0, start_routine, arg, 0, NULL);
    assert(res != -1);
  }
#else
  {
    pthread_attr_t attr;
    size_t stksize= 1024 * 512;
    pthread_attr_init(&attr);
    pthread_attr_setstacksize(&attr, stksize);
    pthread_create(thd, &attr, start_routine, arg);
    pthread_attr_destroy(&attr);
  }
#endif
  return res;
}


static int UdmIndex(UDM_AGENT *A)
{
  UDM_AGENT httpd_agent;
  size_t nbytes;
  
  maxthreads= UdmVarListFindInt(&A->Conf->Vars, "CrawlerThreads", 1);
  nbytes= maxthreads * sizeof(UDM_AGENT);
  ThreadIndexers= (UDM_AGENT*)UdmMalloc(nbytes);
  bzero((void*) ThreadIndexers, nbytes);
  UdmAgentInit(&httpd_agent, A->Conf, 0);
  
  if (UdmVarListFind(&A->Conf->Vars, "Listen"))
  {
    udm_thread_t httpd_thread;
    UdmThreadCreate(&httpd_thread, thread_main_httpd, &httpd_agent);
  }
  
  
#ifdef HAVE_PTHREAD
  {
    int i;

#ifdef WIN32
    threads= (HANDLE*)UdmMalloc(maxthreads * sizeof(HANDLE));
#else
    threads= (pthread_t*)UdmMalloc(maxthreads * sizeof(pthread_t));
#endif
               
    for(i= 0; i < maxthreads; i++)
    {
      UDM_AGENT *Indexer;
      if (seconds) UDMSLEEP(seconds);

      UDM_GETLOCK(A, UDM_LOCK_THREAD);
      UDM_GETLOCK(A, UDM_LOCK_CONF);
      Indexer= UdmAgentInit(&ThreadIndexers[i], A->Conf, i + 1);
      Indexer->flags = flags;
      UDM_RELEASELOCK(A, UDM_LOCK_CONF);
      UDM_RELEASELOCK(A, UDM_LOCK_THREAD);
      
      UdmThreadCreate(&threads[i], &thread_main, Indexer);

      UDM_GETLOCK(A, UDM_LOCK_THREAD);
      total_threads= i + 1;
      UDM_RELEASELOCK(A,UDM_LOCK_THREAD);
    }
#ifndef WIN32        
    for (i = 0; i < maxthreads; i++)
      pthread_join(threads[i], NULL);
#else
    while(1)
    {
      int num;
      UDM_GETLOCK(A,UDM_LOCK_THREAD);
      num=total_threads;
      UDM_RELEASELOCK(A,UDM_LOCK_THREAD);
      if (!num)break;
      UDMSLEEP(1);
    }
#endif
    UDM_FREE(threads);
  }
#else
  Main.handle = 1;
  thread_main(&Main);
#endif

  UdmAgentFree(&httpd_agent);
  UDM_FREE(ThreadIndexers);
  return thd_errors ? UDM_ERROR : UDM_OK;
}


static int UdmBitrixIndex(UDM_AGENT *Agent)
{
  char buf[1024];
  UDM_DSTR docbuf;
  ssize_t rs;
  const char *document;
  UDM_DOCUMENT Doc;
  const char *url= NULL;
  const char *action= NULL;
  int rc= UDM_OK;
  size_t i;
  UDM_SERVER *Srv= NULL;
  
  UdmDSTRInit(&docbuf, 1024);
  while ((rs = read(0, buf, sizeof(buf))) > 0)
  {
    UdmDSTRAppend(&docbuf, buf, rs);
  }
  UdmParseQueryString(Agent,&Agent->Conf->Vars,docbuf.data);
  UdmDSTRFree(&docbuf);
  document= UdmVarListFindStr(&Agent->Conf->Vars, "document", NULL);
  if (!document)
  {
    printf("Error: no 'document' variable\n");
    goto ex;
  }
  UdmDocInit(&Doc);
  UdmVarListReplaceLst(&Doc.Sections, &Agent->Conf->Sections,NULL,"*");
  Doc.Buf.buf = Doc.Buf.content = UdmStrdup(document);
  Doc.lcs= Agent->Conf->lcs;

  if (UdmXMLParse(Agent, &Doc) != UDM_OK)
  {
    printf("Error: %s\n", UdmVarListFindStr(&Doc.Sections, "X-Reason", ""));
    goto ex;
  }
  
  for (i=0; i < Doc.TextList.nitems; i++)
  {
    UDM_TEXTITEM *t= &Doc.TextList.Item[i];
    UDM_VAR *Sec;
    printf("[%d] %s: %s\n", i, t->section_name, t->str);
    if (!strcmp(t->section_name, "document.url"))
    {
      Srv= UdmServerFind(Agent->Conf, &Agent->Conf->Servers, t->str, NULL);
      url= t->str;
    }
    else if (!strcmp(t->section_name, "document.action"))
    {
      action= t->str;
    }
    else if (!strcmp(t->section_name, "document.Last-Modified"))
    {
      UdmVarListReplaceStr(&Doc.Sections, "Last-Modified", t->str);
    }
    else if (! strcmp(t->section_name, "document.Content-Length"))
    {
      UdmVarListReplaceStr(&Doc.Sections, "Content-Length", t->str);
    }
    else if (! strcmp(t->section_name, "document.Charset"))
    {
      UdmVarListReplaceStr(&Doc.Sections, "Charset", t->str);
    }
    if (!strncmp(t->section_name, "document.",9))
    {
      size_t len= strlen(t->section_name + 9);
      memmove(t->section_name, t->section_name + 9, len + 1);
    }
    if (! t->section && (Sec = UdmVarListFind(&Doc.Sections, t->section_name)))
    {
      t->section = Sec->section;
    }
  }

  printf("URL: '%s'\n",url ? url : "NULL");
  printf("Action: '%s'\n", action ? action : "NULL");
  if (!action)
  {
    printf("No Action is given, exiting.\n");
    goto ex;
  }

  if (!strcmp(action, "DELETE"))
  {
    if (!url)
    {
      printf("No URL is given, exiting.");
      goto ex;
    }
    UdmVarListReplaceStr(&Agent->Conf->Vars, "u", url);
    UdmClearDatabase(Agent);
  }
  else if (!strcmp(action, "CLEAR"))
  {
    UdmClearDatabase(Agent);
  }
  else
  {
    /* Indexing */
    if (!url)
    {
      printf("No URL give, exiting.");
      rc= UDM_ERROR;
      goto ex;
    }

    if (!Srv)
    {
      printf("No 'Server' command for url, exiting.\n");
      goto ex;
    }
    UdmVarListReplaceLst(&Doc.Sections, &Srv->Vars, NULL, "*");

    for (i=0; i < Doc.Sections.nvars; i++)
    {
      UDM_VAR *var= &Doc.Sections.Var[i];
      printf("%s: %s\n", var->name, var->val);
    }

    UdmPrepareWords(Agent, &Doc);
    UdmVarListReplaceStr(&Doc.Sections, "URL", url);
    UdmVarListReplaceInt(&Doc.Sections, "Status", 200);
    UdmURLAction(Agent, &Doc, UDM_URL_ACTION_ADD);
    UdmURLAction(Agent, &Doc, UDM_URL_ACTION_FLUSH);
    UdmWordCacheFlush(Agent);
  }

ex:
  UdmDocFree(&Doc);
  /* FIXME: add freeing memory allocated in Doc etc. */
  return rc;
}

#ifdef CONSOLE

static void
UdmEnvSetDirs(UDM_ENV *Env)
{
#ifdef WIN32
  {
    char *conf_dir= WinUdmConfDir();
    UdmVarListReplaceStr(&Env->Vars,"ConfDir", conf_dir);
	{
	  char dir[256];
	  udm_snprintf(dir, sizeof(dir), "%s\\%s", conf_dir, "create");
      UdmVarListReplaceStr(&Env->Vars,"ShareDir", dir);
	}
    UdmVarListReplaceStr(&Env->Vars,"TmpDir", conf_dir);
    UdmVarListReplaceStr(&Env->Vars,"VarDir", conf_dir);
    UDM_FREE(conf_dir);
  }
#else
  {
    char *env;
    if (!(env= getenv("UDM_CONF_DIR")))
      env= getenv("UDM_ETC_DIR");
    UdmVarListReplaceStr(&Env->Vars, "ConfDir", env ?env : UDM_CONF_DIR);
    
    env= getenv("UDM_SHARE_DIR");
    UdmVarListReplaceStr(&Env->Vars, "ShareDir", env ? env : UDM_SHARE_DIR);
    
    env= getenv("UDM_VAR_DIR");
    UdmVarListReplaceStr(&Env->Vars, "VarDir", env ? env : UDM_VAR_DIR);
    
    if (!(env= getenv("UDM_TMP_DIR")))
      env= getenv("TMPDIR");
    UdmVarListReplaceStr(&Env->Vars, "TmpDir", env ? env : UDM_TMP_DIR);
  }
#endif

}

int main(int argc, char **argv)
{
  char  *language=NULL,*affix=NULL,*dictionary=NULL;
  int   pid_fd, rc= 0;
  char  pidbuf[1024];
  char  *REQUEST_METHOD= getenv("REQUEST_METHOD");
  FILE  *logfile= REQUEST_METHOD ? stdout : stderr;

#ifdef CHASEN
  char  *chasen_argv[] = { "chasen", "-b", "-f", "-F", "%m ", NULL };
  chasen_getopt_argv(chasen_argv, NULL);
#endif

  if (REQUEST_METHOD)
    printf("Content-Type: text/plain\r\n\r\n");
  
  UdmWSAStartup();
     
  UdmInit(); /* Initialize library */
     
  UdmInitMutexes();
  UdmEnvInit(&Conf);
  UdmVarListAddEnviron(&Conf.Vars,"ENV");
  UdmEnvSetDirs(&Conf);
  UdmSetLockProc(&Conf,UdmLockProc);
  UdmSetRefProc(&Conf,UdmRefProc);
#ifdef THINFO_TEST
  UdmSetThreadProc(&Conf,UdmShowThreadInfo);
#endif
  UdmAgentInit(&Main,&Conf,0);
     
  UdmARGC= argc;
  UdmARGV= argv;

  UdmParseCmdLine();
  
  if (cmd == UDM_IND_AMBIGUOUS)
  {
    fprintf(stderr, "Ambiguous indexer command in -E\n");
    help++;
  }
  
  if (cmd == UDM_IND_UNKNOWN)
  {
    fprintf(stderr, "Unknown indexer command in -E\n");
    help++;
  }

  if (cmd == UDM_IND_DUMPCONF)
  {
    load_for_dump|= UDM_FLAG_DONT_ADD_TO_DB;
    load_langmaps= 0;
    load_spells= 0;
  }
  else if (cmd != UDM_IND_INDEX)
  {
    add_servers=0;
    load_langmaps=0;
    if (cmd != UDM_IND_HASHSPELL && cmd  != UDM_IND_DUMPSPELL)
      load_spells=0;
  }

  flags|=add_servers;
  flags |= add_server_urls;
  Main.flags = flags;

  argc -= optind;
  argv += optind;

  if ((argc>1) || (help))
  {
    usage(help);
    UdmEnvFree(&Conf);
    return 1;
  }
  
  if (!*cname)
  {
    if (argc == 1)
    {
      strncpy(cname,argv[0],sizeof(cname));
      cname[sizeof(cname)-1]='\0';
    }
    else
    {
      const char *cd=UdmVarListFindStr(&Conf.Vars,"ConfDir",UDM_CONF_DIR);
      udm_snprintf(cname,sizeof(cname),"%s%s%s",cd,UDMSLASHSTR,"indexer.conf");
      cname[sizeof(cname)-1]='\0';
    }
  }
     
  if (UDM_OK != UdmIndexerEnvLoad(&Main, cname,
                                  add_servers + load_langmaps + load_spells +
                                  add_server_urls + load_for_dump))
  {
    fprintf(logfile, "%s\n", UdmEnvErrMsg(&Conf));
    UdmEnvFree(&Conf);
    return 1;
  }

 
  if (cmd == UDM_IND_DUMPCONF)
  {
    if (UDM_OK != (rc= UdmEnvSave(&Main, "-", 0)))
      fprintf(logfile, "%s\n", UdmEnvErrMsg(&Conf));
    UdmEnvFree(&Conf);
    return rc;
  }
     
  if (cmd == UDM_IND_CHECKCONF)
    return 0;
  
#ifndef WIN32
   UdmSetLogLevel(NULL, have_loglevel ? loglevel :
                  UdmVarListFindInt(&Main.Conf->Vars, "LogLevel", UDM_LOG_INFO));
   UdmOpenLog("indexer",&Conf, log2stderr);
#endif
   
  if (REQUEST_METHOD)
  {
    UdmBitrixIndex(&Main);
    return 0;
  }
  
  if (cmd == UDM_IND_HASHSPELL)
  {
    char errmsg[256];
    if (UDM_OK != (rc= UdmSpellListListLoad(&Conf.Spells, errmsg, sizeof(errmsg)))||
        UDM_OK != (rc= UdmSpellListListWriteHash(&Conf.Spells, errmsg, sizeof(errmsg))))
    {
      fprintf(stderr, "error: %s\n", errmsg);
      return 1;
    }
    return 0;
  }
  
  if (cmd == UDM_IND_DUMPSPELL)
  {
    char errmsg[256];
    int spflags= UDM_SPELL_NOPREFIX;
    if (UDM_OK != (rc= UdmSpellListListLoad(&Conf.Spells,
                                            errmsg, sizeof(errmsg)))||
        UDM_OK != (rc= UdmAffixListListLoad(&Conf.Affixes, spflags,
                                            errmsg, sizeof(errmsg)))||
        UDM_OK != (rc= UdmSpellDump(&Conf.Spells, &Conf.Affixes,
                                    errmsg, sizeof(errmsg))))
    {
      fprintf(stderr, "error: %s\n", errmsg);
      return 1;
    }
    return 0;
  }
  
  if (cmd==UDM_IND_SQLMON)
  {
    UDM_SQLMON_PARAM prm;
    bzero((void*)&prm,sizeof(prm));
    prm.infile= stdin;
    prm.outfile= stdout;
    prm.flags= UDM_SQLMON_DISPLAY_FIELDS;
    prm.gets= sqlmongets;
    prm.display= UdmDisplaySQLQuery;
    prm.prompt= sqlmonprompt;
    UdmSQLMonitor(&Main, &Conf, &prm);
    return 0;
  }

  if (cmd == UDM_IND_MULTI2BLOB)
  {
    rc= UdmMulti2Blob(&Main);
    return rc;
  }
  else if (cmd == UDM_IND_EXPORT)
  {
    rc= UdmExport(&Main);
    return rc;
  }
  else if (cmd == UDM_IND_WRDSTAT)
  {
    rc= UdmURLAction(&Main, NULL, UDM_URL_ACTION_WRDSTAT);
    return rc;
  }
  else if (cmd == UDM_IND_REWRITEURL)
  {
    rc= UdmRewriteURL(&Main);
    return rc;
  }     
  else if (cmd == UDM_IND_REWRITELIMITS)
  {
    rc= UdmRewriteLimits(&Main);
    return rc;
  }
     
  if (url_filename && strcmp(url_filename,"-"))
  {
    /* Make sure URL file is readable if not STDIN */
    FILE *url_file;
    if (!(url_file=fopen(url_filename,"r")))
    {
      UdmLog(&Main,UDM_LOG_ERROR,"Error: can't open url file '%s': %s",url_filename, strerror(errno));
      goto ex;
    }
    fclose(url_file);
  }
  
  if (insert && url_filename)
  {
    if (strcmp(url_filename,"-"))
    {
      /* Make sure all URLs to be inserted are OK */
      if (UDM_OK!=UdmURLFile(&Main, url_filename,UDM_URL_FILE_PARSE))
      {
        UdmLog(&Main,UDM_LOG_ERROR,"Error: Invalid URL in '%s'",url_filename);
        goto ex;
      }
    }      
    if (UDM_OK != UdmURLFile(&Main,url_filename,UDM_URL_FILE_INSERT))
    {
      UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
      goto ex;
    }
  }
     
  if (expire)
  {
    int  res;
          
    if (url_filename)
    {
      res= UdmURLFile(&Main,url_filename,UDM_URL_FILE_REINDEX);
    }
    else
    {
      res= UdmURLAction(&Main, NULL, UDM_URL_ACTION_EXPIRE);
    }
    if (res!=UDM_OK)
    {
      UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
      goto ex;
    }
  }
     
  if (affix||dictionary)
  {
    if (!language)
    {
      UdmLog(&Main,UDM_LOG_ERROR,"Error: Language is not specified for import!");
    }
    else if (strlen(language)!=2)
    {
      UdmLog(&Main,UDM_LOG_ERROR,"Error: Language should be 2 letters!");
    }
    goto ex;
  }

  switch(cmd)
  {
    case UDM_IND_DELETE:
      if (UDM_OK != UdmClear(&Main,url_filename))
      {
        UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
      }
      break;
    case UDM_IND_STAT:
      if (UDM_OK != ShowStatistics(&Main))
      {
        UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
      }
      break;
    case UDM_IND_REFERERS:
      if (UDM_OK != ShowReferers(&Main))
      {
        UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
      }
      break;
    case UDM_IND_CREATE:
    case UDM_IND_DROP:
      if (UDM_OK != CreateOrDrop(&Main,cmd))
      {
        UdmLog(&Main,UDM_LOG_ERROR,"Error: '%s'",UdmEnvErrMsg(Main.Conf));
      }
      break;
    default:
      {
        if (block)
        {
          /* Check that another instance isn't running and create PID file. */
          const char *vardir= UdmVarListFindStr(&Conf.Vars,"VarDir",UDM_VAR_DIR);
          sprintf(pidname,"%s/%s", vardir ,"indexer.pid");
          pid_fd = open(pidname,O_CREAT|O_EXCL|O_WRONLY,0644);
          if (pid_fd < 0)
          {
            fprintf(stderr,"%s Can't create '%s': %s\n", time_pid_info(), pidname, strerror(errno));
            if (errno == EEXIST)
            {
              fprintf(stderr,"It seems that another indexer is already running!\n");
              fprintf(stderr,"Remove '%s' if it is not true.\n",pidname);
            }
            goto ex;
          }
          sprintf(pidbuf,"%d\n",(int)getpid());
          write(pid_fd,&pidbuf,strlen(pidbuf));
#ifdef HAVE_ATEXIT
          atexit(&exitproc);
#endif
        }
        UdmLog(&Main,UDM_LOG_ERROR, "indexer from %s-%s-%s started with '%s'", PACKAGE, VERSION, UDM_DBTYPE, cname);
        UdmStoreHrefs(&Main);    /**< store hrefs from config and command line */
        UdmSigHandlersInit(&Main);
        rc= UdmIndex(&Main);
      }
  }
     
#ifdef HAVE_SQL
  if (pop_rank)
  {
    UdmSrvAction(&Main, NULL, UDM_SRV_ACTION_POPRANK);
  }
#endif
     
ex:
  total_threads= 0;
  UdmAgentFree(&Main);
  UdmEnvFree(&Conf);
  UdmDestroyMutexes();
  UdmWSACleanup();
#ifndef HAVE_ATEXIT
  exitproc();
#endif
  return rc;
}

#endif /* ifdef CONSOLE */
