/*
 * ConnectionMgr manages a single client connection, 
 * forwarding information from battery, network, and applications
 * 
 * Copyright © 2007, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307 USA.
 *
 */

#include "ConnectionMgr.h"
#include "main.h"
#include "Exception.h"
#include <iostream>
#include <sstream>
#include <sys/poll.h> //polling during read
#include <netdb.h>
#include <errno.h>
#include <unistd.h>

using namespace std;

ConnectionMgr::ConnectionMgr (int sockfd)
{
   m_sockfd = sockfd;
}

ConnectionMgr::~ConnectionMgr()
{
   //dbgout << "ConnectionMgr destructor!  (" << this << "):" << m_sockfd << endl;
   if (m_sockfd > 0) {
      close (m_sockfd);
      m_sockfd = -1;
   }
}

void ConnectionMgr::manageConnection()
{
   if (m_sockfd < 0) {
      return;
   }
   try {
      Message msg;
      //receive messages until error
      while (true) { 
	 TRACE ( "[" << m_sockfd << "] Listening for messages...");
	 receiveMessage (&msg, -1);
	 TRACE ( "[" << m_sockfd << "] Received message. Sending to handler.");
	 handleMsg(msg);
	 TRACE ( "[" << m_sockfd << "] Message handled.");
      }
   } catch (Exception &e) {
      TRACE ("[" << m_sockfd << "] Socket exception (" << e.getString() << ")");
   } catch (...) {
      TRACE ("Socket exception (Unknown!) (" << this << ") sockfd: " << m_sockfd);
   }
}

void ConnectionMgr::handleMsg (Message &inMsg)
{
   bool msgHandled = false;

   if (inMsg.isType(MSG_CONNECT_TEST)) {
      msgHandled = true;
      //tbd: reply
      return;
   } else if (inMsg.isType(MSG_APP_LAUNCH)) {
      string errMsg = "";
      ProcessMgr *pm = ProcessMgr::createInstance();
      if (!pm) {
	 return;
      }
      int retVal = pm->launchProcess (inMsg.getField("path"));
      if (retVal < 0) {
	 errMsg = pm->getErrorMessage();
      }

      //reply
      sleep(2);
      Message outMsg;
      stringstream ss;
      ss << "<AppLaunch id=\"" << inMsg.getField("id") << "\" status=\"" << retVal << "\" message=\"" << errMsg << "\"/>";
      outMsg.load (ss.str());
      sendMessage (outMsg);

      msgHandled = true;
      return;
   }

   if (!msgHandled) {
      stringstream err;
      err << "Invalid message type received: " << inMsg.getType();
      TRACE (err.str());
      //msg.set(RET_INVALID_TYPE, -1, err.str());
      //sendMessage(msg);
   }
}

//Get the XML input buffer.  Load the Message object
bool ConnectionMgr::receiveMessage (Message *msg, int msecTimeout)
{
   size_t sizeReceived = 0;
   if (!receiveBuffer (m_buf, &sizeReceived, msecTimeout)) {
      TRACE ("Error: receive Message failed");
      string err = "Error: ";
      err += m_buf;
      write (m_sockfd, err.c_str(), err.length());
      return false;
   }
   //write (m_sockfd, m_buf, sizeReceived);
   string xmlstr (m_buf, sizeReceived);
   int ret = msg->load(xmlstr);
   return ret;
}

bool ConnectionMgr::receiveBuffer (char *retBuf, size_t *retBufSize, int msecTimeout)
{
   if (m_sockfd < 0) {
      throw Exception (TCP_UNINITIALIZED, "ConnectionMgr", "receiveBuffer", 
			  "Socket is uninitialized");
   }
   if (!canReadData(msecTimeout)) {
      return false;
   }

   //TRACE ("dataCanRead (" << this << ") sockfd: " << m_sockfd << endl;

   int ret = read (m_sockfd, retBuf, MAX_BUF_SIZE);
   if (ret == 0) {
      throw Exception (TCP_EXIT_NORMAL, "ConnectionMgr", "receiveBuffer", "Client terminated");
   } else if (ret < 0) {
      char errBuf [256];
      strerror_r (errno, errBuf, 256);
      throw Exception (TCP_EXIT_NORMAL, "ConnectionMgr", "receiveBuffer", errBuf);
   } /*else if (strncmp (retBuf, XMLTAG, strlen(XMLTAG))!= 0) {
      TRACE ("Received: " << retBuf);
      return false;
      //throw Exception (MSG_INVALID_MSG, "ConnectionMgr", "Invalid message received", tmpBuf);
      }*/
   *retBufSize = ret;
   return true;
}


bool ConnectionMgr::canReadData (int msecTimeout)
{
   //see if data is available. Poll for msecTimeout seconds
   struct pollfd fds[1];
   fds[0].fd = m_sockfd;
   fds[0].events = POLLRDNORM; //POLLIN;

   int errVal;
   while (true) {
      errVal = poll(fds, 1, msecTimeout);
      if (errVal == -1) {
	 if (errno == EAGAIN || errno == EINTR) {
	    continue;
	 }
	 throw Exception (TCP_POLL_ERROR, "ConnectionMgr", "canReadData", 
			     "Socket poll error");
      }
      break; //may be 0 (timeout) or >0 (something to read)
   }
   return (errVal > 0); 
}


bool ConnectionMgr::canWriteData (int msecTimeout)
{
   //see if can write to socket
   struct pollfd fds[1];
   fds[0].fd = m_sockfd;
   fds[0].events = POLLWRNORM; 

   int errVal;
   while (true) {
      errVal = poll(fds, 1, msecTimeout);
      if (errVal == -1) {
	 if (errno == EAGAIN || errno == EINTR) {
	    continue;
	 }
	 throw Exception (TCP_POLL_ERROR, "ConnectionMgr", "canWriteData", 
			     "Socket poll error");
      }
      break;
   }
   return (errVal > 0);
}


bool ConnectionMgr::sendMessage (Message &msg)
{
   if (m_sockfd < 0) {
      throw Exception (TCP_UNINITIALIZED, "ConnectionMgr", "sendMessage", 
			  "Socket is uninitialized");
   }
   // send data
   if (!canWriteData (1000)) { //check for availability, wait 1s
      throw Exception (TCP_POLL_ERROR, "ConnectionMgr", "sendMessage", 
			  "Can not write to socket");
   }

   //very simple -- just send the xml string
   string xmlstr = msg.getData();
   if (xmlstr.empty()) {
      return false;
   }
   int bufSize = xmlstr.size()+1;
   char *sendBuf = new char[bufSize];
   memset(sendBuf, 0, bufSize);
   strncpy (sendBuf, xmlstr.c_str(), xmlstr.size());
   sendBuf[bufSize-1]='\0';
   TRACE ("Sending msg: " << xmlstr);

   int ret = write (m_sockfd, sendBuf, bufSize);
   delete (sendBuf);

   if (ret < 0) {
      throw Exception (TCP_WRITE_ERROR, "ConnectionMgr", "sendMessage", 
			  "Error writing to the socket");
   }
   return true;
}


