/*
 * SysMonitor.cpp performs the following tasks:
 *   - starts the thread listening for client connections
 *   - creates battStat class (which registers for ACPI events)
 *   - creates appStat class (used to start, stop, and monitor applications)
 * 	 - creates netStat class (used to start, stop, and monitor network, and get existing wireless info)
 * 
 * Copyright 2007, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307 USA.
 *
 */

#include <iostream> //includes all string
#include <sstream>
#include <sys/signal.h> 
#include <netdb.h> //for all socket structures
#include <pthread.h>
#include <sys/stat.h> //for umask (making daemon in init)

#include "main.h"
#include "errno.h"
#include "ConnectionMgr.h"
//#include "BattStat.h"
//#include "AppStat.h"
//#include "NetStat.h"

using namespace std;

// thread function for connection management.  Function format defined by pthread_create()
// This is the primary thread for managing client connections.
static void *connectionThread(void *arg)
{
   int clientfd = *(int*)arg;
   delete (int*)arg;

   pthread_detach (pthread_self()); //can no longer join (doesn't retain thread status or ret val)
   
   ConnectionMgr connMgr(clientfd);
   connMgr.manageConnection(); //will close clientfd in destructor

   TRACE ("Client exited");
   return (NULL);
}

void handleSignals()
{
   struct sigaction act, prev_act;

   act.sa_flags = 0;
   act.sa_handler = SIG_IGN;
   sigemptyset(&act.sa_mask);
   if (sigaction (SIGPIPE, &act, &prev_act) < 0) {
      TRACE ("Whoa! Could not ignore SIGPIPE");
   }
   if (sigaction (SIGCHLD, &act, &prev_act) < 0) {
      TRACE ("Whoa! Could not ignore SIGCHILD");
   }
}

int startClientListener(int clientPort, bool forkProcess)
{
   int ret, listenfd, clientfd;
   struct sockaddr_in clientSock, listenSock;
   socklen_t clientLen;
   pthread_t tid;

   cout << "Listening for clients on port: " << clientPort << endl;

   // create single listening socket 
   listenfd = socket (AF_INET, SOCK_STREAM, 0);
   if (listenfd < 0) {
      TRACE ("Error: cannot open socket");
      return -1;
   }

   memset (&listenSock, 0, sizeof(struct sockaddr_in));
   listenSock.sin_family = AF_INET;
   listenSock.sin_addr.s_addr = htonl(INADDR_ANY); //any network card (incase multihomed)
   listenSock.sin_port = htons(clientPort);

   int useAddr = 1;
   ret = setsockopt(listenfd, SOL_SOCKET, SO_REUSEADDR, &useAddr, sizeof(int));
   if (ret < 0) {
      perror("setsockopt");
      TRACE ("Error: Set socket option 'REUSEADDR' failed");
   }
   ret = bind (listenfd, (struct sockaddr*)&listenSock, sizeof(listenSock));
   if (ret < 0) {
      perror("bind");
      TRACE ("Error: Binding to socket failed. (Port=" << clientPort << ")");
      return -1;
   }
   ret = listen (listenfd, SOMAXCONN);
   if (ret < 0) {
      TRACE ("Error: socket listen failed");
      return -1;
   }

   //signal handling for threads
   /*   sigset_t sigvec; // signal handling 
   sigemptyset(&sigvec);
   sigaddset(&sigvec,SIGPIPE);
   sigaddset(&sigvec,SIGCHLD);
   pthread_sigmask(SIG_BLOCK,&sigvec,NULL); //blocks for 1 thread only
   */
   handleSignals(); //sets handlers for process (all threads)


   // At this point we know we are good to go.  Return success for daemon startup
   if (forkProcess) {
      int daemonRet = daemon (true,true); //int nochdir, int noclose);
   }

   while (true) {
      //wait for client connection and start new thread
      TRACE ("Waiting to accept client...");
      clientLen = sizeof(struct sockaddr_in);
      clientfd = accept (listenfd, (struct sockaddr*)&clientSock, &clientLen); //blocking call
      if (clientfd < 0) {
	 if (errno == EINTR) { //signal rcvd during accept
	    TRACE ("EINTR rcvd!");
	    continue; 
	 } else {
	    TRACE ("Error: accept failed. Exiting...");
	    break;
	 }
      }
      int *threadfd = new (int);
      *threadfd = clientfd; //so quick loop doesn't change fd
      pthread_create(&tid, NULL, &connectionThread, threadfd);
   }
   close(listenfd);
   return 0;
}

/*
void forkDaemon()
{
   pid_t pid = fork();
   //pid = 0;  //uncomment to debug
   pid = fork();
   if (pid<0) {      //client process
      return false;
   } else if (pid != 0) { //error, pid should be >0
      exit (0);
   }
   setsid();
   chdir ("/");
   umask (0);

   return true;
}
*/

void showHelp()
{
   cout << "Usage: sysMonitor [OPTIONS]" << endl
	<< "\t-f\t\t(sorry, not impl yet)Do not fork from the shell" << endl
	<< "\t-c PORT\t\tThe port for client connections" << endl
	<< "\t\t\t(default: " << DEFAULT_CLIENT_PORT << ")" << endl;
}

int main(int argc, char *argv[])
{
   //check arguments
   bool goodArgs = true;
   bool forkProcess = true;
   int clientPort = DEFAULT_CLIENT_PORT;

   string argStr;
   for (int i=1; i<argc; i++) {
      argStr = argv[i];
      //TRACE ("arg[" << i << "]:" << argStr);
      if (argStr=="-f") {
	 forkProcess = false;
      } else if (argStr=="-c") {
	 i++; //get next arg
	 if (i >= argc) { //error, no port specified
	    goodArgs = false;
	    break;
	 }
   	 stringstream ss;
	 ss << argv[i];
	 if (!(ss >> clientPort)) {
	    TRACE ("Invalid client port specified");
	    goodArgs = false;
	    break;
	 }
      } else if (argStr=="--pidfile") {
	 i++; //get next arg
	 if (i >= argc) { //error, no file specified
	    goodArgs = false;
	    break;
	 }
	 continue;
      } else {
	 goodArgs = false;
      }
   }

   if (!goodArgs) {
      showHelp();
      exit(1);
   }

   //listen for clients (indefinately)
   int ret = startClientListener(clientPort, forkProcess);
   TRACE ("System monitor exiting");
   exit (ret); 
}

/*
 back and forth better than barbarshop
 "starting..." maybe?
 "50% alpha--good idea, shows disabled"
 can click back and forth between 2
 need 10s timeout

*/
