#ifndef __BTP_TORRENT_H
#define __BTP_TORRENT_H

#include <apr.h>
#include <apr_pools.h>
#include <apr_network_io.h>
#include <apr_tables.h>

#include <libbtutil.h>
#include <libbtpeer/types/btp_peer.h>
#include <libbtpeer/types/btp_file_pool.h>
#include <libbtpeer/types/btp_peer_list.h>

#define BTP_TORRENT_MAX_PIECE   (1 << 14)
#define BTP_TORRENT_BLOCK(h)    ( \
    h->piece_size < BTP_TORRENT_MAX_PIECE ? \
        h->piece_size : BTP_TORRENT_MAX_PIECE \
)

#define BTP_TORRENT_MAX_PEERS   50

typedef struct btp_torrent_pieces_s {
    int         block_count;
    uint8_t*    block_bits;
    uint8_t*    block_downloaders;
    int*        block_missing;
    uint8_t*    piece_bits;
} btp_torrent_pieces;

/* Ideally, we're ACTIVE | ANNOUNCED | CONNECTED
 * It should be possible to be STOPPED | CHECK as well
 * (don't download the torrent, but check existing files)
 */

#define BTP_TORRENT_STATUS_STOPPED      0x0001 /* don't do anything */
#define BTP_TORRENT_STATUS_CHECK        0x0002 /* checking exisitng files */
#define BTP_TORRENT_STATUS_ACTIVE       0x0004 /* we're using this torrent */
#define BTP_TORRENT_STATUS_ANNOUNCED    0x0008 /* we've /announced ourselves as "started" */
#define BTP_TORRENT_STATUS_DOWNLOADING  0x0010 /* we've got traction */
#define BTP_TORRENT_STATUS_SEED         0x0020 /* we're a seed (set on startup, *or* when we send "completed") */
#define BTP_TORRENT_STATUS_ERROR        0x0040 /* we've had an *unrecoverable* error (out of HD space, etc) */
#define BTP_TORRENT_STATUS_QUIT         0x0080 /* shutting down */
#define BTP_TORRENT_STATUS_NEW          0x0100 /* torrent file was *just* added */

#define BTP_TORRENT_EVENT_STARTED       0
#define BTP_TORRENT_EVENT_STOPPED       1
#define BTP_TORRENT_EVENT_COMPLETED     2
#define BTP_TORRENT_EVENT_NONE          3

#define BTP_TORRENT_STATUS_CONNECTED    \
    (BTP_TORRENT_STATUS_DOWNLOADING | BTP_TORRENT_STATUS_SEEDING)

#define BTP_TORRENT_ANNOUNCE_RETRY      30

typedef struct btp_torrent_s {
    apr_pool_t*         pool;
    apr_port_t          port;
    /* our peer id */
    char                id[BT_PEERID_LEN];
    /* IP address to send to tracker */
    char                ip[BT_SHORT_STRING];
        
    
    bt_metainfo*        info;
    btp_file_pool*      files;
    int                 myfiles;
    
    int                 status;
    char                error[BT_SHORT_STRING];

    /* Where to download */
    char                destination[BT_PATH_LEN];
    
    /* How many bytes we ask for per request */
    int                 block_size;
    int                 block_count;
    
    btp_torrent_pieces* pieces;
    btp_peer_list*      peers;
    apr_table_t*        known_peers;

    uint64_t            downloaded;
    uint64_t            uploaded;
    time_t              start_t;        /* when we started up */
    time_t              l_announce_t;   /* last time we sent an announce */
    time_t              l_download_t;   /* last reception */
    time_t              l_upload_t;     /* last upload */
    time_t              l_peer_t;       /* last time we had a peer */
    time_t              n_announce_t;   /* next time we should announce */
} btp_torrent;

extern btp_torrent* btp_torrent_create(
    apr_pool_t* p, bt_metainfo* info, btp_file_pool* file_pool
);
extern int btp_torrent_piece_count_blocks(btp_torrent* tor, int piece);
extern uint64_t btp_torrent_bytes_left(btp_torrent* torrent);
extern void btp_torrent_destroy(btp_torrent* t);

extern apr_status_t btp_torrent_check_peer_input(btp_torrent* t, btp_peer* p);

extern apr_status_t btp_torrent_receive_peerlist(
    btp_torrent* t, bt_bcode* peers
);

static inline double btp_torrent_percent_complete(btp_torrent* torrent) {
    return (double)(
        (
            (double)(
                torrent->info->total_size - btp_torrent_bytes_left(torrent)
            ) / (double)torrent->info->total_size
        ) * 100
    );
}

static inline int btp_torrent_piece_to_block(
    btp_torrent* t, int piece, int poffset, int* boffset
) {
    int block =
        ((t->info->piece_size / t->block_size) * piece) +
        poffset / t->block_size;

    if(boffset)
        *boffset = poffset % t->block_size;
    return block;
}

extern apr_status_t btp_torrent_run(btp_torrent* t);

/* from btp_torrent/announce.c */
extern apr_status_t btp_torrent_announce(btp_torrent* t);

extern apr_status_t btp_torrent_send_announce(
    btp_torrent* t, const char* event, int want,
    char* result, apr_size_t* result_len
);



#endif
