/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
/* other libs */
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <db.h>
/* local */
#include <libbttracker.h>

static const char* default_reg_document =
    "<HTML>\n"
    " <HEAD>\n"
    "  <TITLE>BitTorrent Infohash Registration - mod_bt/"VERSION"</TITLE>\n"
    "  <LINK REL=\"stylesheet\" TYPE=\"text/css\" HREF=\"%s\">\n"
    " </HEAD>\n"
    " <BODY>\n"
    "  <H1>BitTorrent Infohash Registration</H1>\n"
    "  <FORM METHOD=\"GET\">\n"
    "   <TABLE CLASS=\"info\">\n"
    "    <TR><TH>Filename:</TH>"
    "<TD><INPUT TYPE=\"TEXT\" NAME=\"filename\"></TD></TR>\n"
    "    <TR><TH>Filesize:</TH>"
    "<TD><INPUT TYPE=\"TEXT\" NAME=\"filesize\"> bytes</TD></TR>\n"
    "    <TR><TH>Infohash:</TH>"
    "<TD><INPUT TYPE=\"TEXT\" NAME=\"info_hash_str\"></TD></TR>\n"
    "    <TR><TD COLSPAN=2 STYLE=\"text-align:center\">"
    "<INPUT TYPE=\"SUBMIT\" VALUE=\"Register\"></TD></TR>\n"
    "   </TABLE>\n"
    "  </FORM>\n"
    "  <BR/>\n"
    "  <HR/>\n"
    " </BODY>\n"
    "</HTML>\n"
;

int btt_cxn_register(
    btt_tracker* tracker, apr_pool_t* p, DB_TXN* ptxn,
    const char* args, struct sockaddr_in address,
    char** content, int* content_length
) {
    char* aargs = apr_pstrdup(p, args);
    char* rargs = apr_psprintf(p, "html=1");
    bt_form_pair fp = bt_next_form_pair_unesc(aargs);
    char* infohash = NULL;
    char* filename = NULL;
    bt_size_t filesize = 0;
    uint32_t s_hashes = 0;
    int rv = HTTP_OK;
    int ret = 0;
    DB_TXN* txn = NULL;
 
    tracker->s->server_time = time(NULL);
 
    if((ret = btt_txn_start(tracker, ptxn, &txn, 0)) != 0)
        return HTTP_SERVER_ERROR;
 
    while((rv == HTTP_OK) && (fp.key[0])) {
        if(!strcmp("filename", fp.key)) {
            filename = apr_pstrdup(p, fp.value);
        } else if(!strcmp("filesize", fp.key)) {
            filesize = (bt_size_t) atoll(fp.value);
        } else if(!strcmp("info_hash", fp.key)) {
            if(fp.value_len == BT_INFOHASH_LEN) {
                infohash = apr_palloc(p, BT_INFOHASH_LEN);
                memcpy(infohash, fp.value, BT_INFOHASH_LEN);
                s_hashes++;
            } else {
                rv = HTTP_BAD_REQUEST;
            }
        } else if(!strcmp("info_hash_str", fp.key)) {
            if(fp.value_len == BT_INFOHASH_LEN * 2) {
                infohash = bt_infohash_str(p, fp.value);
                s_hashes++;
            } else {
                rv = HTTP_BAD_REQUEST;
            }
        } else {
            rv = HTTP_BAD_REQUEST;
        }

        if(infohash) {
            if(
                (ret = btt_txn_register_hash(
                    tracker, p, txn, infohash, filename, filesize
                ))
                == 0
            )
                rargs = apr_pstrcat(
                    p, rargs, "&info_hash=", bt_str_esc_infohash(p, infohash),
                    NULL
                );
            else
                rv = HTTP_SERVER_ERROR;
  
            infohash = NULL;
            filename = NULL;
            filesize = 0;
        }

        if((rv == HTTP_OK) && (fp.eop))
            fp = bt_next_form_pair_unesc(fp.eop + 1);
        else
            fp.key[0] = 0;
    }

    if(rv != HTTP_OK)
        goto err;

    if(txn)
        if((ret = txn->commit(txn, 0)) != 0) {
            tracker->db.env->err(
                tracker->db.env, ret, "bt_cxn_register: commit()"
            );
            goto err;
        }
  
    if(s_hashes)
        return btt_cxn_scrape(
            tracker, p, ptxn, rargs, address, content, content_length
        );
 
    *content = apr_psprintf(p, default_reg_document, tracker->c->stylesheet);
    *content_length = strlen(*content);
    return HTTP_OK;

    err:
    *content = NULL;
    *content_length = 0;
 
    if(txn)
        txn->abort(txn);
 
    return rv;
}
