//
// Gendarme.Rules.Concurrency.WriteStaticFieldFromInstanceMethodRule
//
// Authors:
//	Sebastien Pouliot <sebastien@ximian.com>
//
// Copyright (C) 2006,2008 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections;
using System.Globalization;

using Mono.Cecil;
using Mono.Cecil.Cil;

using Gendarme.Framework;
using Gendarme.Framework.Rocks;

namespace Gendarme.Rules.Concurrency {

	[Problem ("This instance method writes to static fields. This may cause problem with multiple instances and in multithreaded applications.")]
	[Solution ("Move initialization to the static constructor or ensure appropriate locking.")]
	public class WriteStaticFieldFromInstanceMethodRule : Rule, IMethodRule {

		public RuleResult CheckMethod (MethodDefinition method)
		{
			// rule does not apply if 
			// - the method has no body (e.g. p/invokes, icalls don't)
			// - the method is static
			// - the method was generated by the compiler or a tool
			if (!method.HasBody || method.IsStatic || method.IsGeneratedCode ())
				return RuleResult.DoesNotApply;

			// *** ok, the rule applies! ***

			// look for stsfld instructions on static fields

			foreach (Instruction ins in method.Body.Instructions) {
				switch (ins.OpCode.Code) {
				case Code.Stsfld:
					FieldDefinition fd = (ins.Operand as FieldDefinition);
					// skip instance fields and generated static field (likely by the compiler)
					if ((fd != null) && fd.IsStatic && !fd.IsGeneratedCode ()) {
						string text = String.Format ("The static field '{0}', of type '{1}'. is being set in an instance method.", fd.Name, fd.FieldType);
						Runner.Report (method, ins, Severity.Medium, Confidence.High, text);
					}
					break;
				}
			}

			return Runner.CurrentRuleResult;
		}
	}
}
