//
// Gendarme.Rules.Naming.DetectNonAlphaNumericInTypeNamesRule
//
// Authors:
//	Nidhi Rawal <sonu2404@gmail.com>
//	Sebastien Pouliot <sebastien@ximian.com>
//
// Copyright (c) <2007> Nidhi Rawal
// Copyright (C) 2007-2008 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

using System;

using Mono.Cecil;
using Mono.Cecil.Cil;

using Gendarme.Framework;
using Gendarme.Framework.Rocks;

namespace Gendarme.Rules.Naming {

	[Problem ("This type, or method, name contains underscore(s) in its name.")]
	[Solution ("Remove the underscore from the name.")]
	public class DetectNonAlphanumericInTypeNamesRule: Rule, IMethodRule, ITypeRule {

		private const string GuidAttribute = "System.Runtime.InteropServices.GuidAttribute";
		private const string InterfaceTypeAttribute = "System.Runtime.InteropServices.InterfaceTypeAttribute";

		// Compiler generates an error for any other non alpha-numerics than underscore ('_'), 
		// so we just need to check the presence of underscore in method names
		private static bool CheckName (string name)
		{
			return (name.IndexOf ("_", StringComparison.Ordinal) == -1);
		}

		private static bool UsedForComInterop (TypeDefinition type)
		{
			return (type.IsInterface && type.HasAttribute (GuidAttribute) && 
				type.HasAttribute (InterfaceTypeAttribute));
		}

		public RuleResult CheckType (TypeDefinition type)
		{
			// type must be visible and not generated by the compiler (or a tool)
			if (!type.IsVisible () || type.IsGeneratedCode ())
				return RuleResult.DoesNotApply;

			// the rule does not apply if the code is an interface to COM objects
			if (UsedForComInterop (type))
				return RuleResult.DoesNotApply;

			// check the type name
			if (CheckName (type.Name))
				return RuleResult.Success;

			Runner.Report (type, Severity.Medium, Confidence.High, String.Empty);
			return RuleResult.Failure;
		}

		public RuleResult CheckMethod (MethodDefinition method)
		{
			// exclude non-public methods and special names (like Getter and Setter)
			if (!method.IsPublic || method.IsSpecialName || method.IsGeneratedCode ())
				return RuleResult.DoesNotApply;

			// the rule does not apply if the code is an interface to COM objects
			if (UsedForComInterop (method.DeclaringType.Resolve ()))
				return RuleResult.DoesNotApply;

			// check the method name
			if (CheckName (method.Name))
				return RuleResult.Success;

			Runner.Report (method, Severity.Medium, Confidence.High, String.Empty);
			return RuleResult.Failure;
		}
	}
}
