/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Fragments of this code are taken from Mozilla Spellchecker Component:
 *
 * The Initial Developer of the Original Code of Mozilla Spellchecker Component is
 * David Einstein.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 * Adapted for use of Voikko by Andris Pavenis <andris.pavenis@iki.fi>
 *
 * Contributor(s): David Einstein <Deinst@world.std.com>
 *                 Kevin Hendricks <kevin.hendricks@sympatico.ca>
 *                 Michiel van Leeuwen <mvl@exedo.nl>
 *                 Harri Pitkänen <hatapitk@iki.fi>
 *
 *  This spellchecker is based on the MySpell spellchecker made for Open Office
 *  by Kevin Hendricks.  Although the algorithms and code, have changed 
 *  slightly, the architecture is still the same. The Mozilla implementation
 *  is designed to be compatible with the Open Office dictionaries.
 *  Please do not make changes to the affix or dictionary file formats 
 *  without attempting to coordinate with Kevin.  For more information 
 *  on the original MySpell see 
 *  http://whiteboard.openoffice.org/source/browse/whiteboard/lingucomponent/source/spellcheck/myspell/
 *
 *  A special thanks and credit goes to Geoff Kuenning
 * the creator of ispell.  MySpell's affix algorithms were
 * based on those of ispell which should be noted is
 * copyright Geoff Kuenning et.al. and now available
 * under a BSD style license. For more information on ispell
 * and affix compression in general, please see:
 * http://www.cs.ucla.edu/ficus-members/geoff/ispell.html
 * (the home page for ispell)
 *
 * ***** END LICENSE BLOCK ***** */

#include "mozVoikkoSpell.hxx"
#include "mozVoikkoUtils.hxx"
#include <nsISimpleEnumerator.h>
#include <nsServiceManagerUtils.h>
#include <nsICategoryManager.h>
#include <mozISpellI18NManager.h>
#include <nsICharsetConverterManager.h>
#include <nsISupportsPrimitives.h>
#include <nsUnicharUtilCIID.h>
#include <nsUnicharUtils.h>
#include <nsStringAPI.h>
#include <nsEmbedString.h>
#include <nsCRTGlue.h>
#include <nsMemory.h>
#include <stdlib.h>

#define MOZVOIKKO_LANGUAGE_STRING "fi-FI"

const PRInt32 kFirstDirSize=8;
static NS_DEFINE_CID(kCharsetConverterManagerCID, NS_ICHARSETCONVERTERMANAGER_CID);
static NS_DEFINE_CID(kUnicharUtilCID, NS_UNICHARUTIL_CID);

NS_IMPL_ISUPPORTS1(mozVoikkoSpell, mozISpellCheckingEngine)

mozVoikkoSpell::mozVoikkoSpell()
{
    voikkoSpell = NULL;
}

mozVoikkoSpell::~mozVoikkoSpell()
{
    mPersonalDictionary = nsnull;
    delete voikkoSpell;
}

/* attribute wstring dictionary; */
NS_IMETHODIMP mozVoikkoSpell::GetDictionary(PRUnichar **aDictionary)
{
    NS_ENSURE_ARG_POINTER(aDictionary);

    *aDictionary = ToNewUnicode(mDictionary);
    return *aDictionary ? NS_OK : NS_ERROR_OUT_OF_MEMORY;
}

/* set the Dictionary.
 * This also Loads the dictionary and initializes the converter using the dictionaries converter
 */
NS_IMETHODIMP mozVoikkoSpell::SetDictionary(const PRUnichar *aDictionary)
{
    NS_ENSURE_ARG_POINTER(aDictionary);

    nsString newDict(aDictionary);
    nsresult rv = NS_OK;

    if (!newDict.Equals(NS_LITERAL_STRING(MOZVOIKKO_LANGUAGE_STRING)))
    {
        logMessage("mozVoikko: dictionary '%s' is not supported",
                   NS_ConvertUTF16toUTF8(newDict).get());
        return NS_ERROR_FAILURE;
    }

    if (!mDictionary.Equals(newDict))
    {
        mDictionary = aDictionary;
        // FIXME: So we need that?.
        delete voikkoSpell;

        voikkoSpell = new MozVoikko();
        if (!voikkoSpell)
            return NS_ERROR_FAILURE;

        nsCOMPtr<nsICharsetConverterManager> ccm = do_GetService(kCharsetConverterManagerCID, &rv);
        NS_ENSURE_SUCCESS(rv, rv);

        rv = ccm->GetUnicodeDecoder(voikkoSpell->get_dic_encoding(), getter_AddRefs(mDecoder));
        NS_ENSURE_SUCCESS(rv, rv);
        rv = ccm->GetUnicodeEncoder(voikkoSpell->get_dic_encoding(), getter_AddRefs(mEncoder));
        if (mEncoder && NS_SUCCEEDED(rv))
        {
            mEncoder->SetOutputErrorBehavior(mEncoder->kOnError_Signal, nsnull, '?');
        }

        NS_ENSURE_SUCCESS(rv, rv);
        mLanguage.Assign(newDict);

        return rv;
    }
    else
    {
        return NS_OK;
    } 
}

/* readonly attribute wstring language; */
NS_IMETHODIMP mozVoikkoSpell::GetLanguage(PRUnichar **aLanguage)
{
    NS_ENSURE_ARG_POINTER(aLanguage);

    *aLanguage = ToNewUnicode(mLanguage);
    return *aLanguage ? NS_OK : NS_ERROR_OUT_OF_MEMORY;
}

/* readonly attribute boolean providesPersonalDictionary; */
NS_IMETHODIMP mozVoikkoSpell::GetProvidesPersonalDictionary(PRBool *aProvidesPersonalDictionary)
{
    NS_ENSURE_ARG_POINTER(aProvidesPersonalDictionary);

    *aProvidesPersonalDictionary = PR_FALSE;
    return NS_OK;
}

/* readonly attribute boolean providesWordUtils; */
NS_IMETHODIMP mozVoikkoSpell::GetProvidesWordUtils(PRBool *aProvidesWordUtils)
{
    NS_ENSURE_ARG_POINTER(aProvidesWordUtils);

    *aProvidesWordUtils = PR_FALSE;
    return NS_OK;
}

/* readonly attribute wstring name; */
NS_IMETHODIMP mozVoikkoSpell::GetName(PRUnichar * *aName)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

/* readonly attribute wstring copyright; */
NS_IMETHODIMP mozVoikkoSpell::GetCopyright(PRUnichar * *aCopyright)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

/* attribute mozIPersonalDictionary personalDictionary; */
NS_IMETHODIMP mozVoikkoSpell::GetPersonalDictionary(mozIPersonalDictionary * *aPersonalDictionary)
{
    *aPersonalDictionary = mPersonalDictionary;
    NS_IF_ADDREF(*aPersonalDictionary);
    return NS_OK;
}

NS_IMETHODIMP mozVoikkoSpell::SetPersonalDictionary(mozIPersonalDictionary * aPersonalDictionary)
{
    mPersonalDictionary = aPersonalDictionary;
    return NS_OK;
}

/* void GetDictionaryList ([array, size_is (count)] out wstring dictionaries, out PRUint32 count); */
NS_IMETHODIMP mozVoikkoSpell::GetDictionaryList(PRUnichar ***aDictionaries, PRUint32 *aCount)
{
    if (!aDictionaries || !aCount)
        return NS_ERROR_NULL_POINTER;

    *aDictionaries = 0;
    *aCount = 0;

    PRUnichar **tmpPtr = (PRUnichar **) NS_Alloc(sizeof(PRUnichar *));
    if (!tmpPtr)
        return NS_ERROR_OUT_OF_MEMORY;

    do
    {
        MozVoikko voikkoSpell;
        if (voikkoSpell)
        {
            nsAutoString voikkoDictName(NS_LITERAL_STRING(MOZVOIKKO_LANGUAGE_STRING).get());
            tmpPtr[0] = ToNewUnicode(voikkoDictName);
            *aCount = 1;
            *aDictionaries = tmpPtr;
            return NS_OK;
        }
    } while(false);

    NS_Free(tmpPtr);	
    return NS_OK;
}

nsresult mozVoikkoSpell::ConvertCharset(const PRUnichar* aStr, char ** aDst)
{
    NS_ENSURE_ARG_POINTER(aDst);
    NS_ENSURE_TRUE(mEncoder, NS_ERROR_NULL_POINTER);

    PRInt32 outLength;
    PRInt32 inLength = NS_strlen(aStr);
    nsresult rv = mEncoder->GetMaxLength(aStr, inLength, &outLength);
    NS_ENSURE_SUCCESS(rv, rv);

    *aDst = (char *) NS_Alloc(sizeof(char) * (outLength+1));
    NS_ENSURE_TRUE(*aDst, NS_ERROR_OUT_OF_MEMORY);

    rv = mEncoder->Convert(aStr, &inLength, *aDst, &outLength);
    if (NS_SUCCEEDED(rv))
        (*aDst)[outLength] = '\0'; 

    return rv;
}

/* boolean Check (in wstring word); */
NS_IMETHODIMP mozVoikkoSpell::Check(const PRUnichar *aWord, PRBool *aResult)
{
    NS_ENSURE_ARG_POINTER(aWord);
    NS_ENSURE_ARG_POINTER(aResult);
    NS_ENSURE_TRUE(voikkoSpell, NS_ERROR_FAILURE);

    char *charsetWord;
    nsresult rv = ConvertCharset(aWord, &charsetWord);
    NS_ENSURE_SUCCESS(rv, rv);

    *aResult = voikkoSpell->spell(charsetWord);
  
    NS_Free(charsetWord);

    if (!*aResult && mPersonalDictionary) 
        rv = mPersonalDictionary->Check(aWord, mLanguage.get(), aResult);
  
    return rv;
}

/* void Suggest (in wstring word, [array, size_is (count)] out wstring suggestions, out PRUint32 count); */
NS_IMETHODIMP mozVoikkoSpell::Suggest(const PRUnichar *aWord, PRUnichar ***aSuggestions, PRUint32 *aSuggestionCount)
{
    NS_ENSURE_ARG_POINTER(aSuggestions);
    NS_ENSURE_ARG_POINTER(aSuggestionCount);
    NS_ENSURE_TRUE(voikkoSpell, NS_ERROR_FAILURE);

    nsresult rv;
    *aSuggestionCount = 0;

    char *charsetWord;
    rv = ConvertCharset(aWord, &charsetWord);
    NS_ENSURE_SUCCESS(rv, rv);

    char ** wlst;
    *aSuggestionCount = voikkoSpell->suggest(&wlst, charsetWord);
    NS_Free(charsetWord);

    // This code is copy/pasted from mySpell (with format changes). Maybe
    // it can be optimized.
    if (*aSuggestionCount)
    {
        *aSuggestions  = (PRUnichar **) NS_Alloc(*aSuggestionCount * sizeof(PRUnichar *));
        if (*aSuggestions)
        {
            PRUint32 index = 0;
            for (index = 0; index < *aSuggestionCount && NS_SUCCEEDED(rv); ++index)
            {
                // Convert the suggestion to utf16
                PRInt32 inLength = strlen(wlst[index]);
                PRInt32 outLength;
                rv = mDecoder->GetMaxLength(wlst[index], inLength, &outLength);
                if (NS_SUCCEEDED(rv))
                {
                    (*aSuggestions)[index] = (PRUnichar *) NS_Alloc(sizeof(PRUnichar) * (outLength+1));
                    if ((*aSuggestions)[index])
                    {
                        rv = mDecoder->Convert(wlst[index], &inLength, (*aSuggestions)[index], &outLength);
                        if (NS_SUCCEEDED(rv))
                            (*aSuggestions)[index][outLength] = 0;
                    } 
                    else
                        rv = NS_ERROR_OUT_OF_MEMORY;
                }
            }

            if (NS_FAILED(rv))
                NS_FREE_XPCOM_ALLOCATED_POINTER_ARRAY(index, *aSuggestions); // free the PRUnichar strings up to the point at which the error occurred
        }
        else // if (*aSuggestions)
            rv = NS_ERROR_OUT_OF_MEMORY;
    }

    voikkoSpell->freeSuggestions(wlst);

    return rv;
}
