/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#ifndef CCameraSensor_H
#define CCameraSensor_H

#include <mrpt/poses/CPose3D.h>
#include <mrpt/utils/CDebugOutputCapable.h>
#include <mrpt/hwdrivers/CGenericSensor.h>
#include <mrpt/vision/CImageGrabber_OpenCV.h>
#include <mrpt/vision/CImageGrabber_dc1394.h>

#include <mrpt/gui/CDisplayWindow.h>

namespace mrpt
{
	namespace hwdrivers
	{
		/** A container of different camera grabbers under the MRPT's "generic sensor" implementation.
		  *
		  *  \code
		  *  PARAMETERS IN THE ".INI"-LIKE CONFIGURATION STRINGS:
		  * -------------------------------------------------------
		  *   [supplied_section_name]
		  *    grabber_type       = opencv | dc1394   // Select one of two grabber implementations.
		  *    preview_decimation = 0     // N<=0 : No preview; N>0, display 1 out of N captured frames.
		  *
		  *    // Options for grabber_type= opencv
		  *    cv_camera_index  = 0       // [opencv] Number of camera to open
		  *    cv_camera_type   = CAMERA_CV_AUTODETECT
		  *    cv_frame_width   = 640     // [opencv] Capture width (not present or set to 0 for default)
		  *    cv_frame_height  = 480     // [opencv] Capture height (not present or set to 0 for default)
		  *    cv_fps           = 15      // [opencv] IEEE1394 cams only: Capture FPS (not present or 0 for default)
		  *    cv_gain          = 0       // [opencv] Camera gain, if available (nor present or set to 0 for default).
		  *
		  *    // Options for grabber_type= dc1394
		  *    dc1394_camera_guid   = 0 | 0x11223344    // 0 (or not present): the first camera; A hexadecimal number: The GUID of the camera to open
		  *    dc1394_camera_unit   = 0     			// 0 (or not present): the first camera; 0,1,2,...: The unit number (within the given GUID) of the camera to open (Stereo cameras: 0 or 1)
		  *    dc1394_frame_width	= 640
		  *    dc1394_frame_height	= 480
		  *    dc1394_framerate		= 15					// eg: 7.5, 15, 30, 60, etc... For posibilities see mrpt::vision::TCaptureOptions_dc1394
		  *    dc1394_mode7         = -1                    // -1: Ignore, i>=0, set to MODE7_i
		  *    dc1394_color_coding	= COLOR_CODING_YUV422	// For posibilities see mrpt::vision::TCaptureOptions_dc1394
		  *    dc1394_shutter		= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_gain			= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_gamma			= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_brightness	= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_exposure		= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_sharpness		= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *    dc1394_white_balance	= -1	// A value, or -1 (or not present) for not to change this parameter in the camera
		  *
		  *    // Pose of the sensor on the robot:
		  *    pose_x=0		; (meters)
		  *    pose_y=0
		  *    pose_z=0
		  *    pose_yaw=0	; (Angles in degrees)
		  *    pose_pitch=0
		  *    pose_roll=0
		  *
		  *  \endcode
		  *
		  *  - "grabber_type" is the class to use internally for image capturing. Choices are "opencv" (Windows & Linux) or "dc1394" (Linux only for now, requires libdc1394-2).
		  *  - For the meaning of cv_camera_type and other parameters, refer to mrpt::vision::CImageGrabber_OpenCV
		  *  - For the parameters of dc1394 parameters, refer to generic IEEE1394 documentation, and to mrpt::vision::TCaptureOptions_dc1394.
		  *
		  *  Images can be saved in the "external storage" mode. See setPathForExternalImages and setExternalImageFormat. These methods
		  *   are called automatically from rawlog-grabber.
		  *
		  *  \note The execution rate (in rawlog-grabber) should be greater than the required capture FPS.
		  *  \note In Linux you may need to execute "chmod 666 /dev/video1394/ * " and "chmod 666 /dev/raw1394" for allowing any user R/W access to firewire cameras.
		  *  \sa mrpt::vision::CImageGrabber_OpenCV, mrpt::vision::CImageGrabber_dc1394, CGenericSensor
		  */
		class HWDLLIMPEXP CCameraSensor : public utils::CDebugOutputCapable, public CGenericSensor
		{
			DEFINE_GENERIC_SENSOR(CCameraSensor)

		public:
			/** Constructor
			  *  The camera is not open until "initialize" is called.
			  */
			CCameraSensor();

			/** Destructor
			  */
			virtual ~CCameraSensor();

			/** This method should be called periodically (at least at 1Hz to capture ALL the real-time data)
			*  It is thread safe, i.e. you can call this from one thread, then to other methods from other threads.rip

			*  This method processes data from the GPS and update the object state accordingly.
			*/
			void  doProcess();

			/** Loads specific configuration for the device from a given source of configuration parameters, for example, an ".ini" file, loading from the section "[iniSection]" (see utils::CConfigFileBase and derived classes)
			  *  See hwdrivers::CCameraSensor for the possible parameters
			  */
			void  loadConfig(
				const mrpt::utils::CConfigFileBase &configSource,
				const std::string	  &iniSection );

			/** Tries to open the camera, after setting all the parameters with a call to loadConfig.
			  *  \exception This method must throw an exception with a descriptive message if some critical error is found.
			  */
			virtual void initialize();


			/** Close the camera (if open).
			  *   This method is called automatically on destruction.
			  */
			void close();


			/**  Set the path where to save off-rawlog images: empty (default) means save images embedded in the rawlog.
			  * \exception std::exception If the directory cannot be created
			  */
			void setPathForExternalImages( const std::string &directory );

			/**  Set the extension ("jpg","gif","png",...) that determines the format of images saved externally
			  *   The default is "jpg".
			  * \sa setPathForExternalImages
			  */
			void setExternalImageFormat( const std::string &ext )
			{
				m_external_images_format = ext;
			}

		protected:
			poses::CPose3D		m_sensorPose;
			std::string			m_sensorLabel;

			std::string			m_grabber_type; //!< Can be "opencv",...
			int					m_cv_camera_index;
			std::string			m_cv_camera_type;
			mrpt::vision::TCaptureCVOptions			m_cv_options;

			uint64_t			m_dc1394_camera_guid;
			int					m_dc1394_camera_unit;
			mrpt::vision::TCaptureOptions_dc1394	m_dc1394_options;
			int					m_preview_decimation;


			std::string			m_path_for_external_images; //!< The path where to save off-rawlog images: empty means save images embedded in the rawlog.
			std::string			m_external_images_format; //!< The extension ("jpg","gif","png",...) that determines the format of images saved externally \sa setPathForExternalImages

		private:
			/*  The OpenCV capture object. */
			mrpt::vision::CImageGrabber_OpenCV 	*m_cap_cv;

			/*  The dc1394 capture object. */
			mrpt::vision::CImageGrabber_dc1394 	*m_cap_dc1394;

			int							m_preview_counter;
			mrpt::gui::CDisplayWindow	*m_preview_win;

		}; // end class

	} // end namespace
} // end namespace

#endif
