/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Vicente Arevalo  <varevalo@ctima.uma.es>                      |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef opengl_CSetOfTexturedTriangles_H
#define opengl_CSetOfTexturedTriangles_H

#include <mrpt/opengl/CRenderizable.h>
#include <mrpt/utils/CMRPTImage.h>

namespace mrpt
{
	namespace opengl
	{
		class MRPTDLLIMPEXP CSetOfTexturedTriangles;

		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CSetOfTexturedTriangles, CRenderizable )

		/** A set of textured triangles.
		  *  This class can be used to draw any solid, arbitrarily complex object with textures.
		  *  \sa opengl::COpenGLScene
		  */
		class MRPTDLLIMPEXP CSetOfTexturedTriangles : public CRenderizable
		{
			DEFINE_SERIALIZABLE( CSetOfTexturedTriangles )

		public:
			/** Triangle vertex. This structure encapsulates the vertex coordinates and the image pixels.
			  */
			struct MRPTDLLIMPEXP TVertex
			{
			/** Default constructor.
			  */
				TVertex( ) :
					m_x(0.0), m_y(0.0), m_z(0.0), m_tx(0), m_ty(0) { }
			/** Constructor.
			  */
				TVertex(float x, float y, float z, unsigned int tx, unsigned int ty) :
					m_x(x), m_y(y), m_z(z), m_tx(tx), m_ty(ty) { }
			/** 3D vertex coordinates.
			  */
				float			m_x, m_y, m_z;
			/** 2D texture coordinates. Notice that the texture coordinates are 2D pixels!!!
			  */
				unsigned int	m_tx, m_ty;
			};

			/** Triangle. This structure encapsulates the triangle vertices.
			  */
			struct MRPTDLLIMPEXP TTriangle
			{
			/** Default constructor.
			  */
				TTriangle( ) :
					m_v1(), m_v2(), m_v3() { }
			/** Constructor.
			  */
				TTriangle(TVertex v1, TVertex v2, TVertex v3) :
					m_v1(v1), m_v2(v2), m_v3(v3) { }
			/** Vertices.
			  */
				TVertex	m_v1, m_v2, m_v3;
			};

		protected:
			mutable bool					m_init;
			mutable bool					m_enableTransparency;
			mutable unsigned int			m_glTextureName;
			mutable mrpt::utils::CMRPTImage	m_textureImage,
									m_textureImageAlpha;
			mutable int				r_width,
									r_height;

			/** VERY IMPORTANT: If you use a multi-thread application, you MUST call this from the same thread that will later destruct the object in order to the OpenGL texture memory to be correctly deleted.
			  *  Calling this method more than once has no effects. If you user one thread, this method will be automatically called when rendering, so there is no need to explicitly call it.
			  */
			void 		loadTextureInOpenGL() const;

			/** Triangle array.  */
			std::vector<TTriangle>	m_triangles;

		public:
			void clearTriangles() { m_triangles.clear(); }
			size_t getTrianglesCount() const { return m_triangles.size(); }
			void getTriangle(size_t idx, TTriangle &t) const { ASSERT_(idx<m_triangles.size()); t=m_triangles[idx]; }
			void insertTriangle( const TTriangle &t ) { m_triangles.push_back(t); }


			/** Class factory  */
			static CSetOfTexturedTrianglesPtr Create()
			{
				return CSetOfTexturedTrianglesPtr( new CSetOfTexturedTriangles() );
			}

			/** Render
			  */
			void  render() const;

			/** Ray Trace
			  */
			virtual bool traceRay(const mrpt::poses::CPose3D &o,float &dist) const;

			/** Assigns a texture and a transparency image, and enables transparency (If the images are not 2^N x 2^M, they will be internally filled to its dimensions to be powers of two)
			  */
			void  assignImage(
				const mrpt::utils::CMRPTImage&	img,
				const mrpt::utils::CMRPTImage&	imgAlpha );

			/** Assigns a texture image, and disable transparency.
			  */
			void  assignImage(
				const mrpt::utils::CMRPTImage&	img );

			/** Similar to assignImage, but the passed images will be returned as empty: it avoids making a copy of the whole image, just copies a pointer.
			  */
			void  assignImage_fast(
				mrpt::utils::CMRPTImage&	img,
				mrpt::utils::CMRPTImage&	imgAlpha );

			/** Similar to assignImage, but the passed images will be returned as empty: it avoids making a copy of the whole image, just copies a pointer.
			  */
			void  assignImage_fast(
				mrpt::utils::CMRPTImage&	img );

		private:
			/** Constructor
			  */
			CSetOfTexturedTriangles( ) :
				m_init(false),
				m_enableTransparency(false),
				m_glTextureName(0),
				m_triangles()
			{
			}

			/** Private, virtual destructor: only can be deleted from smart pointers */
			virtual ~CSetOfTexturedTriangles();
		};

	} // end namespace

} // End of namespace

#endif
