/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  MRPT_OS_H
#define  MRPT_OS_H

#include <mrpt/config.h>

#include <mrpt/utils/types.h>

#include <cstdarg>
#include <cstdlib>
#include <cstring>
#include <deque>
#include <vector>

// Duplicated here since MRPT_OS.h is the only header that cannot include "utils_defs.h"
#define _IAMINUTILSDEFS_H
#include <mrpt/utils/utils_impexp.h>  // DLL import/export definitions
#undef _IAMINUTILSDEFS_H

// Define a decl. modifier for printf-like format checks at compile time:
#ifdef __GNUC__
#	define MRPT_printf_format_check(_FMT_,_VARARGS_)  __attribute__ ((__format__ (__printf__, _FMT_,_VARARGS_)))
#else
#	define MRPT_printf_format_check(_FMT_,_VARARGS_)
#endif

// Define a decl. modifier for scanf-like format checks at compile time:
#ifdef __GNUC__
#	define MRPT_scanf_format_check(_FMT_,_VARARGS_)  __attribute__ ((__format__ (__scanf__, _FMT_,_VARARGS_)))
#else
#	define MRPT_scanf_format_check(_FMT_,_VARARGS_)
#endif


/** Used after member declarations */
#define MRPT_NO_THROWS		throw()

/** Represents an invalid timestamp, where applicable.
  */
#define INVALID_TIMESTAMP (0)


namespace mrpt
{
	namespace utils { class CFileOutputStream; }

	/** This namespace provides a OS-independent interface to many useful functions: filenames manipulation, time and date, string parsing, file I/O, threading, memory allocation, etc.
	 *  \sa mrpt::system::os
	 */
	namespace system
	{
		/** @name Threads
		@{*/

		/** This structure contains the information needed to interface the threads API on each platform:
		  * \sa createThread
		  */
		struct TThreadHandle
		{
#ifdef MRPT_OS_WINDOWS
			TThreadHandle()  :  //!< Sets the handle to a predefined value meaning it is uninitialized.
				hThread(NULL),
				idThread(0)
			{
			}

			/** Mark the handle as invalid.
			  * \sa isClear
			  */
			void clear()
			{
				idThread = 0;
				hThread  = NULL;
			}
			void			*hThread;		//!< The thread "HANDLE"
# if  defined(HAVE_OPENTHREAD) // defined(_MSC_VER) && (_MSC_VER>=1400)
			uintptr_t		idThread;		//!< The thread ID.
# else
			unsigned long	idThread;		//!< The thread ID.
# endif
#endif
#ifdef MRPT_OS_LINUX
			TThreadHandle() : idThread(0)  //!< Sets the handle to a predefined value meaning it is uninitialized.
			{
			}
			unsigned long	idThread;		//!< The thread ID.

			/** Mark the handle as invalid.
			  * \sa isClear
			  */
			void clear()
			{
				idThread = 0;
			}
#endif
			/** Returns true if the handle is uninitialized */
			bool isClear() { return idThread==0; }
		};

		/**  The type for cross-platform process (application) priorities.
		  * \sa changeCurrentProcessPriority
		  */
		enum TProcessPriority {
			ppIdle = 0,
			ppNormal,
			ppHigh,
			ppVeryHigh
		};

		/**  The type for cross-platform thread priorities.
		  * \sa changeThreadPriority
		  */
		enum TThreadPriority {
			tpLow = 0,
			tpNormal,
			tpHigh
		};

        /** Creates a new thread.
          *  This function creates, and start, a new thread running some code given by a function.
          *  The thread function should end by returning as normal.
          * \param func The function with the code to run in the thread.
          * \param param The parameter to be passed to the new thread function.
          * \return A structure that represents the thread (it contains its ID and, in Windows, its HANDLE).
          * \exception std::exception If the operation fails
          * \sa endThread, joinThread, changeThreadPriority
          */
        TThreadHandle MRPTDLLIMPEXP createThread(
             void       ( *func )( void * ),
             void       *param
             );

        /** Waits until the given thread ends.
          * \sa createThread, endThread
          */
        void MRPTDLLIMPEXP joinThread( const TThreadHandle &threadHandle );

        /** Returns the ID of the current thread.
          */
        unsigned long MRPTDLLIMPEXP getCurrentThreadId() MRPT_NO_THROWS;

		/** Returns the creation and exit times of the current thread and its CPU time consumed.
		  * \param creationTime The creation time of the thread.
		  * \param exitTime The exit time of the thread, or undefined if it is still running.
		  * \param cpuTime The CPU time consumed by the thread, in seconds.
          * \exception std::exception If the operation fails
          * \sa getCurrentThreadId, createThread
          */
		void MRPTDLLIMPEXP getCurrentThreadTimes(
			time_t			&creationTime,
			time_t			&exitTime,
			double			&cpuTime );

        /** Change the priority of the given thread.
          * \sa createThread, changeCurrentProcessPriority
          */
        void MRPTDLLIMPEXP changeThreadPriority( const TThreadHandle &threadHandle, TThreadPriority priority );

        /** Change the priority of the given process (it applies to all the threads, plus independent modifiers for each thread).
          * \sa createThread, changeThreadPriority
          */
        void MRPTDLLIMPEXP changeCurrentProcessPriority( TProcessPriority priority );

		/**  @} */


		/** This namespace provides a OS-independent interface to low-level functions.
		 *   Most of these functions are converted into calls to standard functions, unless we are into Visual Studio 2005 (or newer). In that case the secure version
		 *     of the standard library functions (prefix "_s") are used instead.
		 */
		namespace os
		{
			/** An OS-independent version of sprintf (Notice the bufSize param, which may be ignored in some compilers)
			  *  \sa utils::format
			  */
			int MRPTDLLIMPEXP sprintf(char *buf, size_t bufSize, const char *format, ...) MRPT_NO_THROWS MRPT_printf_format_check(3,4);

			/** An OS-independent version of vsprintf (Notice the bufSize param, which may be ignored in some compilers)
			  */
			int MRPTDLLIMPEXP vsprintf(char *buf, size_t bufSize, const char *format, va_list args) MRPT_NO_THROWS;

			/** An OS-independent version of vsnprintf (Notice the bufSize param, which may be ignored in some compilers)
			  */
			int MRPTDLLIMPEXP vsnprintf(char *buf, size_t bufSize, const char *format, va_list args) MRPT_NO_THROWS;

			/** An OS-independent version of fopen.
			  * \return It will always return NULL on any error.
			  */
			FILE MRPTDLLIMPEXP *fopen(const char *fileName,const char *mode) MRPT_NO_THROWS;

			/** An OS-independent version of fopen (std::string version)
			  * \return It will always return NULL on any error.
			  */
			FILE MRPTDLLIMPEXP *fopen(const std::string &fileName,const char *mode) MRPT_NO_THROWS;

			/** An OS-independent version of fprintf
			  */
			int MRPTDLLIMPEXP fprintf(FILE *fil, const char *format, ...)  MRPT_NO_THROWS MRPT_printf_format_check(2,3);

			/** An OS-independent version of fscanf
			  * \return The number of fields correctly assigned
			  */
			int MRPTDLLIMPEXP fscanf(FILE *fil, const char *format, ...)  MRPT_NO_THROWS MRPT_scanf_format_check(2,3);

			/** An OS-independent version of fclose.
			  */
			void MRPTDLLIMPEXP fclose(FILE *f) MRPT_NO_THROWS;

			/** An OS-independent version of strcat.
			  * \return It will always return the "dest" pointer.
			  */
			char MRPTDLLIMPEXP * strcat(char *dest, size_t destSize, const char *source) MRPT_NO_THROWS;

			/** An OS-independent version of strcpy.
			  * \return It will always return the "dest" pointer.
			  */
			char  MRPTDLLIMPEXP *strcpy(char *dest, size_t destSize, const char *source) MRPT_NO_THROWS;

			/** An OS-independent version of strcmp.
			  * \return It will return 0 when both strings are equal, casi sensitive.
			  */
			int MRPTDLLIMPEXP _strcmp(const char*str1,const char*str2) MRPT_NO_THROWS;

			/** An OS-independent version of strcmpi.
			  * \return It will return 0 when both strings are equal, casi insensitive.
			  */
			int MRPTDLLIMPEXP _strcmpi(const char*str1,const char*str2) MRPT_NO_THROWS;

			/** An OS-independent version of strtoll.
			  */
			int64_t MRPTDLLIMPEXP _strtoll(const char *nptr, char **endptr, int base);

			/** An OS-independent version of strtoull.
			  */
			uint64_t MRPTDLLIMPEXP _strtoull(const char *nptr, char **endptr, int base);


			/** An OS and compiler independent version of "memcpy"
			  */
			void MRPTDLLIMPEXP memcpy(
				void		*dest,
				size_t		destSize,
				const void	*src,
				size_t		copyCount ) MRPT_NO_THROWS;

			/** An OS-independent version of getch, which waits until a key is pushed.
			  * \return The pushed key code
			  */
			int MRPTDLLIMPEXP getch() MRPT_NO_THROWS;

			/** An OS-independent version of kbhit, which returns true if a key has been pushed.
			  */
			bool MRPTDLLIMPEXP kbhit() MRPT_NO_THROWS;

		}	// end namespace "os"

		/** An OS-independent method for sending the current thread to "sleep" for a given period of time.
		  * \param time_ms The sleep period, in miliseconds.
		  */
		void MRPTDLLIMPEXP sleep( int time_ms ) MRPT_NO_THROWS;

		/** An OS-independent method for tokenizing a string.
		  * The extra parameter "context" must be a pointer to a "char*" variable, which needs no initialization and is used to save information between calls to strtok.
		  * \sa system::tokenize
		  */
		char MRPTDLLIMPEXP  *strtok( char *str, const char *strDelimit, char **context ) MRPT_NO_THROWS;

		/** Tokenizes a string according to a set of delimiting characters.
		  * Example:
		  * \code
		  std::vector<std::string>	tokens;
		  tokenize( " - Pepe-Er  Muo"," -",tokens);
		  * \endcode
		  *
		  *  Will generate 3 tokens:
		  *		- "Pepe"
		  *		- "Er"
		  *		- "Muo"
		  */
		void  MRPTDLLIMPEXP tokenize(
			const std::string			&inString,
			const std::string			&inDelimiters,
			std::deque<std::string>		&outTokens ) MRPT_NO_THROWS;

		/** Tokenizes a string according to a set of delimiting characters.
		  * Example:
		  * \code
		  std::vector<std::string>	tokens;
		  tokenize( " - Pepe-Er  Muo"," -",tokens);
		  * \endcode
		  *
		  *  Will generate 3 tokens:
		  *		- "Pepe"
		  *		- "Er"
		  *		- "Muo"
		  */
		void MRPTDLLIMPEXP  tokenize(
			const std::string			&inString,
			const std::string			&inDelimiters,
			std::vector<std::string>		&outTokens ) MRPT_NO_THROWS;

		/**  Removes leading and trailing spaces:
		  */
		std::string MRPTDLLIMPEXP trim(const std::string &str);

        /** Returns true if the number is NaN. */
        bool MRPTDLLIMPEXP  isNaN(float  f) MRPT_NO_THROWS;

        /** Returns true if the number is NaN. */
        bool  MRPTDLLIMPEXP isNaN(double f) MRPT_NO_THROWS;

        /** Returns true if the number is non infinity. */
        bool MRPTDLLIMPEXP  isFinite(float f) MRPT_NO_THROWS;

        /** Returns true if the number is non infinity.  */
        bool  MRPTDLLIMPEXP isFinite(double f) MRPT_NO_THROWS;

#ifdef HAVE_LONG_DOUBLE
		/** Returns true if the number is NaN. */
        bool  MRPTDLLIMPEXP isNaN(long double f) MRPT_NO_THROWS;

        /** Returns true if the number is non infinity. */
        bool MRPTDLLIMPEXP  isFinite(long double f) MRPT_NO_THROWS;
#endif

		/** Shows the message "Press any key to continue" to the current standard output and returns when a key is pressed.
		  */
		void MRPTDLLIMPEXP pause() MRPT_NO_THROWS;

		/** Clears the console window */
		void MRPTDLLIMPEXP clearConsole();

		/** In platforms and compilers with support to "alloca", allocate a memory block on the stack; if alloca is not supported, it is emulated as a normal "malloc" - NOTICE: Since in some platforms alloca will be emulated with malloc, alloca_free MUST BE ALWAYS CALLED to avoid memory leaks.
		  *   This method MUST BE a macro rather than a function in order to operate on the caller's stack.
		  *  \sa mrpt_alloca_free
		  */
#if defined(_MSC_VER) && (_MSC_VER>=1400)
		// Visual Studio 2005, 2008
#	define		mrpt_alloca( nBytes )	_malloca(nBytes)
#elif defined(HAVE_ALLOCA)
		// GCC
#	define		mrpt_alloca( nBytes )	::alloca(nBytes)
#else
	// Default: Emulate with memory in the heap:
#	define		mrpt_alloca( nBytes )	::malloc( nBytes );
#endif

		/** This method must be called to "free" each memory block allocated with "system::alloca": If the block was really allocated in the stack, no operation is actually performed, otherwise it will be freed from the heap.
		  *   This method MUST BE a macro rather than a function in order to operate on the caller's stack.
		  * \sa mrpt_alloca
		  */
#if defined(_MSC_VER) && (_MSC_VER>=1400)
		// Visual Studio 2005, 2008
#	define		mrpt_alloca_free( mem_block )	_freea(mem_block)
#elif defined(HAVE_ALLOCA)
		// GCC
#	define		mrpt_alloca_free( mem_block )
#else
	// Default: Emulate with memory in the heap:
#	define		mrpt_alloca_free( mem_block )	free(mem_block)
#endif


		/** @name Time and date functions
		@{*/

		/** A system independent time type, it holds the the number of 100-nanosecond intervals since January 1, 1601 (UTC).
		 * \sa system::getCurrentTime, system::timeDifference, INVALID_TIMESTAMP, TTimeParts
		 */
		typedef uint64_t TTimeStamp;

		/** The parts of a date/time (it's like the standard 'tm' but with fractions of seconds).
		 * \sa TTimeStamp, timestampToParts, buildTimestampFromParts
		 */
		struct TTimeParts
		{
			uint16_t	year;	/** The year */
			uint8_t		month;  /** Month (1-12) */
			uint8_t		day;    /** Day (1-31) */
			uint8_t		hour;   /** Hour (0-23) */
			uint8_t		minute; /** Minute (0-59) */
			double		second; /** Seconds (0.0000-59.9999) */
			uint8_t		day_of_week; /** Day of week (1:Sunday, 7:Saturday) */
			int			daylight_saving;
		};

		/** Builds a timestamp from the parts (Parts are in UTC)
		  * \sa timestampToParts
		  */
		mrpt::system::TTimeStamp MRPTDLLIMPEXP buildTimestampFromParts( const mrpt::system::TTimeParts &p );

		/** Builds a timestamp from the parts (Parts are in local time)
		  * \sa timestampToParts, buildTimestampFromParts
		  */
		mrpt::system::TTimeStamp MRPTDLLIMPEXP buildTimestampFromPartsLocalTime( const mrpt::system::TTimeParts &p );

		/** Gets the individual parts of a date/time (days, hours, minutes, seconds)
		  * \sa buildTimestampFromParts
		  */
		void MRPTDLLIMPEXP timestampToParts( TTimeStamp t, TTimeParts &p );

		/** Returns the current (UTC) system time.
		  * \sa now,getCurrentLocalTime
		  */
		mrpt::system::TTimeStamp  MRPTDLLIMPEXP getCurrentTime( );

		/** A shortcut for system::getCurrentTime
		  * \sa getCurrentTime, getCurrentLocalTime
		 */
		inline mrpt::system::TTimeStamp  MRPTDLLIMPEXP now()
		{
			return getCurrentTime();
		}

		/** Returns the current (local) time.
		  * \sa now,getCurrentTime
		  */
		mrpt::system::TTimeStamp  MRPTDLLIMPEXP getCurrentLocalTime( );

		/** Transform from standard "time_t" (actually a double number, it can contain fractions of seconds) to TTimeStamp.
		  * \sa timestampTotime_t
		  */
		mrpt::system::TTimeStamp  MRPTDLLIMPEXP time_tToTimestamp( const double &t );

		/** Transform from standard "time_t" to TTimeStamp.
		  * \sa timestampTotime_t
		  */
		mrpt::system::TTimeStamp  MRPTDLLIMPEXP time_tToTimestamp( const time_t &t );

		/** Transform from TTimeStamp to standard "time_t" (actually a double number, it can contain fractions of seconds).
		  * \sa time_tToTimestamp
		  */
		double MRPTDLLIMPEXP timestampTotime_t( const mrpt::system::TTimeStamp  &t );

		/** Retuns the time difference from t1 to t2 (positive if t2 is posterior to t1), in seconds.
          	*/
		double MRPTDLLIMPEXP timeDifference( const mrpt::system::TTimeStamp &t_first, const mrpt::system::TTimeStamp &t_later );

		/** Transform a time interval (in seconds) into TTimeStamp (e.g. which can be added to an existing valid timestamp)
		  */
		mrpt::system::TTimeStamp MRPTDLLIMPEXP secondsToTimestamp( const double &nSeconds );

		/** Returns a formated string with the given time difference (passed as the number of seconds), as a string [H]H:MM:SS.MILISECS
		  */
		std::string MRPTDLLIMPEXP formatTimeInterval( const double &timeSeconds );

		/** Convert a timestamp into this textual form (UTC time): YEAR/MONTH/DAY,HH:MM:SS.MMM
		  * \sa dateTimeLocalToString
		  */
		std::string  MRPTDLLIMPEXP dateTimeToString(const mrpt::system::TTimeStamp &t);

		/** Convert a timestamp into this textual form (in local time): YEAR/MONTH/DAY,HH:MM:SS.MMM
		  * \sa dateTimeToString
		  */
		std::string  MRPTDLLIMPEXP dateTimeLocalToString(const mrpt::system::TTimeStamp &t);

		/** Convert a timestamp into this textual form: YEAR/MONTH/DAY
		 */
		std::string  MRPTDLLIMPEXP dateToString(const mrpt::system::TTimeStamp &t);

		/** Returns the number of seconds ellapsed from midnight in the given timestamp
		 */
		double  MRPTDLLIMPEXP extractDayTimeFromTimestamp(const mrpt::system::TTimeStamp &t);

		/** Convert a timestamp into this textual form (UTC): HH:MM:SS.MMMMMM
		 */
		std::string  MRPTDLLIMPEXP timeToString(const mrpt::system::TTimeStamp &t);

		/** Convert a timestamp into this textual form (in local time): HH:MM:SS.MMMMMM
		 */
		std::string  MRPTDLLIMPEXP timeLocalToString(const mrpt::system::TTimeStamp &t);

		/** @} */

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  MRPTDLLIMPEXP vectorToTextFile( const std::vector<float> &vec, const std::string &fileName, bool append = false, bool byRows=false );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  MRPTDLLIMPEXP vectorToTextFile( const std::vector<double> &vec, const std::string &fileName, bool append = false, bool byRows=false );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  MRPTDLLIMPEXP vectorToTextFile( const std::vector<int> &vec, const std::string &fileName, bool append = false, bool byRows=false );

		/** A useful function for debuging, which saves a std::vector into a text file (compat. with MATLAB)
		* \return Returns false on any error, true on everything OK.
		*/
		bool  MRPTDLLIMPEXP vectorToTextFile( const std::vector<size_t> &vec, const std::string &fileName, bool append = false, bool byRows=false );

		/** Saves a vector directly as a binary dump to a file:
		* \return Returns false on any error, true on everything OK.
		*/
		bool MRPTDLLIMPEXP vectorToBinaryFile( const vector_byte &vec, const std::string &fileName );

		/** Loads a entire file as a vector of bytes.
		* \return Returns false on any error, true on everything OK.
		*/
		bool MRPTDLLIMPEXP loadBinaryFile( vector_byte &out_data, const std::string &fileName );

		/** Returns the memory occupied by this process, in bytes
		  */
		unsigned long  MRPTDLLIMPEXP getMemoryUsage();



		/** @name Directories, files, and file names
		@{*/

		/** Returns the name of a proposed temporary file name */
		std::string MRPTDLLIMPEXP getTempFileName();

		/** Returns the current working directory.
		  */
		std::string MRPTDLLIMPEXP getcwd();

		/** Creates a directory
		  * \return Returns false on any error, true on everything OK.
		  * \todo Seems to return false on Linux even after creating the directory OK.
		  */
		bool  MRPTDLLIMPEXP createDirectory( const std::string &dirName );

		/** Deletes a single file. For multiple files see deleteFiles
		  * \return Returns false on any error, true on everything OK.
		  * \sa deleteFiles
		  */
		bool MRPTDLLIMPEXP deleteFile( const std::string &fileName );

		/** Delete one or more files, especified by the (optional) path and the file name (including '?' or '*') - Use forward slash ('/') for directories for compatibility between Windows and Linux, since they will be internally traslated into backward slashes ('\') if MRPT is compiled under Windows.
		  * \sa deleteFile
		  */
		void MRPTDLLIMPEXP deleteFiles(const std::string &s);

		/** Renames a file - If the target path is different and the filesystem allows it, it will be moved to the new location.
		  * \return false on any error. In that case, if a pointer to a receiver string is passed in error_msg, a description of the error is saved there.
		  */
		bool MRPTDLLIMPEXP renameFile( const std::string &oldFileName, const std::string &newFileName,  std::string *error_msg=NULL );

		/** Delete all the files in a given directory
		  * \sa deleteFile
		  */
		void MRPTDLLIMPEXP deleteFilesInDirectory(const std::string &s);

		/** Extract just the name (without extension) of a filename from a complete path plus name plus extension.
		  *  This function works for either "/" or "\" directory separators.
		  * \sa extractFileExtension,extractFileDirectory
		  */
		std::string  MRPTDLLIMPEXP extractFileName(const std::string& filePath);

		/** Extract the extension of a filename.
		  *  For example, for "dummy.cpp", it will return "cpp".
		  *  If "ignore_gz" is true, the second extension will be returned if the file name
		  *   ends in ".gz", for example, for "foo.map.gz", this will return "map".
		  * \sa extractFileName,extractFileDirectory
		  */
		std::string  MRPTDLLIMPEXP extractFileExtension(const std::string &filePath, bool ignore_gz = false );

		/** Extract the whole path (the directory) of a filename from a complete path plus name plus extension.
		  *  This function works for either "/" or "\" directory separators.
		  * \sa extractFileName,extractFileExtension
		  */
		std::string  MRPTDLLIMPEXP extractFileDirectory(const std::string &filePath);

		/** Test if a given file (or directory) exists.
		 */
		bool  MRPTDLLIMPEXP fileExists(const std::string& fileName);

		/** Replace invalid filename chars by underscores ('_').
		  *  Invalid chars are identified by those not being alphanumeric or: ".-#%$&()+[]{}"
		  */
		std::string MRPTDLLIMPEXP fileNameStripInvalidChars( const std::string &filename);

		/** @} */

		/** Returns a lower-case version of a string.
		  * \sa toLowerCase
		  */
		std::string  MRPTDLLIMPEXP toUpperCase(const std::string& str);

		/** Returns an upper-case version of a string.
		  * \sa toUpperCase
		  */
		std::string  MRPTDLLIMPEXP toLowerCase(const std::string& str);

		/** Returns a lower-case version of a string.
		  * \sa lowerCase
		  */
		std::string  MRPTDLLIMPEXP upperCase(const std::string& str);

		/** Returns an upper-case version of a string.
		  * \sa upperCase
		  */
		std::string  MRPTDLLIMPEXP lowerCase(const std::string& str);


		/** Returns the MRPT compilation date
		  */
		std::string MRPTDLLIMPEXP MRPT_getCompilationDate();

		/** Returns a string describing the MRPT version including the SVN number.
		  */
		std::string MRPTDLLIMPEXP MRPT_getVersion();

		/** Call this to register handlers for fatal erros (memory access,etc) that show useful debug information (It is called automatically normally, no need for the user to explicitly call this method.).
		  */
		void MRPTDLLIMPEXP registerFatalExceptionHandlers();

		/** Decodes a UTF-8 string into an UNICODE string.
		  *  See http://en.wikipedia.org/wiki/UTF-8  and http://www.codeguru.com/cpp/misc/misc/multi-lingualsupport/article.php/c10451/.
		  */
		void MRPTDLLIMPEXP decodeUTF8( const std::string &strUTF8, vector_word &out_uniStr );

		/** Encodes a 2-bytes UNICODE string into a UTF-8 string.
		  *  See http://en.wikipedia.org/wiki/UTF-8 and http://www.codeguru.com/cpp/misc/misc/multi-lingualsupport/article.php/c10451/.
		  */
        void MRPTDLLIMPEXP encodeUTF8( const vector_word &input, std::string &output );

	} // End of namespace

} // End of namespace

#endif
