/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include "zlib.h"

#include <mrpt/compress/zip.h>
#include <mrpt/system/os.h>
#include <mrpt/utils/CStream.h>

#include <mrpt/utils/CFileGZOutputStream.h>
#include <mrpt/utils/CFileGZInputStream.h>

using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::compress::zip;

/*---------------------------------------------------------------
						compress
---------------------------------------------------------------*/
void  mrpt::compress::zip::compress(
	void						*inData,
	size_t						inDataSize,
	std::vector<unsigned char>	&outData)
{
	int ret=0;
	MRPT_TRY_START;

	unsigned long	resSize;

	outData.resize(inDataSize+inDataSize/1000+50);
	resSize = (unsigned long)outData.size();
	ret = ::compress(
		&outData[0],
		&resSize,
		(unsigned char*)inData,
		(unsigned long)inDataSize
		);
	ASSERT_(ret==Z_OK);

	outData.resize(resSize);

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("[zlib] Error code=%i\n",ret);\
		);
}

/*---------------------------------------------------------------
						compress
---------------------------------------------------------------*/
void  mrpt::compress::zip::compress(
	const std::vector<unsigned char>	&inData,
	std::vector<unsigned char>			&outData)
{
	int ret=0;
	MRPT_TRY_START;

	unsigned long	resSize;

	outData.resize(inData.size()+inData.size()/1000+50);
	resSize = (unsigned long)outData.size();
	ret = ::compress(
		&outData[0],
		&resSize,
		&inData[0],
		(unsigned long)inData.size()
		);
	ASSERT_(ret==Z_OK);

	outData.resize(resSize);

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("[zlib] Error code=%i\n",ret);\
		);
}

/*---------------------------------------------------------------
						compress
---------------------------------------------------------------*/
void  mrpt::compress::zip::compress(
	void						*inData,
	size_t						inDataSize,
	CStream						&out)
{
	int ret=0;
	MRPT_TRY_START;

	unsigned long	resSize;
	std::vector<unsigned char>	outData;

	outData.resize(inDataSize+inDataSize/1000+50);
	resSize = (unsigned long)outData.size();

	ret = ::compress(
		&outData[0],
		&resSize,
		(unsigned char*)inData,
		(unsigned long)inDataSize
		);
	ASSERT_(ret==Z_OK);

	outData.resize(resSize);

	// Write to stream:
	out.WriteBuffer( &outData[0], resSize );

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("[zlib] Error code=%i\n",ret);\
		);
}


/*---------------------------------------------------------------
						compress
---------------------------------------------------------------*/
void  mrpt::compress::zip::compress(
	const std::vector<unsigned char>	&inData,
	CStream						&out)
{
	int ret=0;
	MRPT_TRY_START;

	unsigned long	resSize;
	std::vector<unsigned char>	outData;
	outData.resize(inData.size()+inData.size()/1000+50);
	resSize = (unsigned long)outData.size();

	ret = ::compress(
		&outData[0],
		&resSize,
		&inData[0],
		(unsigned long)inData.size()
		);

	ASSERT_(ret==Z_OK);

	outData.resize(resSize);

	// Write to stream:
	out.WriteBuffer( &outData[0], resSize );

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("[zlib] Error code=%i\n",ret);\
		);
}

/*---------------------------------------------------------------
						decompress
---------------------------------------------------------------*/
void  mrpt::compress::zip::decompress(
	void						*inData,
	size_t						inDataSize,
	std::vector<unsigned char>	&outData,
	size_t						outDataEstimatedSize)
{
	int ret=0;
	MRPT_TRY_START;

	outData.resize( outDataEstimatedSize );
	unsigned long	actualOutSize;

	ret = ::uncompress(
		&outData[0],
		&actualOutSize,
		(unsigned char*)inData,
		(unsigned long)inDataSize
		);

	ASSERT_(ret==Z_OK);

	outData.resize(actualOutSize);

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("[zlib] Error code=%i\n",ret);\
		);
}

/*---------------------------------------------------------------
						decompress
---------------------------------------------------------------*/
void  mrpt::compress::zip::decompress(
	void						*inData,
	size_t						inDataSize,
	void						*outData,
	size_t						outDataBufferSize,
	size_t						&outDataActualSize)
{
	int ret=0;
	MRPT_TRY_START;

	unsigned long	actualOutSize = (unsigned long)outDataBufferSize;

	ret = ::uncompress(
		(unsigned char*)outData,
		&actualOutSize,
		(unsigned char*)inData,
		(unsigned long)inDataSize
		);

	ASSERT_(ret==Z_OK);

	outDataActualSize = actualOutSize;

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("[zlib] Error code=%i\n",ret);\
		);
}

/*---------------------------------------------------------------
						decompress
---------------------------------------------------------------*/
void  mrpt::compress::zip::decompress(
	CStream						&inStream,
	size_t						inDataSize,
	void						*outData,
	size_t						outDataBufferSize,
	size_t						&outDataActualSize)
{
	int ret=0;
	MRPT_TRY_START;

	unsigned long	actualOutSize = (unsigned long)outDataBufferSize;
	std::vector<unsigned char>		inData;

	inData.resize(inDataSize);
	inStream.ReadBuffer( &inData[0], inDataSize );

	ret = ::uncompress(
		(unsigned char*)outData,
		&actualOutSize,
		&inData[0],
		(unsigned long)inDataSize
		);

	ASSERT_(ret==Z_OK);

	outDataActualSize = actualOutSize;

	MRPT_TRY_END_WITH_CLEAN_UP( \
		printf("[zlib] Error code=%i\n",ret);\
		);
}


/*---------------------------------------------------------------
					decompress_gz_file
---------------------------------------------------------------*/
bool mrpt::compress::zip::decompress_gz_file(const std::string &file_path, vector_byte & buffer)
{
	CFileGZInputStream	iss(file_path);
	if (!iss.fileOpenCorrectly())
		return false;

	buffer.clear();

	const int bytes2read = 1 << 20;
	int	act_read;
	size_t	total_bytes = 0;

	buffer.reserve( iss.getTotalBytesCount() );

	do
	{
		buffer.resize( 1000+bytes2read + buffer.size() );
		act_read = iss.ReadBuffer( &buffer[total_bytes], bytes2read );
		total_bytes += act_read;
	} while (act_read==bytes2read);

	buffer.resize( total_bytes );
	return true;
}

/*---------------------------------------------------------------
					compress_gz_file
---------------------------------------------------------------*/
bool mrpt::compress::zip::compress_gz_file(
	const std::string &file_path,
	const vector_byte &buffer,
	const int compress_level
	)
{
#if MRPT_HAS_GZ_STREAMS
	CFileGZOutputStream		oss;
	oss.open(file_path,compress_level);
	if (!oss.fileOpenCorrectly())
		return false;

	if (buffer.size())
	{
		try
		{
			oss.WriteBuffer(&buffer[0], buffer.size());
			return true;
		}
		catch(...) { return false; }
	}
	else
	{
		return true;
	}
#else
	THROW_EXCEPTION("MRPT has been compiled with MRPT_HAS_GZ_STREAMS=0")
#endif
}

