/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/config.h>

#include <mrpt/gui/CDisplayWindowPlots.h>

#include <mrpt/system/os.h>
#include <mrpt/utils/CMRPTImage.h>
#include <mrpt/utils/CMRPTImageFloat.h>

#include <mrpt/gui/WxSubsystem.h>
#include <mrpt/gui/WxUtils.h>

#include <mrpt/math/utils.h>

using namespace mrpt;
using namespace mrpt::gui;
using namespace mrpt::utils;
using namespace mrpt::system;
using namespace std;

IMPLEMENTS_SERIALIZABLE(CDisplayWindowPlots,CSerializable,mrpt::gui)


#if MRPT_HAS_WXWIDGETS

BEGIN_EVENT_TABLE(CWindowDialogPlots,wxFrame)

END_EVENT_TABLE()


const long CWindowDialogPlots::ID_PLOT = wxNewId();
const long ID_MENUITEM1 = wxNewId();
const long ID_MENUITEM2 = wxNewId();


CWindowDialogPlots::CWindowDialogPlots(
    CDisplayWindowPlots *winPlots,
    WxSubsystem::CWXMainFrame* parent,
    wxWindowID id,
    const std::string &caption,
    wxSize initialSize )
    :
    m_winPlots( winPlots ),
    m_mainFrame(parent)
{
	// Load my custom icons:
//    wxArtProvider::Push(new MyArtProvider);

    Create(
        parent,
        id,
        _U(caption.c_str()),
        wxDefaultPosition,
        initialSize,
        wxDEFAULT_FRAME_STYLE,
        _T("id"));
    SetClientSize(wxSize(672,539));

/*    {
    	wxIcon FrameIcon;
    	FrameIcon.CopyFromBitmap(wxArtProvider::GetBitmap(wxART_MAKE_ART_ID_FROM_STR(_T("MAIN_ICON")),wxART_FRAME_ICON));
    	SetIcon(FrameIcon);
    }
*/

    // Create the mpWindow object:
    m_plot = new mpWindow( this, ID_PLOT );
    m_plot->AddLayer( new mpScaleX() );
    m_plot->AddLayer( new mpScaleY() );
    m_plot->LockAspect( false );
    m_plot->EnableDoubleBuffer(true);

    m_plot->Fit( -10,10,-10,10 );

    //wxFlexGridSizer  *flex1 = new wxFlexGridSizer(1,1);
    //flex1->Add(m_plot,wxLEFT | wxTOP | wxEXPAND );
    //SetSizer(flex1);

    //m_canvas = new CMyGLCanvas( win2D, this, wxID_ANY, wxDefaultPosition, wxDefaultSize );


    // Menu:
    wxMenuBar *MenuBar1 = new wxMenuBar();

    wxMenu *Menu1 = new wxMenu();
    wxMenuItem *MenuItem1 = new wxMenuItem(Menu1, ID_MENUITEM1, _("Close"), _(""), wxITEM_NORMAL);
    Menu1->Append(MenuItem1);
    MenuBar1->Append(Menu1, _("&File"));

    wxMenu *Menu2 = new wxMenu();
    wxMenuItem *MenuItem2 = new wxMenuItem(Menu2, ID_MENUITEM2, _("About..."), _(""), wxITEM_NORMAL);
    Menu2->Append(MenuItem2);
    MenuBar1->Append(Menu2, _("&Help"));

    SetMenuBar(MenuBar1);


    // Events:
    Connect(wxID_ANY,wxEVT_CLOSE_WINDOW,(wxObjectEventFunction)&CWindowDialogPlots::OnClose);
    Connect(ID_MENUITEM1,wxEVT_COMMAND_MENU_SELECTED,(wxObjectEventFunction)&CWindowDialogPlots::OnMenuClose);
    Connect(ID_MENUITEM2,wxEVT_COMMAND_MENU_SELECTED,(wxObjectEventFunction)&CWindowDialogPlots::OnMenuAbout);

	Connect(wxID_ANY,wxEVT_KEY_DOWN,(wxObjectEventFunction)&CWindowDialogPlots::OnChar);
	m_plot->Connect(wxID_ANY,wxEVT_KEY_DOWN,(wxObjectEventFunction)&CWindowDialogPlots::OnChar);

    // Increment number of windows:
    //int winCount =
    WxSubsystem::CWXMainFrame::notifyWindowCreation();
    //cout << "[CWindowDialogPlots] Notifying new window: " << winCount << endl;

}

// Destructor
CWindowDialogPlots::~CWindowDialogPlots()
{
}

// OnClose event:
void CWindowDialogPlots::OnClose(wxCloseEvent& event)
{
    // Set the m_hwnd=NULL in our parent object.
    m_winPlots->notifyChildWindowDestruction();

    // Decrement number of windows:
    int winCount = WxSubsystem::CWXMainFrame::notifyWindowDestruction();
    //cout << "[~CWindowDialogPlots] Notifying window destruction: " << winCount << endl;

    // Close main frame if we were the last window. If new windows are created, a new main frame will be created again.
    if (!winCount)
    {
        //cout << "[~CWindowDialogPlots] Closing main frame." << endl;
        m_mainFrame->Close();
    }

	// Signal we are destroyed:
	//cout << "[~CWindowDialogPlots] Signaling semaphore meaning window is actually closed." << endl;
    m_winPlots->m_semWindowDestroyed.release();

    event.Skip();
}

void CWindowDialogPlots::OnChar(wxKeyEvent& event)
{
	if (m_winPlots)
	{
		m_winPlots->m_keyPushedCode = event.GetKeyCode();
		m_winPlots->m_keyPushed = true;
	}
}

// Menu: Close
void CWindowDialogPlots::OnMenuClose(wxCommandEvent& event)
{
    Close();
}
// Menu: About
void CWindowDialogPlots::OnMenuAbout(wxCommandEvent& event)
{
    ::wxMessageBox(_("Plot viewer\n Class gui::CDisplayWindowPlots\n MRPT C++ & wxMathPlot library"),_("About..."));
}

// Add / Modify a 2D plot using a MATLAB-like format string
void CWindowDialogPlots::plot(
	const vector_float &x,
	const vector_float &y,
	const std::string  &lineFormat,
	const std::string  &plotName)
{
	mpFXYVector  *theLayer;

	wxString    lyName = _U(plotName.c_str());
	bool        updateAtTheEnd = false; // If we update an existing layer, update manually to refresh the changes!

	// Already existing layer?
	mpLayer* existingLy = m_plot->GetLayerByName( lyName );

	if (existingLy)
	{
		// Assure the class:
		mpFXYVector  *lyPlot2D = static_cast<mpFXYVector*> ( existingLy );

		if (!lyPlot2D)
		{
			cerr << "[CWindowDialogPlots::plot] Plot name '" << plotName << "' is not of expected class mpFXYVector!."<< endl;
			return;
		}

		// Ok:
		theLayer = lyPlot2D;
		updateAtTheEnd = true;
	}
	else
	{
		// Create it:
		theLayer = new mpFXYVector( lyName );
		m_plot->AddLayer( theLayer );
	}

	// Set data:
	theLayer->SetData( x,y );

	// Line style:
	// -------------------
	bool isContinuous=true;
	int  lineColor[] = {0,0,255};
	int  lineWidth = 1;
	int  lineStyle = wxSOLID;

	// parse string:
	if ( string::npos != lineFormat.find(".") )		{ isContinuous=false; }
	if ( string::npos != lineFormat.find("-") )		{ isContinuous=true; lineStyle = wxSOLID; }
	if ( string::npos != lineFormat.find(":") )		{ isContinuous=true; lineStyle = wxLONG_DASH; }

	if ( string::npos != lineFormat.find("r") )		{ lineColor[0]=0xFF; lineColor[1]=0x00; lineColor[2]=0x00; }
	if ( string::npos != lineFormat.find("g") )		{ lineColor[0]=0x00; lineColor[1]=0xFF; lineColor[2]=0x00; }
	if ( string::npos != lineFormat.find("b") )		{ lineColor[0]=0x00; lineColor[1]=0x00; lineColor[2]=0xFF; }
	if ( string::npos != lineFormat.find("k") )		{ lineColor[0]=0x00; lineColor[1]=0x00; lineColor[2]=0x00; }
	if ( string::npos != lineFormat.find("m") )		{ lineColor[0]=192; lineColor[1]=0; lineColor[2]=192; }
	if ( string::npos != lineFormat.find("c") )		{ lineColor[0]=0; lineColor[1]=192; lineColor[2]=192; }

	if ( string::npos != lineFormat.find("1") )		{ lineWidth=1; }
	if ( string::npos != lineFormat.find("2") )		{ lineWidth=2; }
	if ( string::npos != lineFormat.find("3") )		{ lineWidth=3; }
	if ( string::npos != lineFormat.find("4") )		{ lineWidth=4; }
	if ( string::npos != lineFormat.find("5") )		{ lineWidth=5; }
	if ( string::npos != lineFormat.find("6") )		{ lineWidth=6; }
	if ( string::npos != lineFormat.find("7") )		{ lineWidth=7; }
	if ( string::npos != lineFormat.find("8") )		{ lineWidth=8; }
	if ( string::npos != lineFormat.find("9") )		{ lineWidth=9; }

	theLayer->SetContinuity(isContinuous);

	wxPen  pen( wxColour(lineColor[0],lineColor[1],lineColor[2]), lineWidth, lineStyle );
	theLayer->SetPen(pen);

	if (updateAtTheEnd)
		m_plot->Refresh();

}

// Add / Modify a 2D ellipse
// x[0,1]: Mean
// y[0,1,2]: Covariance matrix (0,0),(1,1),(0,1)
void CWindowDialogPlots::plotEllipse(
	const vector_float &x,
	const vector_float &y,
	const std::string  &lineFormat,
	const std::string  &plotName)
{
	mpCovarianceEllipse  *theLayer;

	if (x.size()!=3 || y.size()!=3)
	{
		cerr << "[CWindowDialogPlots::plotEllipse] vectors do not have expected size!!" << endl;
		return;
	}

	wxString    lyName = _U(plotName.c_str());
	bool        updateAtTheEnd = false; // If we update an existing layer, update manually to refresh the changes!

	// Already existing layer?
	mpLayer* existingLy = m_plot->GetLayerByName( lyName );

	if (existingLy)
	{
		// Assure the class:
		mpCovarianceEllipse  *lyPlotEllipse = static_cast<mpCovarianceEllipse*> ( existingLy );

		if (!lyPlotEllipse)
		{
			cerr << "[CWindowDialogPlots::plotEllipse] Plot name '" << plotName << "' is not of expected class mpCovarianceEllipse!."<< endl;
			return;
		}

		// Ok:
		theLayer = lyPlotEllipse;
		updateAtTheEnd = true;
	}
	else
	{
		// Create it:
		theLayer = new mpCovarianceEllipse( 1,1,0,2,32, lyName );
		m_plot->AddLayer( theLayer );
	}

	// Set data:
	theLayer->SetCovarianceMatrix(y[0],y[2],y[1]);
	theLayer->SetCoordinateBase(x[0],x[1]);
	theLayer->SetQuantiles(x[2]);
	theLayer->ShowName(false);

	// Line style:
	// -------------------
	bool isContinuous=true;
	int  lineColor[] = {0,0,255};
	int  lineWidth = 1;
	int  lineStyle = wxSOLID;

	// parse string:
	if ( string::npos != lineFormat.find(".") )		{ isContinuous=false; }
	if ( string::npos != lineFormat.find("-") )		{ isContinuous=true; lineStyle = wxSOLID; }
	if ( string::npos != lineFormat.find(":") )		{ isContinuous=true; lineStyle = wxLONG_DASH; }

	if ( string::npos != lineFormat.find("r") )		{ lineColor[0]=0xFF; lineColor[1]=0x00; lineColor[2]=0x00; }
	if ( string::npos != lineFormat.find("g") )		{ lineColor[0]=0x00; lineColor[1]=0xFF; lineColor[2]=0x00; }
	if ( string::npos != lineFormat.find("b") )		{ lineColor[0]=0x00; lineColor[1]=0x00; lineColor[2]=0xFF; }
	if ( string::npos != lineFormat.find("k") )		{ lineColor[0]=0x00; lineColor[1]=0x00; lineColor[2]=0x00; }
	if ( string::npos != lineFormat.find("m") )		{ lineColor[0]=192; lineColor[1]=0; lineColor[2]=192; }
	if ( string::npos != lineFormat.find("c") )		{ lineColor[0]=0; lineColor[1]=192; lineColor[2]=192; }

	if ( string::npos != lineFormat.find("1") )		{ lineWidth=1; }
	if ( string::npos != lineFormat.find("2") )		{ lineWidth=2; }
	if ( string::npos != lineFormat.find("3") )		{ lineWidth=3; }
	if ( string::npos != lineFormat.find("4") )		{ lineWidth=4; }
	if ( string::npos != lineFormat.find("5") )		{ lineWidth=5; }
	if ( string::npos != lineFormat.find("6") )		{ lineWidth=6; }
	if ( string::npos != lineFormat.find("7") )		{ lineWidth=7; }
	if ( string::npos != lineFormat.find("8") )		{ lineWidth=8; }
	if ( string::npos != lineFormat.find("9") )		{ lineWidth=9; }

	theLayer->SetContinuity(isContinuous);

	wxPen  pen( wxColour(lineColor[0],lineColor[1],lineColor[2]), lineWidth, lineStyle );
	theLayer->SetPen(pen);

	if (updateAtTheEnd)
		m_plot->Refresh();
}


void CWindowDialogPlots::image(
	void *theWxImage,
	const float &x0,
	const float &y0,
	const float &w,
	const float &h,
	const std::string &plotName)
{
	mpBitmapLayer	*theLayer;

	wxString    lyName = _U(plotName.c_str());
	bool        updateAtTheEnd = false; // If we update an existing layer, update manually to refresh the changes!

	// Already existing layer?
	mpLayer* existingLy = m_plot->GetLayerByName( lyName );

	if (existingLy)
	{
		// Assure the class:
		mpBitmapLayer  *ly = static_cast<mpBitmapLayer*> ( existingLy );

		if (!ly)
		{
			cerr << "[CWindowDialogPlots::image] Plot name '" << plotName << "' is not of expected class mpBitmapLayer!."<< endl;
			return;
		}

		// Ok:
		theLayer = ly;
		updateAtTheEnd = true;
	}
	else
	{
		// Create it:
		theLayer = new mpBitmapLayer();
		m_plot->AddLayer( theLayer );
	}

	// Set data:
	wxImage *ii = static_cast<wxImage *>(theWxImage);
	theLayer->SetBitmap( *ii, x0,y0,w,h );

	delete ii;theWxImage=NULL;

	if (updateAtTheEnd) m_plot->Refresh();
}

#endif

/*---------------------------------------------------------------
					Constructor
 ---------------------------------------------------------------*/
CDisplayWindowPlots::CDisplayWindowPlots(
	const std::string &windowCaption,
	unsigned int initialWidth,
	unsigned int initialHeight )
	: m_semThreadReady(0,1),
	  m_semWindowDestroyed(0,1),
	  m_caption(windowCaption)
{
	MRPT_TRY_START

    m_hwnd                      = 0;
    m_keyPushed                 = 0;

#if MRPT_HAS_WXWIDGETS
	// Create the main wxThread:
	// -------------------------------
	if (!WxSubsystem::createOneInstanceMainThread() )
        return; // Error!

    // Create window:
    WxSubsystem::TRequestToWxMainThread  *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->str      = m_caption;
    REQ->OPCODE   = 400;
	REQ->voidPtr  = m_hwnd.getPtrToPtr();
    WxSubsystem::pushPendingWxRequest( REQ );

    // Wait for the window to realize and signal it's alive:
    if (!WxSubsystem::isConsoleApp)
    {
    	mrpt::system::sleep(20);	// Force at least 1-2 timer ticks for processing the event:
    	wxApp::GetInstance()->Yield(true);
    }
	if(!m_semThreadReady.waitForSignal(6000))  // 2 secs should be enough...
	{
	    cerr << "[CDisplayWindowPlots::CDisplayWindowPlots] Timeout waiting window creation." << endl;
	}

	// Set window size:
	resize(initialWidth,initialHeight);

#else
	THROW_EXCEPTION("MRPT compiled without wxWidgets!")
#endif

	MRPT_TRY_END
}

/*---------------------------------------------------------------
					Destructor
 ---------------------------------------------------------------*/
CDisplayWindowPlots::~CDisplayWindowPlots( )
{
#if MRPT_HAS_WXWIDGETS
    // Send close request:
    WxSubsystem::TRequestToWxMainThread  *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 499;
    WxSubsystem::pushPendingWxRequest( REQ );

	// Wait until the thread ends:
    if (!WxSubsystem::isConsoleApp)
    {
    	mrpt::system::sleep(20);	// Force at least 1-2 timer ticks for processing the event:
    	wxApp::GetInstance()->Yield(true);
    }
	if(!m_semWindowDestroyed.waitForSignal(6000))  // 2 secs should be enough...
	{
	    cerr << "[CDisplayWindowPlots::~CDisplayWindowPlots] Timeout waiting window destruction." << endl;
	}

	WxSubsystem::waitWxShutdownsIfNoWindows();
#endif
}


/*---------------------------------------------------------------
					isOpen
 ---------------------------------------------------------------*/
bool CDisplayWindowPlots::isOpen()
{
    return m_hwnd!=NULL;
}

/*---------------------------------------------------------------
					waitForKey
 ---------------------------------------------------------------*/
int CDisplayWindowPlots::waitForKey()
{
	int k = 0;

    m_keyPushed = false;
	m_keyPushedCode = 0;
	while (!m_keyPushed && !os::kbhit())
	{
		system::sleep(10);
		// Are we still alive?
		if (!isOpen()) break;
	}
	if (os::kbhit())	k=os::getch();
	if (m_keyPushed)	k=m_keyPushedCode;
	return k;
}


/*---------------------------------------------------------------
					resize
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::resize(
	unsigned int width,
	unsigned int height )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen())
	{
		cerr << "[CDisplayWindowPlots::resize] Window closed!: " << m_caption << endl;
		return;
	}

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 403;
    REQ->x        = width;
    REQ->y        = height;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}

/*---------------------------------------------------------------
					setPos
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::setPos( int x, int y )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen())
	{
		cerr << "[CDisplayWindowPlots::setPos] Window closed!: " << m_caption << endl;
		return;
	}

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 402;
    REQ->x        = x;
    REQ->y        = y;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}

/*---------------------------------------------------------------
					setWindowTitle
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::setWindowTitle( const std::string &str )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen())
	{
		cerr << "[CDisplayWindowPlots::setWindowTitle] Window closed!: " << m_caption << endl;
		return;
	}

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 404;
    REQ->str      = str;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}

/*---------------------------------------------------------------
					notifyChildWindowDestruction
 ---------------------------------------------------------------*/
void CDisplayWindowPlots::notifyChildWindowDestruction()
{
    //cout << "[CDisplayWindowPlots::notifyChildWindowDestruction] Called." << endl;
    m_hwnd = NULL;
}


/*---------------------------------------------------------------
					enableMousePanZoom
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::enableMousePanZoom( bool enabled )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen()) return;

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 410;
    REQ->boolVal  = enabled;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}

/*---------------------------------------------------------------
					axis_equal
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::axis_equal( bool enabled )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen()) return;

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 411;
    REQ->boolVal  = enabled;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}


/*---------------------------------------------------------------
					plot
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::plot(
	const vector_float &x,
	const vector_float &y,
	const std::string  &lineFormat,
	const std::string  &plotName )
{
	MRPT_TRY_START
#if MRPT_HAS_WXWIDGETS
	if (!isOpen()) return;

	ASSERT_(x.size() == y.size());

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 420;
    REQ->str      = lineFormat;
    REQ->plotName = plotName;
    REQ->vector_x = x;
    REQ->vector_y = y;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
	MRPT_TRY_END
}

/*---------------------------------------------------------------
					plot (double)
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::plot(
	const vector_double &x,
	const vector_double &y,
	const std::string  &lineFormat,
	const std::string  &plotName )
{
	MRPT_TRY_START

	ASSERT_(x.size() == y.size());

	vector_float  x2(x.size()),y2(y.size());
	for (size_t i=0;i<x.size();i++)
	{
		x2[i]=x[i];
		y2[i]=y[i];
	}

	plot(x2,y2,lineFormat,plotName);

	MRPT_TRY_END
}


/*---------------------------------------------------------------
					axis
 ---------------------------------------------------------------*/
void CDisplayWindowPlots::axis( float x_min, float x_max, float y_min, float y_max, bool aspectRatioFix )
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen()) return;

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 412;
	REQ->vector_x.resize(2);
	REQ->vector_x[0] = x_min;
	REQ->vector_x[1] = x_max;
	REQ->vector_y.resize(2);
	REQ->vector_y[0] = y_min;
	REQ->vector_y[1] = y_max;
	REQ->boolVal  = aspectRatioFix;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}

/*---------------------------------------------------------------
					axis_fit
 ---------------------------------------------------------------*/
void CDisplayWindowPlots::axis_fit(bool aspectRatioFix)
{
#if MRPT_HAS_WXWIDGETS
	if (!isOpen()) return;

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 413;
	REQ->boolVal  = aspectRatioFix;
    WxSubsystem::pushPendingWxRequest( REQ );
#endif
}



/*---------------------------------------------------------------
					plotEllipse
 ---------------------------------------------------------------*/
void CDisplayWindowPlots::plotEllipse(
	const float &mean_x,
	const float &mean_y,
	const CMatrixTemplateNumeric<float> &cov22,
	const float &quantiles,
	const std::string  &lineFormat,
	const std::string  &plotName )
{
	MRPT_TRY_START
#if MRPT_HAS_WXWIDGETS
	if (!isOpen()) return;

	ASSERT_(cov22.getColCount()==2 && cov22.getRowCount()==2);
	ASSERT_(cov22(0,0)>=0);
	ASSERT_(cov22(1,1)>=0);
	ASSERT_(cov22(0,1) == cov22(1,0) );

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 421;
    // 421: Add/update a 2D ellipse: format string=str, plot name =plotName, vector_x[0,1]:X/Y center,  vector_y[0,1,2]: Covariance matrix entries 00,11,01.
    REQ->str      = lineFormat;
    REQ->plotName = plotName;

    REQ->vector_x.resize(3);
    REQ->vector_x[0]=mean_x;
    REQ->vector_x[1]=mean_y;
    REQ->vector_x[2]=quantiles;

    REQ->vector_y.resize(3);
    REQ->vector_y[0] = cov22(0,0);
    REQ->vector_y[1] = cov22(1,1);
    REQ->vector_y[2] = cov22(0,1);

    WxSubsystem::pushPendingWxRequest( REQ );
#endif
	MRPT_TRY_END
}


/*---------------------------------------------------------------
					image
 ---------------------------------------------------------------*/
void CDisplayWindowPlots::image(
	const utils::CMRPTImage &img,
	const float &x_left,
	const float &y_bottom,
	const float &x_width,
	const float &y_height,
	const std::string  &plotName  )
{
	MRPT_TRY_START
#if MRPT_HAS_WXWIDGETS
	if (!isOpen()) return;

    // Send a request to destroy this object:
    WxSubsystem::TRequestToWxMainThread *REQ = new WxSubsystem::TRequestToWxMainThread[1];
    REQ->sourcePlots = this;
    REQ->OPCODE   = 422;

    // 422: Add/update a bitmap: plot name =plotName, vector_x[0,1]:X/Y corner, vector_x[2,3]: X/Y widths, voidPtr2: pointer to a newly created wxImage with the bitmap.
    REQ->plotName = plotName;

    REQ->vector_x.resize(4);
    REQ->vector_x[0]=x_left;
    REQ->vector_x[1]=y_bottom;
    REQ->vector_x[2]=x_width;
    REQ->vector_x[3]=y_height;

	REQ->voidPtr2 = mrpt::gui::MRPTImage2wxImage(img);

    WxSubsystem::pushPendingWxRequest( REQ );
#endif
	MRPT_TRY_END
}

/*---------------------------------------------------------------
					writeToStream
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::writeToStream(mrpt::utils::CStream &out, int *getVersion) const
{
	THROW_EXCEPTION("Class not serializable");
}

/*---------------------------------------------------------------
					readFromStream
 ---------------------------------------------------------------*/
void  CDisplayWindowPlots::readFromStream(mrpt::utils::CStream &in, int version)
{
	THROW_EXCEPTION("Class not serializable");
}



/*---------------------------------------------------------------
					plot
 ---------------------------------------------------------------*/
void CDisplayWindowPlots::plot(
	const vector_float &y,
	const std::string  &lineFormat,
	const std::string  &plotName  )
{
	vector_float x;
 	mrpt::math::linspace(1.0f,(float)y.size(),y.size(),x);
	plot(x,y,lineFormat,plotName);
}

/*---------------------------------------------------------------
					plot
 ---------------------------------------------------------------*/
void CDisplayWindowPlots::plot(
	const vector_double &y,
	const std::string  &lineFormat,
	const std::string  &plotName  )
{
	vector_double x;
	mrpt::math::linspace(1.0,(double)y.size(),y.size(),x);
	plot(x,y,lineFormat,plotName);
}

