/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/math/CLevenbergMarquardt.h>
#include <mrpt/math/utils.h>
#include <mrpt/math/vector_ops.h>
#include <mrpt/math/CMatrix.h>
#include <mrpt/math/CMatrixD.h>

using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;


/*---------------------------------------------------------------
						execute
 ---------------------------------------------------------------*/
template <typename NUMTYPE>
void CLevenbergMarquardtTempl<NUMTYPE>::execute(
	std::vector<NUMTYPE>		&x,
	const std::vector<NUMTYPE>	&x0,
	typename CLevenbergMarquardtTempl<NUMTYPE>::TFunctor functor,
	const std::vector<NUMTYPE>	&increments,
	const std::vector<NUMTYPE>	&userParam,
	TResultInfo					&out_info,
	bool						VERBOSE,
	const size_t				&maxIter,
	const NUMTYPE				tau,
	const NUMTYPE				e1,
	const NUMTYPE				e2
	)
{
	MRPT_TRY_START;

	// Asserts:
	ASSERT_( increments.size() == x0.size() );

	x=x0;									// Start with the starting point
	vector<NUMTYPE>	f_x;					// The vector error from the user function
	CMatrixTemplateNumeric<NUMTYPE> J;						// The Jacobian of "f"
	CMatrixTemplateNumeric<NUMTYPE>	H;						// The Hessian of "f"
	vector<NUMTYPE>	g;						// The gradient

	// Compute the jacobian and the Hessian:
	math::estimateJacobian( x, functor, increments, userParam, J);
	H.multiply_AtA(J);

	const size_t  H_len = H.getColCount();

//	cout << "J:" << endl << J << endl;
//	cout << "H:" << endl << H << endl;

	// Compute the gradient:
	functor(x, userParam ,f_x);
	J.multiply_Atb(f_x, g);

//	cout << "f_x:" << f_x << endl;
//	cout << "g:  " << g << endl;

	// Start iterations:
	bool	found = math::norm_inf(g)<=e1;
	if (VERBOSE && found)	cout << "[LM] End condition: math::norm_inf(g)<=e1 :" << math::norm_inf(g) << endl;

	NUMTYPE	lambda = tau * H.maximumDiagonal();
	size_t  iter = 0;
	NUMTYPE	v = 2;

	CMatrixTemplateNumeric<NUMTYPE>	AUX;
	vector<NUMTYPE>	h_lm;
	vector<NUMTYPE>	xnew, f_xnew ;
	NUMTYPE			F_x  = pow( math::norm( f_x ), 2);

	const size_t	N = x.size();

	out_info.path.setSize(maxIter,N+1);
	out_info.path.insertRow(iter,x);

	while (!found && ++iter<maxIter)
	{
		// H_lm = -( H + \lambda I ) ^-1 * g
		for (size_t k=0;k<H_len;k++)
			H(k,k)*= 1+lambda;

		H.inv_fast(AUX);
		AUX.multiply_Ab(g,h_lm);
		h_lm *= NUMTYPE(-1.0);

		double h_lm_n2 = math::norm(h_lm);
		double x_n2 = math::norm(x);

		if (VERBOSE)
			cout << "[LM] Iter: " << iter << " x:" << x << endl;

		if (h_lm_n2<e2*(x_n2+e2))
		{
			// Done:
			found = true;
			if (VERBOSE)
			{
				cout.precision(10);
				cout << "[LM] End condition: " << scientific << h_lm_n2 << " < " << e2*(x_n2+e2) << endl;
			}
		}
		else
		{
			// Improvement:
			xnew = x;
			xnew += h_lm;
			functor(xnew, userParam ,f_xnew );
			const double F_xnew = pow( math::norm(f_xnew), 2);

			// denom = h_lm^t * ( \lambda * h_lm - g )
			vector<NUMTYPE>	tmp(h_lm);
			tmp *= lambda;
			tmp -= g;
			tmp *= h_lm;
			double denom = math::sum(tmp);
			double l = (F_x - F_xnew) / denom;

			//cout << "l:" << l << endl << tmp;

			if (l>0) // There is an improvement:
			{
				// Accept new point:
				x   = xnew;
				f_x = f_xnew;
				F_x = F_xnew;

				math::estimateJacobian( x, functor, increments, userParam, J);
				H.multiply_AtA(J);
				J.multiply_Atb(f_x, g);

				found = math::norm_inf(g)<=e1;
				if (VERBOSE && found)	cout << "[LM] End condition: math::norm_inf(g)<=e1 :" << math::norm_inf(g) << endl;

				lambda *= max(0.33, 1-pow(2*l-1,3) );
				v = 2;
			}
			else
			{
				// Nope...
				lambda *= v;
				v*= 2;
			}

			out_info.path.insertRow(iter,x);
			out_info.path(iter,x.size()) = F_x;
		}

	} // end while

	// Output info:
	out_info.final_sqr_err = F_x;
	out_info.iterations_executed = iter;
	out_info.path.setSize(iter,N+1);

	MRPT_TRY_END;
}




// Template instantiation:
template class MRPTDLLIMPEXP CLevenbergMarquardtTempl<double>;
template class MRPTDLLIMPEXP CLevenbergMarquardtTempl<float>;

#ifdef HAVE_LONG_DOUBLE
template class MRPTDLLIMPEXP CLevenbergMarquardtTempl<long double>;
#endif
