/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/math/CMatrixD.h>

#include <mrpt/poses/CPoint2D.h>
#include <mrpt/poses/CPoint3D.h>
#include <mrpt/poses/CPose2D.h>
#include <mrpt/poses/CPose3D.h>

using namespace mrpt;
using namespace mrpt::math;
using namespace mrpt::utils;

// This must be added to any CSerializable class implementation file.
IMPLEMENTS_SERIALIZABLE(CMatrixD, CSerializable, mrpt::math)



/*---------------------------------------------------------------
						writeToStream
 ---------------------------------------------------------------*/
void  CMatrixD::writeToStream(CStream &out, int *out_Version) const
{
	if (out_Version)
		*out_Version = 0;
	else
	{
		// First, write the number of rows and columns:
		out << (uint32_t)m_Rows << (uint32_t)m_Cols;

		if (m_Rows>0 && m_Cols>0)
			for (unsigned int i=0;i<m_Rows;i++)
				out.WriteBuffer(m_Val[i],sizeof(m_Val[0][0])*m_Cols);
	}

}

/*---------------------------------------------------------------
						readFromStream
 ---------------------------------------------------------------*/
void  CMatrixD::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
		{
			uint32_t nRows,nCols;

			// First, write the number of rows and columns:
			in >> nRows >> nCols;

			setSize(nRows,nCols);

			if (nRows>0 && nCols>0)
				for (unsigned int i=0;i<nRows;i++)
					in.ReadBuffer(m_Val[i],sizeof(m_Val[0][0])*m_Cols);
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/*---------------------------------------------------------------
					= for CPose2D
---------------------------------------------------------------*/
CMatrixD::CMatrixD(const mrpt::poses::CPose2D &p) : CMatrixDouble(3,1)
{
	(*this)(0,0)=p.x;
	(*this)(1,0)=p.y;
	(*this)(2,0)=p.phi;
}

/*---------------------------------------------------------------
					= for CPose3D
---------------------------------------------------------------*/
CMatrixD::CMatrixD(const mrpt::poses::CPose3D &p) : CMatrixDouble(6,1)
{
	(*this)(0,0)=p.x;
	(*this)(1,0)=p.y;
	(*this)(2,0)=p.z;
	(*this)(3,0)=p.yaw;
	(*this)(4,0)=p.pitch;
	(*this)(5,0)=p.roll;
}

/*---------------------------------------------------------------
					= for CPoint2D
---------------------------------------------------------------*/
CMatrixD::CMatrixD(const mrpt::poses::CPoint2D &p) : CMatrixDouble(2,1)
{
	(*this)(0,0)=p.x;
	(*this)(1,0)=p.y;
}

/*---------------------------------------------------------------
					= for CPoint3D
---------------------------------------------------------------*/
CMatrixD::CMatrixD(const mrpt::poses::CPoint3D &p) : CMatrixDouble(3,1)
{
	(*this)(0,0)=p.x;
	(*this)(1,0)=p.y;
	(*this)(2,0)=p.z;
}
