/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/opengl/CRenderizable.h>		// Include these before windows.h!!
#include <mrpt/synch.h>
#include <mrpt/utils/CStringList.h>
#include <mrpt/math/utils.h>
#include <mrpt/poses/CPose3D.h>

#include "opengl_internals.h"


using namespace std;
using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::synch;

IMPLEMENTS_VIRTUAL_SERIALIZABLE( CRenderizable, CSerializable, mrpt::opengl )



// This is not the ideal solution, since in theory two threads
// could create 2 different cs objects. But we cannot declare
// this object as static & global since we are not sure about
// the destroy order of all global objects!  (JLBC OCT-2008)
struct TTextureNamesInfo
{
	TTextureNamesInfo() :
		freeTextureNames(1000,false),
		csTextureNames(),
		texturesInUseCount(0)
	{
	}

	std::vector<bool>		freeTextureNames;
	synch::CCriticalSection	csTextureNames;
	int						texturesInUseCount;
};
TTextureNamesInfo	*textureNamesInfo = NULL;


/** Returns the lowest, free texture name.
  */
unsigned int CRenderizable::getNewTextureNumber()
{
	MRPT_TRY_START;

	if (!textureNamesInfo)
		textureNamesInfo = new TTextureNamesInfo[1];

	CCriticalSectionLocker lock ( &textureNamesInfo->csTextureNames );
	for (unsigned int i = 1;i<textureNamesInfo->freeTextureNames.size();i++)
	{
		if (!textureNamesInfo->freeTextureNames[i])
		{
			textureNamesInfo->freeTextureNames[i] = true;
			textureNamesInfo->texturesInUseCount++;
			return i;
		}
	}
	THROW_EXCEPTION("Maximum number of textures (1000) excedeed! (are you deleting them?)");
	MRPT_TRY_END;
}

void CRenderizable::releaseTextureName(unsigned int i)
{
#if MRPT_HAS_OPENGL_GLUT
	MRPT_TRY_START;
	// DO NOT use CCriticalSectionLocker here
	textureNamesInfo->csTextureNames.enter();

	glDeleteTextures( 1, &i );
	textureNamesInfo->freeTextureNames[i] = false;
	textureNamesInfo->texturesInUseCount--;

	if (! textureNamesInfo->texturesInUseCount )
	{
		textureNamesInfo->csTextureNames.leave();

		// This was the last texture:
		delete[] textureNamesInfo;
		textureNamesInfo = NULL;
	}
	else
		textureNamesInfo->csTextureNames.leave();

	MRPT_TRY_END;
#endif
}


void  CRenderizable::writeToStreamRender(CStream &out) const
{
	out << m_name << (float)m_color_R << (float)m_color_G << (float)m_color_B << (float)m_color_A;
	out << (float)m_x << (float)m_y << (float)m_z;

	// Version 2: Add scale vars:
	if (m_scale_x==1.0f && m_scale_y==1.0f && m_scale_z==1.0f)
	{
		// Keep old format for compatibility:
		out << (float)m_yaw << (float)m_pitch << (float)m_roll;
		out << m_show_name;
	}
	else
	{
		const float dummy = 16.0f;
		out << (float)m_yaw << (float)m_pitch << dummy << (float)m_roll;
		out << m_show_name;
		out << m_scale_x << m_scale_y << m_scale_z;
	}
}

void  CRenderizable::readFromStreamRender(CStream &in)
{
	in >> m_name;
	float f;

	in >> f; m_color_R=f;
	in >> f; m_color_G=f;
	in >> f; m_color_B=f;
	in >> f; m_color_A=f;
	in >> f; m_x=f;
	in >> f; m_y=f;
	in >> f; m_z=f;
	in >> f; m_yaw=f;
	in >> f; m_pitch=f;
	in >> f;
	// Version 2: Add scale vars:
	//  JL: Yes, this is a crappy hack since I forgot to enable versions here...what? :-P
	if (f!=16.0f)
	{
		// Old version:
		m_roll=f;
		in >> m_show_name;
		m_scale_x=m_scale_y=m_scale_z=1;	// Default values
	}
	else
	{
		// New version:
		in >> f; m_roll=f;
		in >> m_show_name;

		// Scale data:
		in >> m_scale_x >> m_scale_y >> m_scale_z;
	}
}


void  CRenderizable::checkOpenGLError()
{
#if MRPT_HAS_OPENGL_GLUT
	int	 openglErr;
	if ( ( openglErr= glGetError()) != GL_NO_ERROR )
	{
		THROW_EXCEPTION( std::string("OpenGL error: ") + std::string( (char*)gluErrorString(openglErr) ) );
	}
#endif
}

/*--------------------------------------------------------------
					setPose
  ---------------------------------------------------------------*/
void CRenderizable::setPose( const mrpt::poses::CPose3D &o )
{
	m_x = o.x;
	m_y = o.y;
	m_z = o.z;
	m_yaw = RAD2DEG( o.yaw );
	m_pitch = RAD2DEG( o.pitch );
	m_roll = RAD2DEG( o.roll );
}


/*--------------------------------------------------------------
					traceRay
  ---------------------------------------------------------------*/
bool CRenderizable::traceRay(const mrpt::poses::CPose3D &o,float &dist) const	{
	return false;
}
/*--------------------------------------------------------------
					setColor
  ---------------------------------------------------------------*/
void CRenderizable::setColor( double R, double G, double B, double A)
{
	m_color_R = R;
	m_color_G = G;
	m_color_B = B;
	m_color_A = A;
}
