/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Vicente Arevalo  <varevalo@ctima.uma.es>                      |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.

#include <mrpt/opengl/CSetOfTexturedTriangles.h>
#include <mrpt/math/utils.h>

#include "opengl_internals.h"

using namespace mrpt;
using namespace mrpt::opengl;
using namespace mrpt::utils;
using namespace mrpt::math;
using namespace std;

IMPLEMENTS_SERIALIZABLE( CSetOfTexturedTriangles, CRenderizable, mrpt::opengl )

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CSetOfTexturedTriangles::assignImage(
	const CMRPTImage& img,
	const CMRPTImage& imgAlpha )
{
	MRPT_TRY_START;

	// Make a copy:
	m_textureImage = img;
	m_textureImageAlpha = imgAlpha;

	m_enableTransparency = true;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CSetOfTexturedTriangles::assignImage(
	const CMRPTImage& img )
{
	MRPT_TRY_START;

	// Make a copy:
	m_textureImage = img;

	m_enableTransparency = false;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CSetOfTexturedTriangles::assignImage_fast(
	CMRPTImage& img,
	CMRPTImage& imgAlpha )
{
	MRPT_TRY_START;

	// Make a copy:
	m_textureImage.copyFastFrom(img);
	m_textureImageAlpha.copyFastFrom(imgAlpha);

	m_enableTransparency = true;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
							assignImage
  ---------------------------------------------------------------*/
void  CSetOfTexturedTriangles::assignImage_fast(
	CMRPTImage& img )
{
	MRPT_TRY_START;

	// Make a copy:
	m_textureImage.copyFastFrom(img);

	m_enableTransparency = false;

	MRPT_TRY_END;
}

/*---------------------------------------------------------------
							loadTextureInOpenGL
  ---------------------------------------------------------------*/
void  CSetOfTexturedTriangles::loadTextureInOpenGL() const
{
#if MRPT_HAS_OPENGL_GLUT

	vector<unsigned char>	data;
	unsigned char*			dataAligned = NULL;

	MRPT_TRY_START;

	// previously called in render()
//	glEnable(GL_TEXTURE_2D);	// enable 2D texturing
//	checkOpenGLError();

	if (m_init)
	{
		glBindTexture( GL_TEXTURE_2D, m_glTextureName );
		checkOpenGLError();
	}
	else
	{
		// allocate a texture name
		m_glTextureName = getNewTextureNumber();
		checkOpenGLError();

		// select our current texture
		glBindTexture( GL_TEXTURE_2D, m_glTextureName );
		checkOpenGLError();

		// select modulate to mix texture with color for shading
		glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );

		// when texture area is small, bilinear filter the closest mipmap
		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
		checkOpenGLError();

		// when texture area is large, bilinear filter the first mipmap
		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
	    checkOpenGLError();

		// if wrap is true, the texture wraps over at the edges (repeat)
		//       ... false, the texture ends at the edges (clamp)
		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
		checkOpenGLError();

		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
		checkOpenGLError();

		// Assure that the images do not overpass the maximum dimensions allowed by OpenGL:
		// ------------------------------------------------------------------------------------
		GLint	texSize;
		glGetIntegerv(GL_MAX_TEXTURE_SIZE, &texSize);
		while( m_textureImage.getHeight() > (unsigned int)texSize || m_textureImage.getWidth() > (unsigned int)texSize )
		{
			m_textureImage		= m_textureImage.scaleHalf();
			m_textureImageAlpha = m_textureImageAlpha.scaleHalf();
		}

		int width  = m_textureImage.getWidth();
		int height = m_textureImage.getHeight();

		r_width  = round2up( width );
		r_height = round2up( height );

        if (m_enableTransparency)
		{
			ASSERT_( !m_textureImageAlpha.isColor() );
			ASSERT_( m_textureImageAlpha.getWidth() == m_textureImage.getWidth() );
			ASSERT_( m_textureImageAlpha.getHeight() == m_textureImage.getHeight() );

			// Color texture:
			if (m_textureImage.isColor())
			{
				data.clear();
				data.resize(r_height * r_width * 4);
				dataAligned = (unsigned char*)((POINTER_TYPE)&data[0]);

				for (int y = 0; y < height; y++)
				{
					unsigned char* ptrSrcCol  = m_textureImage     (0,y,2);
					unsigned char* ptrSrcAlfa = m_textureImageAlpha(0,y,0);
					unsigned char* ptr = dataAligned + y * r_width * 4;
					for (int x = 0; x < width; x++)
					{
						*ptr++ = *ptrSrcCol--;	// channel order change BGR->RGB
						*ptr++ = *ptrSrcCol--;
						*ptr++ = *ptrSrcCol--;
						*ptr++ = *ptrSrcAlfa++;
						ptrSrcCol += 6;
					}
				}

				// build our texture mipmaps
				gluBuild2DMipmaps( GL_TEXTURE_2D, 4, r_width, r_height, GL_RGBA, GL_UNSIGNED_BYTE, dataAligned );
			}
			else
			{
				data.clear();
				data.resize(r_height * r_width * 2);
                dataAligned = (unsigned char*)((POINTER_TYPE)&data[0]);

				for (int y = 0; y < height; y++)
				{
					unsigned char* ptrSrcCol  = m_textureImage     (0,y,0);
					unsigned char* ptrSrcAlfa = m_textureImageAlpha(0,y,0);
					unsigned char* ptr = dataAligned + y * r_width * 2;
					for (int x = 0; x < width; x++)
					{
    					*ptr++ = *ptrSrcCol++;
    					*ptr++ = *ptrSrcAlfa++;
					}
				}

				// build our texture mipmaps
				gluBuild2DMipmaps( GL_TEXTURE_2D, 2, r_width, r_height, GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, dataAligned );
			}// End of gray-scale texture WITH trans.
		}
		else
		{
			if (m_textureImage.isColor())
			{
				data.clear();
				data.resize(r_height * r_width * 3);
                dataAligned = (unsigned char*)((POINTER_TYPE)&data[0]);

				for (int y = 0; y < height; y++)
				{
					unsigned char* ptrSrcCol = m_textureImage(0,y,2);
					unsigned char* ptr = dataAligned + y * r_width * 3;
					for (int x = 0; x < width; x++)
					{
    					*ptr++ = *ptrSrcCol--;	// channel order change BGR->RGB
    					*ptr++ = *ptrSrcCol--;
    					*ptr++ = *ptrSrcCol--;
						ptrSrcCol += 6;
					}
				}

				// build our texture mipmaps
				gluBuild2DMipmaps( GL_TEXTURE_2D, 3, r_width, r_height, GL_RGB, GL_UNSIGNED_BYTE, dataAligned );
			}
			else
			{
				data.clear();
				data.resize(r_height * r_width);
                dataAligned = (unsigned char*)((POINTER_TYPE)&data[0]);

				for (int y = 0; y < height; y++)
				{
					unsigned char* ptrSrcCol = m_textureImage(0, y, 0);
					unsigned char* ptr = dataAligned + y * r_width * 1;
					for (int x = 0; x < width; x++)
					{
    					*ptr++ = *ptrSrcCol++;
					}
				}

				// build our texture mipmaps
				gluBuild2DMipmaps( GL_TEXTURE_2D, 1, r_width, r_height, GL_LUMINANCE, GL_UNSIGNED_BYTE, dataAligned );
			}// End of gray-scale texture WITHOUT trans.
		}
		checkOpenGLError();

		m_init = true;
	}

	MRPT_TRY_END;

#endif
}

/*---------------------------------------------------------------
							~CTexturedPlane
  ---------------------------------------------------------------*/
CSetOfTexturedTriangles::~CSetOfTexturedTriangles()
{
    if (m_init)
    {
    	m_init = false;
		releaseTextureName( m_glTextureName );
	}
}

/*---------------------------------------------------------------
							render
  ---------------------------------------------------------------*/
void   CSetOfTexturedTriangles::render() const
{
#if MRPT_HAS_OPENGL_GLUT

	MRPT_TRY_START;

	// setup texture mapping
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, m_glTextureName);
	checkOpenGLError();

	if (m_enableTransparency)
	{
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		checkOpenGLError();
	}

	// This will load and/or select our texture, only for the first time.
	loadTextureInOpenGL();

	glShadeModel(GL_SMOOTH);

	glBegin(GL_TRIANGLES);

	float ax, ay, az, bx, by, bz;

	vector<TTriangle>::const_iterator	it;
	for (it = m_triangles.begin(); it != m_triangles.end(); it++)
	{
		// Compute the normal vector:
		// ---------------------------------
		ax = it->m_v2.m_x - it->m_v1.m_x;
		ay = it->m_v2.m_y - it->m_v1.m_y;
		az = it->m_v2.m_z - it->m_v1.m_z;

		bx = it->m_v3.m_x - it->m_v1.m_x;
		by = it->m_v3.m_y - it->m_v1.m_y;
		bz = it->m_v3.m_z - it->m_v1.m_z;

		glNormal3f(ay * bz - az * by, -ax * bz + az * bx, ax * by - ay * bx);

		glTexCoord2d(float(it->m_v1.m_tx)/r_width, float(it->m_v1.m_ty)/r_height); glVertex3f(it->m_v1.m_x, it->m_v1.m_y, it->m_v1.m_z);
		glTexCoord2d(float(it->m_v2.m_tx)/r_width, float(it->m_v2.m_ty)/r_height); glVertex3f(it->m_v2.m_x, it->m_v2.m_y, it->m_v2.m_z);
		glTexCoord2d(float(it->m_v3.m_tx)/r_width, float(it->m_v3.m_ty)/r_height); glVertex3f(it->m_v3.m_x, it->m_v3.m_y, it->m_v3.m_z);
	}

	glEnd();

	if (m_enableTransparency)
	{
		glDisable(GL_BLEND);
	}

	MRPT_TRY_END;

#endif
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CSetOfTexturedTriangles::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 1;
	else
	{
		uint32_t n;

		writeToStreamRender(out);

		out << m_textureImage;
		out << m_enableTransparency;
		if (m_enableTransparency)
		{
			out << m_textureImageAlpha;
		}

		n = (uint32_t)m_triangles.size();

		out << n;
		if (n)
		{
			out.WriteBuffer( &m_triangles[0], n*sizeof(TTriangle) );
		}
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CSetOfTexturedTriangles::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
	case 1:
		{
			uint32_t n;

			readFromStreamRender(in);

			in >> n;
			m_triangles.resize(n);
			if (n)
			{
				in.ReadBuffer( &m_triangles[0], n*sizeof(TTriangle) );
			}

			in >> m_enableTransparency;
			if (m_enableTransparency)
			{
				in >> m_textureImage;
				in >> m_textureImageAlpha;
				assignImage( m_textureImage, m_textureImageAlpha );
			}
			else
			{
				in >> m_textureImage;
				assignImage( m_textureImage );
			}
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)
	};
}

bool CSetOfTexturedTriangles::traceRay(const mrpt::poses::CPose3D &o,float &dist) const	{
	//TODO
	return false;
}
