/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 


#include <mrpt/poses/CPoint2D.h>
#include <mrpt/poses/CPose2D.h>
#include <mrpt/poses/CPose3D.h>

using namespace mrpt::poses;
using namespace mrpt::utils;
using namespace mrpt::math;

IMPLEMENTS_SERIALIZABLE(CPoint2D, CPoint,mrpt::poses)


/*---------------------------------------------------------------
	Constructors
  ---------------------------------------------------------------*/
CPoint2D::CPoint2D(double x_,double y_) :
	aux_HM(0,0)
{
	this->x		= x_;
	this->y		= y_;
	this->z		= 0;
	is3D = false;
}


/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
CPoint2D::CPoint2D( const CPose2D &o ) :
	aux_HM(0,0)
{
	this->x		= o.x;
	this->y		= o.y;
	this->z		= 0;
	is3D = false;
}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
CPoint2D::CPoint2D( const CPose3D &o ) :
	aux_HM(0,0)
{
	this->x		= o.x;
	this->y		= o.y;
	this->z		= 0;
	is3D = false;

}

/*---------------------------------------------------------------
   Implements the writing to a CStream capability of
     CSerializable objects
  ---------------------------------------------------------------*/
void  CPoint2D::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 1;
	else
	{
		// The coordinates:
		out << x << y;
	}
}

/*---------------------------------------------------------------
	Implements the reading from a CStream capability of
		CSerializable objects
  ---------------------------------------------------------------*/
void  CPoint2D::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
		{
			// The coordinates:
			float f;
			in >> f; x=f;
			in >> f; y=f;
		} break;
	case 1:
		{
			// The coordinates:
			in >> x >> y;
		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};
}

/**  Textual output stream function.
 */
std::ostream& mrpt::poses::operator << (std::ostream& o, const CPoint2D& p)
{
	o << "(" << p.x << "," << p.y <<")";
	return o;
}

/*---------------------------------------------------------------
The operator D="this"-b is the pose inverse compounding operator.
   The resulting pose "D" is the diference between this pose and "b"
 ---------------------------------------------------------------*/
CPoint2D  CPoint2D::operator - (const CPose2D& b) const
{
	CPoint2D		ret;

	double  ccos = cos(b.phi);
	double  ssin = sin(b.phi);

	ret.x	=  (x - b.x) * ccos + (y - b.y) * ssin;
	ret.y	= -(x - b.x) * ssin + (y - b.y) * ccos;

	return ret;
}



/*---------------------------------------------------------------
	Returns the corresponding 4x4 homogeneous
	  transformation matrix for the point(translation),
	  or pose (translation+orientation).
---------------------------------------------------------------*/
const CMatrixDouble& CPoint2D::getHomogeneousMatrix() const
{
	aux_HM.unit(4);
	aux_HM.set_unsafe(0,3, x );
	aux_HM.set_unsafe(1,3, y );

	return aux_HM;
}

/*---------------------------------------------------------------
	Returns the corresponding 4x4 homogeneous
	  transformation matrix for the point(translation),
	  or pose (translation+orientation).
---------------------------------------------------------------*/
void  CPoint2D::getHomogeneousMatrix(const CMatrixDouble *& m) const
{
	getHomogeneousMatrix();
	m = &aux_HM;
}

bool mrpt::poses::operator < (const CPoint2D &a, const CPoint2D &b)
{
	if (a.x<b.x) return true;
	else return a.y<b.y;
}
