/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers. 



#include <mrpt/slam/CMetricMap.h>
#include <mrpt/poses/CPosePDF.h>
#include <mrpt/slam/CSensoryFrame.h>
#include <mrpt/slam/CSensFrameProbSequence.h>

#include <mrpt/math/utils.h>

using namespace mrpt::slam; using namespace mrpt::utils; using namespace mrpt::poses;

IMPLEMENTS_VIRTUAL_SERIALIZABLE(CMetricMap, CSerializable, mrpt::slam)


/*---------------------------------------------------------------
						Virtual constructor
  ---------------------------------------------------------------*/
CMetricMap::CMetricMap() :
	m_disableSaveAs3DObject ( false )
{
}

/*---------------------------------------------------------------
						Virtual destructor
  ---------------------------------------------------------------*/
CMetricMap::~CMetricMap()
{

}

/*---------------------------------------------------------------
Load the map contents from a CSensFrameProbSequence object,
erasing all previous content of the map.
This is automaticed invoking "insertObservation" for each
observation at the mean 3D robot pose as given by
the "poses::CPosePDF" in the CSensFrameProbSequence object.
  ---------------------------------------------------------------*/
void  CMetricMap::loadFromProbabilisticPosesAndObservations(
			CSensFrameProbSequence		&sfSeq )
{
	CPose3DPDFPtr		posePDF;
	CSensoryFramePtr	sf;
	int					i,n = sfSeq.size();

	// Erase previous contents:
	this->clear();


	// Insert new content:
	for (i=0;i<n;i++)
	{
		sfSeq.get(i,posePDF, sf);

		CPose3D		robotPose(posePDF->getEstimatedPose());
		sf->insertObservationsInto(
				this,		// Insert into THIS map.
				&robotPose	// At this pose.
				);
	}

}

/*---------------------------------------------------------------
  Aligns an observation to its maximum likelihood pose (in 2D)
     into this map, by hill climbing in values computed
	 with "computeObservationLikelihood".
	\param obs The observation to be aligned
	\param in_initialEstimatedPose The initial input to the
			algorithm, an initial "guess" for the observation pose in the map.
	\param out_ResultingPose The resulting pose from this algorithm
  ---------------------------------------------------------------*/
void 	CMetricMap::alignBylikelihoodHillClimbing(
					CObservation		*obs,
					CPose2D				in_initialEstimatedPose,
					CPose2D				&out_ResultingPose )
{
	bool			iterate;
	CPose2D			curEstimation,est[10],estBest;
	double			likCur,lik[10],likBest;
	float			Axy,Ap;
	int				i;

	Axy = 0.01f;
	Ap  = DEG2RAD(1);

	// Initial value:
	curEstimation = in_initialEstimatedPose;

	// The main loop:
	// ---------------------------
	do
	{
		// Current pose:
		likCur=computeObservationLikelihood(obs,curEstimation);

		// Neighbourhood:
		est[0]=curEstimation;  est[0].x+=Axy;
		est[1]=curEstimation;  est[1].x-=Axy;
		est[2]=curEstimation;  est[2].y+=Axy;
		est[3]=curEstimation;  est[3].y-=Axy;
		est[4]=curEstimation;  est[4].phi+=Ap;
		est[5]=curEstimation;  est[5].phi-=Ap;

		est[6]=curEstimation;  est[6].x+=Axy;est[6].y+=Axy;
		est[7]=curEstimation;  est[7].x-=Axy;est[7].y+=Axy;
		est[8]=curEstimation;  est[8].x+=Axy;est[8].y-=Axy;
		est[9]=curEstimation;  est[9].x+=Axy;est[9].y-=Axy;


		// Evaluate and find the best:
		likBest=-1.0f;
		for (i=0;i<10;i++)
		{
			lik[i] = computeObservationLikelihood(obs,est[i]) - likCur;
			if (lik[i]>likBest || i==0)
			{
				likBest = lik[i];
				estBest = est[i];
			}
		}

		// for the next iteration:
		// -----------------------------
		if (likBest>0)
		{
			float	gradX = estBest.x - curEstimation.x;
			float	gradY = estBest.y - curEstimation.y;
			float	gradP = estBest.phi - curEstimation.phi;
			float	Alfa  = 1.0f;

			// Follow the incre. gradient of the MI:
			curEstimation.x+= gradX * Alfa;
			curEstimation.y+= gradY * Alfa;
			curEstimation.phi+= gradP * Alfa;

//			curEstimation = estBest;
			printf("MI_incr=%f\tGrad=(%.02f,%.02f,%.02f)\tNew=(%.03f,%.03f,%.02f)\n",likBest,gradX,gradY,RAD2DEG(gradP),curEstimation.x,curEstimation.y,RAD2DEG(curEstimation.phi));
		}


		// End criterion:
		// ---------------------------
		iterate = (likBest>0);

	} while (iterate);


	out_ResultingPose = curEstimation;
}

/*---------------------------------------------------------------
						dumpToFile
  ---------------------------------------------------------------*/
void  CMetricMap::TMatchingPairList::dumpToFile(const char *fileName)
{
	FILE	*f = os::fopen(fileName,"wt");

	iterator	it;
	for (it=begin();it!=end();it++)
	{
		os::fprintf(f,"%u %u %f %f %f %f %f %f\n",
				it->this_idx,
				it->other_idx,
				it->this_x,
				it->this_y,
				it->this_z,
				it->other_x,
				it->other_y,
				it->other_z);
	}
	os::fclose(f);
}

/*---------------------------------------------------------------
						saveAsMATLABScript
  ---------------------------------------------------------------*/
void CMetricMap::TMatchingPairList::saveAsMATLABScript( const std::string &filName )
{
	FILE	*f = os::fopen(filName.c_str(),"wt");

	fprintf(f,"%% ----------------------------------------------------\n");
	fprintf(f,"%%  File generated automatically by the MRPT method:\n");
	fprintf(f,"%%   CMetricMap::saveAsMATLABScript  \n");
	fprintf(f,"%%  Before calling this script, define the color of lines, eg:\n");
	fprintf(f,"%%     colorLines=[1 1 1]");
	fprintf(f,"%%               J.L. Blanco (C) 2005-2009 \n");
	fprintf(f,"%% ----------------------------------------------------\n\n");

	fprintf(f,"axis equal; hold on;\n");
	iterator	it;
	for (it=begin();it!=end();it++)
	{
		fprintf(f,"line([%f %f],[%f %f],'Color',colorLines);\n",
				it->this_x,
				it->other_x,
				it->this_y,
				it->other_y );
		fprintf(f,"set(plot([%f %f],[%f %f],'.'),'Color',colorLines,'MarkerSize',15);\n",
				it->this_x,
				it->other_x,
				it->this_y,
				it->other_y );
	}
	os::fclose(f);
}

/*---------------------------------------------------------------
						indexOtherMapHasCorrespondence
  ---------------------------------------------------------------*/
bool  CMetricMap::TMatchingPairList::indexOtherMapHasCorrespondence(unsigned int idx)
{
	iterator	it;
	bool		has = false;

	for (it=begin();it!=end() && !has;it++)
	{
		has = it->other_idx == idx;
	}

	return has;
}

bool mrpt::slam::operator < (const mrpt::slam::CMetricMap::TMatchingPair& a, const mrpt::slam::CMetricMap::TMatchingPair& b)
{
	if (a.this_idx==b.this_idx)
			return (a.this_idx<b.this_idx);
	else	return (a.other_idx<b.other_idx);
}


bool mrpt::slam::operator == (const mrpt::slam::CMetricMap::TMatchingPair& a,const mrpt::slam::CMetricMap::TMatchingPair& b)
{
	return (a.this_idx==b.this_idx) && (a.other_idx==b.other_idx);
}

bool mrpt::slam::operator == (const mrpt::slam::CMetricMap::TMatchingPairList& a,const mrpt::slam::CMetricMap::TMatchingPairList& b)
{
	if (a.size()!=b.size())
		return false;
	for (mrpt::slam::CMetricMap::TMatchingPairList::const_iterator it1=a.begin(),it2=b.begin();it1!=a.end();it1++,it2++)
		if (!  ( (*it1)==(*it2)))
			return false;
	return true;
}


/*---------------------------------------------------------------
						overallSquareError
  ---------------------------------------------------------------*/
float CMetricMap::TMatchingPairList::overallSquareError( const CPose2D &q ) const
{
	vector_float errs( size() );
	squareErrorVector(q,errs);
	return math::sum( errs );
}

/*---------------------------------------------------------------
						overallSquareErrorAndPoints
  ---------------------------------------------------------------*/
float CMetricMap::TMatchingPairList::overallSquareErrorAndPoints(
	const CPose2D &q,
	vector_float &xs,
	vector_float &ys ) const
{
	vector_float errs( size() );
	squareErrorVector(q,errs,xs,ys);
	return math::sum( errs );
}


/*---------------------------------------------------------------
						squareErrorVector
  ---------------------------------------------------------------*/
void  CMetricMap::TMatchingPairList::squareErrorVector(const CPose2D &q, vector_float &out_sqErrs ) const
{
	out_sqErrs.resize( size() );
	// *    \f[ e_i = | x_{this} -  q \oplus x_{other}  |^2 \f]

	float ccos = cos(q.phi);
	float csin = sin(q.phi);
	float qx   = q.x;
	float qy   = q.y;
	float  xx, yy;		// Transformed points

	const_iterator 			corresp;
	vector_float::iterator	e_i;
	for (corresp=begin(), e_i = out_sqErrs.begin();corresp!=end();corresp++, e_i++)
	{
		xx = qx + ccos * corresp->other_x - csin * corresp->other_y;
		yy = qy + csin * corresp->other_x + ccos * corresp->other_y;
		*e_i = square( corresp->this_x - xx ) + square( corresp->this_y - yy );
	}
}

/*---------------------------------------------------------------
						squareErrorVector
  ---------------------------------------------------------------*/
void  CMetricMap::TMatchingPairList::squareErrorVector(
	const CPose2D &q,
	vector_float &out_sqErrs,
	vector_float &xs,
	vector_float &ys ) const
{
	out_sqErrs.resize( size() );
	xs.resize( size() );
	ys.resize( size() );

	// *    \f[ e_i = | x_{this} -  q \oplus x_{other}  |^2 \f]

	float ccos = cos(q.phi);
	float csin = sin(q.phi);
	float qx   = q.x;
	float qy   = q.y;

	const_iterator 			corresp;
	vector_float::iterator	e_i, xx, yy;
	for (corresp=begin(), e_i = out_sqErrs.begin(), xx = xs.begin(), yy = ys.begin();corresp!=end();corresp++, e_i++, xx++,yy++)
	{
		*xx = qx + ccos * corresp->other_x - csin * corresp->other_y;
		*yy = qy + csin * corresp->other_x + ccos * corresp->other_y;
		*e_i = square( corresp->this_x - *xx ) + square( corresp->this_y - *yy );
	}
}


/*---------------------------------------------------------------
						computeObservationsLikelihood
  ---------------------------------------------------------------*/
double CMetricMap::computeObservationsLikelihood(
	const CSensoryFrame &sf,
	const CPose2D &takenFrom )
{
	double lik = 0;
	for (CSensoryFrame::const_iterator it=sf.begin();it!=sf.end();it++)
		lik += computeObservationLikelihood( it->pointer(), takenFrom );

	return lik;
}

/*---------------------------------------------------------------
				canComputeObservationLikelihood
  ---------------------------------------------------------------*/
bool CMetricMap::canComputeObservationsLikelihood( const CSensoryFrame &sf )
{
	bool can = false;
	for (CSensoryFrame::const_iterator it=sf.begin();!can && it!=sf.end();it++)
		can = can || canComputeObservationLikelihood( it->pointer()  );
	return can;
}
