/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2009  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Perception and Robotics               |
   |      research group, University of Malaga (Spain).                        |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */


#include <mrpt/utils/utils_defs.h>

#ifdef MRPT_OS_LINUX

// Refer to:
//  http://www.easysw.com/~mike/serial/serial.html

#include <stdio.h>   /* Standard input/output definitions */
#include <string.h>  /* String function definitions */
#include <unistd.h>  /* UNIX standard function definitions */
#include <fcntl.h>   /* File control definitions */
#include <errno.h>   /* Error number definitions */

#include <termios.h> /* POSIX terminal control definitions */

#include <sys/ioctl.h>  // FIONREAD,...
#include <signal.h>
#include <map>

#include <mrpt/utils/CTicTac.h>
#include <mrpt/hwdrivers/CSerialPort.h>

using namespace mrpt;
using namespace mrpt::hwdrivers;
using namespace std;

/* -----------------------------------------------------
                Constructor
   ----------------------------------------------------- */
CSerialPort::CSerialPort( const string &portName, bool openNow )
{
    hCOM = -1;	// Not connected

    m_totalTimeout_ms      = 0;
    m_interBytesTimeout_ms = 0;

    m_serialName = portName;

    if (openNow) open();
}

/* -----------------------------------------------------
                Destructor
   ----------------------------------------------------- */
CSerialPort::~CSerialPort()
{
    if ( isOpen() )
        close();
}

/* -----------------------------------------------------
                Open
   ----------------------------------------------------- */
void  CSerialPort::open( )
{
    MRPT_TRY_START

    // Check name:
    if (!m_serialName.size()) THROW_EXCEPTION("Serial port name is empty!!")
        if (m_serialName[0]!='/') m_serialName = string("/dev/") + m_serialName;

    // Open the serial port:
	// The O_NOCTTY flag tells UNIX that this program doesn't want to be the "controlling terminal" for that port.
	// The O_NDELAY flag tells UNIX that this program doesn't care what state the DCD signal line is in - whether the other end of the port is up and running.
    if ( -1==( hCOM= ::open( m_serialName.c_str(),  O_RDWR | O_NOCTTY | O_NDELAY ) ) )
        THROW_EXCEPTION_CUSTOM_MSG1("Error trying to open the serial port %s!!",m_serialName.c_str());

    // Clear flags:
    fcntl( hCOM, F_SETFL, 0 );

    //
    // Start assembling the new port settings.
    //
    termios port_settings;
    bzero( &port_settings,sizeof( port_settings ) ) ;

    //
    // Enable the receiver (CREAD) and ignore modem control lines
    // (CLOCAL).
    //
    port_settings.c_cflag |= CREAD | CLOCAL ;

    //
    // Set the VMIN and VTIME parameters to zero by default. VMIN is
    // the minimum number of characters for non-canonical read and
    // VTIME is the timeout in deciseconds for non-canonical
    // read. Setting both of these parameters to zero implies that a
    // read will return immediately only giving the currently
    // available characters.
    //
    port_settings.c_cc[ VMIN  ] = 0 ;
    port_settings.c_cc[ VTIME ] = 0 ;

    /*
     * Flush the input buffer associated with the port.
     */
    if ( tcflush( hCOM,TCIFLUSH ) < 0 )
		THROW_EXCEPTION_CUSTOM_MSG1("Cannot flush serial port: %s",strerror(errno) );

    /*
     * Write the new settings to the port.
     */
    if ( tcsetattr( hCOM,TCSANOW,&port_settings ) < 0 )
        THROW_EXCEPTION_CUSTOM_MSG1("Cannot set the new config to the serial port: %s",strerror(errno) ) ;


	// Do NOT block on read.
    fcntl(hCOM, F_SETFL, FNDELAY);

    // Success!
    MRPT_TRY_END
}


/* -----------------------------------------------------
                isOpen
   ----------------------------------------------------- */
bool  CSerialPort::isOpen()
{
    return hCOM != -1;
}

/* -----------------------------------------------------
                setConfig
   ----------------------------------------------------- */
void  CSerialPort::setConfig(
    int		baudRate,
    int		parity,
    int		bits,
    int		nStopBits,
    bool    enableFlowControl )
{
    MRPT_TRY_START

    // Port must be open!
    if (!isOpen()) THROW_EXCEPTION("The serial port is not open!");

    termios port_settings;
    if ( tcgetattr( hCOM, & port_settings ) < 0 )
        THROW_EXCEPTION_CUSTOM_MSG1("Cannot get the current settings: %s",strerror(errno) ) ;

    //
    // Apply baud rate
    //
    int BR;
    switch (baudRate)
    {
    	case 50: BR = B50; break;
    	case 75: BR = B75; break;
    	case 110: BR = B110; break;
    	case 134: BR = B134; break;
    	case 200: BR = B200; break;
    	case 300: BR = B300; break;
    	case 600: BR = B600; break;
    	case 1200: BR = B1200; break;
    	case 2400: BR = B2400; break;
    	case 4800: BR = B4800; break;
    	case 9600: BR = B9600; break;
    	case 19200: BR = B19200; break;
    	case 38400: BR = B38400; break;
    	case 57600: BR = B57600; break;
    	case 115200: BR = B115200; break;
    	case 230400: BR = B230400; break;
    default:
        THROW_EXCEPTION_CUSTOM_MSG1("Invalid desired baud rate value: %i",baudRate ) ;
        break;
    }

    if ( ( cfsetispeed( &port_settings,BR ) < 0 ) ||
            ( cfsetospeed( &port_settings,BR) < 0 ) )
        THROW_EXCEPTION_CUSTOM_MSG1("Cannot change baudRate in setting structure: %s",strerror(errno) ) ;

    //
    // Set the character size.
    //
    port_settings.c_cflag &= ~CSIZE ;
    switch (bits)
    {
    case 5:
        port_settings.c_cflag |= CS5;
        break;
    case 6:
        port_settings.c_cflag |= CS6;
        break;
    case 7:
        port_settings.c_cflag |= CS7;
        break;
    case 8:
        port_settings.c_cflag |= CS8;
        break;
    default:
        THROW_EXCEPTION_CUSTOM_MSG1("Invalid character size: %i",bits ) ;
        break;
    }

    // parity  0:No parity, 1:Odd, 2:Even
    switch ( parity )
    {
    case 2:
        port_settings.c_cflag |= PARENB ;
        port_settings.c_cflag &= ~PARODD ;
        port_settings.c_iflag |= INPCK ;
        break ;
    case 1:
        port_settings.c_cflag |= ( PARENB | PARODD );
        port_settings.c_iflag |= INPCK;
        break ;
    case 0:
        port_settings.c_cflag &= ~(PARENB);
        port_settings.c_iflag |= IGNPAR;
        break ;
    default:
        THROW_EXCEPTION_CUSTOM_MSG1("Invalid parity selection: %i",parity) ;
        break;
    }

    // stop bits:
    switch ( nStopBits )
    {
    case 1:
        port_settings.c_cflag &= ~(CSTOPB) ;
        break ;
    case 2:
        port_settings.c_cflag |= CSTOPB ;
        break ;
    default:
        THROW_EXCEPTION_CUSTOM_MSG1("Invalid number of stop bits: %i",nStopBits) ;
        break;
    }

    //
    // Set the flow control.
    //
    if (enableFlowControl)
    {
        // RTS/CTS ON:
        port_settings.c_cflag |= CRTSCTS ;
    }
    else
    {
        // none
        port_settings.c_cflag &= ~(CRTSCTS) ;
    }

    /* Write the new settings to the port.
     */
    if ( tcsetattr( hCOM,TCSANOW,&port_settings ) < 0 )
        THROW_EXCEPTION_CUSTOM_MSG1("Cannot set the new settings: %s",strerror(errno) ) ;

    m_baudRate = baudRate;

    MRPT_TRY_END
}

/* -----------------------------------------------------
                setConfig
   ----------------------------------------------------- */
void  CSerialPort::setTimeouts(
    int		ReadIntervalTimeout,
    int		ReadTotalTimeoutMultiplier,
    int		ReadTotalTimeoutConstant,
    int		WriteTotalTimeoutMultiplier,
    int		WriteTotalTimeoutConstant )
{
    MRPT_TRY_START

    // Port must be open!
    if (!isOpen()) THROW_EXCEPTION("The serial port is not open!");


    // Save variables which are used in other methods:
    m_totalTimeout_ms      = ReadTotalTimeoutConstant;
    m_interBytesTimeout_ms = ReadIntervalTimeout;


    // http://www.unixwiz.net/techtips/termios-vmin-vtime.html
    // VMIN & VTIME
    termios port_settings;
    if ( tcgetattr( hCOM, & port_settings ) < 0 )
        THROW_EXCEPTION_CUSTOM_MSG1("Cannot get the current settings: %s",strerror(errno) ) ;

    //
    // We set VMIN=0 and VTIME=ReadIntervalTimeout (in thenth of seconds)
    //
    port_settings.c_cc[ VMIN  ] = 0;
    port_settings.c_cc[ VTIME ] = max(1,ReadTotalTimeoutConstant / 100);

    /* Write the new settings to the port.
     */
    if ( tcsetattr( hCOM,TCSANOW,&port_settings ) < 0 )
        THROW_EXCEPTION_CUSTOM_MSG1("Cannot set the new settings: %s",strerror(errno) ) ;

    // Success
    MRPT_TRY_END
}

/* -----------------------------------------------------
                Close
   ----------------------------------------------------- */
void  CSerialPort::close(  )
{
    MRPT_TRY_START

    if (hCOM<0) return; // Already closed

    //
//    PosixSignalDispatcher& signal_dispatcher = PosixSignalDispatcher::Instance() ;
//    signal_dispatcher.DetachHandler( SIGIO, *this ) ;

    // Close the serial port file descriptor.
    ::close(hCOM);

    hCOM=-1;	// Means the port is closed

    MRPT_TRY_END
}

/* -----------------------------------------------------
                read
   ----------------------------------------------------- */
size_t  CSerialPort::Read(void *Buffer, size_t Count)
{
    MRPT_TRY_START

    // Port must be open!
    if (!isOpen()) THROW_EXCEPTION("The port is not open yet!");

	if (!Count) return 0;

    // Use the "m_totalTimeout_ms" global timeout
    //  and the "m_interBytesTimeout_ms" for inter-bytes:
    CTicTac tictac;

    tictac.Tic();

    size_t  alreadyRead = 0;
    int		leftTime = m_totalTimeout_ms - (int)(tictac.Tac()*1000);

    while ( alreadyRead<Count && leftTime>=0 )
    {
    	// Bytes waiting in the queue?
		// Check if we are still connected or there is an error...
		int waiting_bytes=0;
		if ( ioctl(hCOM, FIONREAD, &waiting_bytes) < 0)
		{
			if (errno==EIO)
			{
				// The port has been disconnect (for USB ports)
				this->close();
				return alreadyRead;
			}
		}

		// Are there any bytes??
		int nRead=0;

		if (waiting_bytes>0)
		{
			int nToRead = min( (size_t)waiting_bytes, Count-alreadyRead );

			if ( ( nRead=::read(hCOM, ((char*)Buffer)+alreadyRead, nToRead ) ) <0 )
			{
				cerr << "[CSerialPort] Error reading from port..." << endl;
			}

			alreadyRead+= nRead;
		}
		else
		{
			// Nope...
		}

		if (alreadyRead<Count)
		{
			// Wait 1 more ms for new data to arrive.
			mrpt::system::sleep( 1 );
		}

		// Reset interbytes timer:
		leftTime = m_totalTimeout_ms - (int)(tictac.Tac()*1000);
		if (nRead>0)
			leftTime = max(leftTime, m_interBytesTimeout_ms);
    }

    return alreadyRead;

    MRPT_TRY_END
}

/* -----------------------------------------------------
                write
   ----------------------------------------------------- */
size_t  CSerialPort::Write(const void *Buffer, size_t Count)
{
    MRPT_TRY_START
    // Port must be open!
    if (!isOpen()) THROW_EXCEPTION("The port is not open yet!");

    // Write the data to the serial port. Keep retrying if EAGAIN
    // error is received.

    /** \todo Add support for write timeout here
      */
    int num_of_bytes_written = -1 ;
    do
    {
        num_of_bytes_written = write( hCOM,Buffer, Count );
    }
    while ( ( num_of_bytes_written < 0 ) &&
            ( EAGAIN == errno ) ) ;
    //
    if ( num_of_bytes_written < 0 )
        THROW_EXCEPTION_CUSTOM_MSG1("Error writing data to the serial port: %s",strerror(errno) ) ;

    // OK:
    return num_of_bytes_written;

    MRPT_TRY_END
}

/* -----------------------------------------------------
                purgeBuffers
   ----------------------------------------------------- */
void  CSerialPort::purgeBuffers()
{
    MRPT_TRY_START

    // Port must be open!
    if (!isOpen()) THROW_EXCEPTION("The port is not open yet!");

    /*
     * Flush the input buffer associated with the port.
     */
    if ( tcflush( hCOM,TCIFLUSH ) < 0 )
        THROW_EXCEPTION_CUSTOM_MSG1("Cannot flush serial port: %s",strerror(errno) ) ;

    MRPT_TRY_END
}



#endif  // linux

