/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef  CDisplayWindow_H
#define  CDisplayWindow_H

#include <mrpt/gui/CBaseGUIWindow.h>
#include <mrpt/vision/CFeature.h>

namespace mrpt
{
	namespace utils
	{
		class CImage;
		class CImageFloat;
	}

	/** Classes for creating GUI windows for 2D and 3D visualization. */
	namespace gui
	{
		using namespace mrpt::utils;

		DEFINE_SERIALIZABLE_PRE(CDisplayWindow)

		/** This class creates a window as a graphical user interface (GUI) for displaying images to the user.
		 */
		class MRPTDLLIMPEXP CDisplayWindow : public mrpt::utils::CSerializable, public mrpt::gui::CBaseGUIWindow
		{
			// This must be added to any CSerializable derived class:
			DEFINE_SERIALIZABLE( CDisplayWindow )

		protected:

			/** Enables or disables the visualization of cursor coordinates on the window caption.
			  */
			bool			m_enableCursorCoordinates;

		public:
			/** Constructor
			 */
			CDisplayWindow( const std::string &windowCaption = std::string() );

			/** Class factory returning a smart pointer */
			static CDisplayWindowPtr Create(
				const std::string	&windowCaption = std::string() ) 
			{
				return CDisplayWindowPtr(new CDisplayWindow(windowCaption));
			}

			/** Destructor
			 */
			virtual ~CDisplayWindow();

			/** Show a given color or grayscale image on the window and print a set of points on it.
			 *  It adapts the size of the window to that of the image.
			 */
			void  showImageAndPoints( const CImage &img, const vector_float &x, const vector_float &y, const TColor &color = TColor::red );

			/** Show a given color or grayscale image on the window and print a set of points on it.
			 *  It adapts the size of the window to that of the image.
			 */
			void  showImageAndPoints( const	CImageFloat &img, const vector_float &x, const vector_float &y, const TColor &color = TColor::red );

			/** Show a given color or grayscale image on the window and print a set of points on it.
			 *  It adapts the size of the window to that of the image.
			 */
			void  showImageAndPoints( const CImage &img, const mrpt::vision::CFeatureList &list, const TColor &color = TColor::red );

			/** Show a given color or grayscale image on the window and print a set of points on it and a set of lines splitting the image in tiles.
			 *  It adapts the size of the window to that of the image.
			 */
			void  showTiledImageAndPoints( const CImage &img, const mrpt::vision::CFeatureList &list, const TColor &color = TColor::red );

			/** Show a pair of given color or grayscale images (put together) on the window and print a set of matches on them.
			 *  It adapts the size of the window to that of the image.
			 */
			void  showImagesAndMatchedPoints( const	CImage &img1, const	CImage &img2, const mrpt::vision::CMatchedFeatureList &mList, const TColor &color = TColor::red );

			/** Show a given color or grayscale image on the window.
			 *  It adapts the size of the window to that of the image.
			 */
			void  showImage( const CImage	&img );

			/** Show a given grayscale image on the window.
			 *  It adapts the size of the window to that of the image.
			 */
			void  showImage( const CImageFloat	&img );

			/** Plots a graph in MATLAB-like style.
			 */
			void  plot( const vector_float &x, const vector_float &y );

			/** Plots a graph in MATLAB-like style.
			 */
			void  plot( const vector_float &y );

			/** Resizes the window, stretching the image to fit into the display area.
			 */
			void  resize( unsigned int width, unsigned int height );

			/** Changes the position of the window on the screen.
			 */
			void  setPos( int x, int y );

			/** Enables or disables the visualization of cursor coordinates on the window caption (default = enabled).
			  */
			void  enableCursorCoordinatesVisualization(bool enable)
			{
				m_enableCursorCoordinates = enable;
			}

			/** Changes the window title text.
			  */
			void  setWindowTitle( const std::string &str );

		}; // End of class def.

	} // End of namespace

} // End of namespace

#endif
