/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */
#ifndef CObservation3DRangeScan_H
#define CObservation3DRangeScan_H

#include <mrpt/utils/CSerializable.h>
#include <mrpt/slam/CObservation.h>
#include <mrpt/poses/CPose3D.h>
#include <mrpt/poses/CPose2D.h>
#include <mrpt/slam/CPointsMap.h>

#include <mrpt/math/CPolygon.h>


namespace mrpt
{
namespace slam
{

	DEFINE_SERIALIZABLE_PRE_CUSTOM_BASE( CObservation3DRangeScan, CObservation )

	/** Declares a class derived from "CObservation" that
	       encapsules a 3D range scan measurement (e.g. from a time of flight range camera).
	 *
	 * \sa CObservation
	 */
	class MRPTDLLIMPEXP CObservation3DRangeScan : public CObservation
	{
		// This must be added to any CSerializable derived class:
		DEFINE_SERIALIZABLE( CObservation3DRangeScan )

	public:
		class CAuxMapWrapper
		{
			CPointsMap			*auxPointsMap;
		public:
			CAuxMapWrapper() :auxPointsMap(NULL) { }
			CAuxMapWrapper(const CAuxMapWrapper &o) : auxPointsMap(NULL) { }
			CAuxMapWrapper & operator =(const CAuxMapWrapper &o) { clear(); return *this; }

			~CAuxMapWrapper() { clear(); }

			void clear() { if (auxPointsMap) delete auxPointsMap; auxPointsMap=NULL; }

			CPointsMap	*get() { return auxPointsMap; }
			const CPointsMap	*get() const { return auxPointsMap; }

			void set(CPointsMap	*m) { if (auxPointsMap) delete auxPointsMap;  auxPointsMap = m; }

		};

	protected:
		mutable CAuxMapWrapper	m_auxMap;

	public:
		/** Default constructor
		 */
		CObservation3DRangeScan( );

		/** Destructor
		 */
		~CObservation3DRangeScan( );

		/** The x y and z coordinates of the 3D range measurement, in meters.
		  */
		vector_float	    scan_x;
		vector_float	    scan_y;
		vector_float	    scan_z;

		/** It's false (=0) for points outside of [min_distance, max_distance], referenced to elements in "scan_x"
          * \sa truncateByDistance
		  */
		std::vector<char>	validRange;

		/** The maximum range allowed by the device, in meters (e.g. 8.0m, 5.0m,...)
		  */
		float				maxRange;

		/** The 6D pose of the sensor on the robot.
		  */
		CPose3D				sensorPose;

		/** The "sigma" error of the device in meters, used while inserting the scan in an occupancy grid.
		  */
		float				stdError;

		/** Auxiliary members used to speed up some algorithms: it will be NULL normally.
		  * \sa buildAuxPointsMap
		  */
		const CPointsMap	*getAuxPointsMap() const { return m_auxMap.get(); }

		/** Returns an auxiliary points map built from the scan; it is actually build at the first call to this method, and only once.
		  * \param options Can be NULL to use default point maps' insertion options, or a valid pointer to override some params.
		  * \sa auxPointsMap
		  */
		const CPointsMap	*buildAuxPointsMap( CPointsMap::TInsertionOptions *options = NULL ) const;

		 /** Implements the virtual method in charge of finding the likelihood between this
		  *   and another observation, probably only of the same derived class. The operator
		  *   may be asymmetric.
		  *
		  * \param anotherObs The other observation to compute likelihood with.
		  * \param anotherObsPose If known, the belief about the robot pose when the other observation was taken can be supplied here, or NULL if it is unknown.
		  *
		  * \return Returns a likelihood measurement, in the range [0,1].
		  *	\exception std::exception On any error, as another observation being of an invalid class.
		  */
		 float  likelihoodWith( const CObservation *anotherObs, const CPosePDF *anotherObsPose = NULL ) const;

		/** A general method to retrieve the sensor pose on the robot.
		  *  Note that most sensors will return a full (6D) CPose3D, but see the derived classes for more details or special cases.
		  * \sa setSensorPose
		  */
		void getSensorPose( CPose3D &out_sensorPose ) const { out_sensorPose = sensorPose; }

		/** A general method to change the sensor pose on the robot.
		  *  Note that most sensors will use the full (6D) CPose3D, but see the derived classes for more details or special cases.
		  * \sa getSensorPose
		  */
		void setSensorPose( const CPose3D &newSensorPose ) { sensorPose = newSensorPose; }

		/** A method to truncate the scan by defining a minimum and maximum valid distance
		  */
		void truncateByDistance(float min_distance, float max_distance);

		/** Mark as invalid sensed points that fall within any of a set of "exclusion areas", given in coordinates relative to the vehicle (taking into account "sensorPose").
		  * \sa C2DRangeFinderAbstract::loadExclusionAreas
		  */
//		void filterByExclusionAreas( const std::vector<mrpt::math::CPolygon>  &areas );

	}; // End of class def.


	} // End of namespace

	namespace utils
	{
		using namespace ::mrpt::slam;
		// Specialization must occur in the same namespace
		MRPT_DECLARE_TTYPENAME_PTR(CObservation3DRangeScan)
	}

} // End of namespace

#endif
