/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.


#include <mrpt/scan_matching/scan_matching.h>
#include <mrpt/poses/CPosePDFGaussian.h>
#include <mrpt/poses/CPosePDFSOG.h>
#include <mrpt/random.h>
#include <mrpt/math/CMatrixD.h>
#include <mrpt/math/utils.h>
#include <mrpt/math/CQuaternion.h>

#include <algorithm>

using namespace mrpt;
using namespace mrpt::scan_matching;
using namespace mrpt::slam;
using namespace mrpt::random;
using namespace mrpt::utils;
using namespace std;


/*---------------------------------------------------------------
	leastSquareErrorRigidTransformation6D
  ---------------------------------------------------------------*/
bool  scan_matching::leastSquareErrorRigidTransformation6D(
	const CMetricMap::TMatchingPairList	&in_correspondences,
	CPose3D								&out_transformation,
	double								&out_scale,
	const bool 							forceScaleToUnity)
{

	MRPT_START;
	if( in_correspondences.size() < 3 )
	{
		std::cout << "Error in ICP6D: At least 3 correspondences must be provided" << std::endl;
		return false;
	}

	CPoint3D					cL, cR;
	CMatrixD					S, N;
	CMatrixD					Z, D;
	CMatrixD					Rot;				// Rotation matrix

	double						roll, pitch ,yaw;

	std::vector<double>			v;
	int							nMatches = in_correspondences.size();

	double						s;					// Scale

	// Compute the centroid
	CMetricMap::TMatchingPairList::const_iterator	itMatch;

	for(itMatch = in_correspondences.begin(); itMatch != in_correspondences.end(); itMatch++)
	{
		cL.x_incr( itMatch->this_x );
		cL.y_incr( itMatch->this_y );
		cL.z_incr( itMatch->this_z );

		cR.x_incr( itMatch->other_x );
		cR.y_incr( itMatch->other_y );
		cR.z_incr( itMatch->other_z );
	}
	const double F = 1.0/nMatches;
	cL *= F;
	cR *= F;

	CMetricMap::TMatchingPairList			auxList( in_correspondences );
	CMetricMap::TMatchingPairList::iterator auxIt;
	// Substract the centroid
	for( auxIt = auxList.begin(); auxIt != auxList.end(); auxIt++ )
	{
		auxIt->this_x -= cL.x();
		auxIt->this_y -= cL.y();
		auxIt->this_z -= cL.z();

		auxIt->other_x -= cR.x();
		auxIt->other_y -= cR.y();
		auxIt->other_z -= cR.z();
	}

	// Compute the S matrix of products
	S.setSize(3,3);
	S.fill(0);
	for( auxIt = auxList.begin(); auxIt != auxList.end(); auxIt++ )
	{
		S(0,0) += auxIt->this_x * auxIt->other_x;
		S(0,1) += auxIt->this_x * auxIt->other_y;
		S(0,2) += auxIt->this_x * auxIt->other_z;

		S(1,0) += auxIt->this_y * auxIt->other_x;
		S(1,1) += auxIt->this_y * auxIt->other_y;
		S(1,2) += auxIt->this_y * auxIt->other_z;

		S(2,0) += auxIt->this_z * auxIt->other_x;
		S(2,1) += auxIt->this_z * auxIt->other_y;
		S(2,2) += auxIt->this_z * auxIt->other_z;
	}

	N.setSize(4,4);
	N.fill(0);

	N(0,0) = S(0,0) + S(1,1) + S(2,2);
	N(0,1) = S(1,2) - S(2,1);
	N(0,2) = S(2,0) - S(0,2);
	N(0,3) = S(0,1) - S(1,0);

	N(1,0) = N(0,1);
	N(1,1) = S(0,0) - S(1,1) - S(2,2);
	N(1,2) = S(0,1) + S(1,0);
	N(1,3) = S(2,0) + S(0,2);

	N(2,0) = N(0,2);
	N(2,1) = N(1,2);
	N(2,2) = -S(0,0) + S(1,1) - S(2,2);
	N(2,3) = S(1,2) + S(2,1);

	N(3,0) = N(0,3);
	N(3,1) = N(1,3);
	N(3,2) = N(2,3);
	N(3,3) = -S(0,0) - S(1,1) + S(2,2);

	// q is the quaternion correspondent to the greatest eigenvector of the N matrix (last column in Z)
	N.eigenVectors( Z, D );
	Z.extractCol( Z.getColCount()-1, v );

	//CMatrix v = N.largestEigenvector();
	math::CQuaternion<double>	q( v[0], v[1], v[2], v[3] );
	//q = new math::CQuaternion( v[0], v[1], v[2], v[3] );

	// Compute Rotation
	q.rpy( roll, pitch, yaw );					// Rotation angles
	//std::cout << "YPR: " << yaw << "," << pitch << ","  << roll << std::endl;
	q.q_rotation_matrix( Rot );				// Rotation matrix

	// Compute scale
	double	num = 0.0;
	double	den = 0.0;
	for( auxIt = auxList.begin(); auxIt != auxList.end(); auxIt++ )
	{
		num += square(auxIt->other_x) + square(auxIt->other_y) + square(auxIt->other_z);
		den += square(auxIt->this_x) + square(auxIt->this_y) + square(auxIt->this_z);
	}
	s = sqrt( num/den );

	// Enforce scale to be 1
	out_scale = s;
	if (forceScaleToUnity)
		s = 1.0;

	// Compute traslation
	double x = cR.x() - s*( Rot.get_unsafe(0,0)*cL.x() + Rot.get_unsafe(0,1)*cL.y() + Rot.get_unsafe(0,2)*cL.z() );
	double y = cR.y() - s*( Rot.get_unsafe(1,0)*cL.x() + Rot.get_unsafe(1,1)*cL.y() + Rot.get_unsafe(1,2)*cL.z() );
	double z = cR.z() - s*( Rot.get_unsafe(2,0)*cL.x() + Rot.get_unsafe(2,1)*cL.y() + Rot.get_unsafe(2,2)*cL.z() );

	//std::cout << "Rotation matrix: " << std::endl << Rot << std::endl;

	// Set out_transformation
	out_transformation.setFromValues( x, y, z, yaw, pitch, roll );
	//std::cout << "ICP:" << x << "," << y << "," << z << "," << yaw << "," << pitch << "," << roll << std::endl;

	MRPT_END;

	return true;

/*---------------------------------------------------------------
			leastSquareErrorRigidTransformation in 6D
  ---------------------------------------------------------------*/
	// Algorithm:
	// 0. Preliminary
	//		pLi = { pLix, pLiy, pLiz }
	//		pRi = { pRix, pRiy, pRiz }
	// -------------------------------------------------------
	// 1. Find the centroids of the two sets of measurements:
	//		cL = (1/n)*sum{i}( pLi )		cL = { cLx, cLy, cLz }
	//		cR = (1/n)*sum{i}( pRi )		cR = { cRx, cRy, cRz }
	//
	// 2. Substract centroids from the point coordinates:
	//		pLi' = pLi - cL					pLi' = { pLix', pLiy', pLiz' }
	//		pRi' = pRi - cR					pRi' = { pRix', pRiy', pRiz' }
	//
	// 3. For each pair of coordinates (correspondences) compute the nine possible products:
	//		pi1 = pLix'*pRix'		pi2 = pLix'*pRiy'		pi3 = pLix'*pRiz'
	//		pi4 = pLiy'*pRix'		pi5 = pLiy'*pRiy'		pi6 = pLiy'*pRiz'
	//		pi7 = pLiz'*pRix'		pi8 = pLiz'*pRiy'		pi9 = pLiz'*pRiz'
	//
	// 4. Compute S components:
	//		Sxx = sum{i}( pi1 )		Sxy = sum{i}( pi2 )		Sxz = sum{i}( pi3 )
	//		Syx = sum{i}( pi4 )		Syy = sum{i}( pi5 )		Syz = sum{i}( pi6 )
	//		Szx = sum{i}( pi7 )		Szy = sum{i}( pi8 )		Szz = sum{i}( pi9 )
	//
	// 5. Compute N components:
	//			[ Sxx+Syy+Szz	Syz-Szy			Szx-Sxz			Sxy-Syx		 ]
	//			[ Syz-Szy		Sxx-Syy-Szz		Sxy+Syx			Szx+Sxz		 ]
	//		N = [ Szx-Sxz		Sxy+Syx			-Sxx+Syy-Szz	Syz+Szy		 ]
	//			[ Sxy-Syx		Szx+Sxz			Syz+Szy			-Sxx-Syy+Szz ]
	//
	// 6. Rotation represented by the quaternion eigenvector correspondent to the higher eigenvalue of N
	//
	// 7. Scale computation (symmetric expression)
	//		s = sqrt( sum{i}( square(abs(pRi')) / sum{i}( square(abs(pLi')) ) )
	//
	// 8. Translation computation (distance between the Right centroid and the scaled and rotated Left centroid)
	//		t = cR-sR(cL)
}
