/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/precomp_core.h>  // Only for precomp. headers, include all libmrpt-core headers.



#include <mrpt/slam/CObservation3DRangeScan.h>
#include <mrpt/slam/CSimplePointsMap.h>
#include <mrpt/poses/CPosePDF.h>

#include <mrpt/math/utils.h>

using namespace std;
using namespace mrpt::slam;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace mrpt::math;

// This must be added to any CSerializable class implementation file.
IMPLEMENTS_SERIALIZABLE(CObservation3DRangeScan, CObservation,mrpt::slam)

/*---------------------------------------------------------------
							Constructor
 ---------------------------------------------------------------*/
CObservation3DRangeScan::CObservation3DRangeScan( ) :
	m_auxMap(),
	scan_x(),
	scan_y(),
	scan_z(),
	validRange(),
	maxRange( 8.0f ), // XXX May need to be adjusted
	sensorPose(),
	stdError( 0.01f )
{
	// Reserve typical sizes for a Swissranger 3D scan (176*144 resolution)
	scan_x.reserve( 25344 );
	scan_y.reserve( 25344 );
	scan_z.reserve( 25344 );
	validRange.reserve( 25344 );
}

/*---------------------------------------------------------------
							Destructor
 ---------------------------------------------------------------*/
CObservation3DRangeScan::~CObservation3DRangeScan()
{
}

/*---------------------------------------------------------------
  Implements the writing to a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CObservation3DRangeScan::writeToStream(CStream &out, int *version) const
{
	if (version)
		*version = 0;
	else
	{
		// The data
		out << maxRange << sensorPose;
		uint32_t	N = scan_x.size();
		out << N;
		ASSERT_(validRange.size() == scan_x.size() && scan_x.size() == scan_y.size() && scan_y.size() == scan_z.size() );
		out.WriteBuffer( &scan_x[0], sizeof(scan_x[0])*N );
		out.WriteBuffer( &scan_y[0], sizeof(scan_y[0])*N );
		out.WriteBuffer( &scan_z[0], sizeof(scan_z[0])*N );
		out.WriteBuffer( &validRange[0],sizeof(validRange[0])*N );
		out << stdError;
		out << timestamp;

		out << sensorLabel;
	}
}

/*---------------------------------------------------------------
  Filter out invalid points by a minimum distance, a maximum angle and a certain distance at the end (z-coordinate of the lasers must be provided)
 ---------------------------------------------------------------*/
void CObservation3DRangeScan::truncateByDistance(float min_distance, float max_distance)
{
	// FILTER OUT INVALID POINTS!!
	vector_float::iterator		itScan;
	std::vector<char>::iterator itValid;

	for( itScan = scan_x.begin(), itValid = validRange.begin();
		 itScan != scan_x.end();
		 itScan++, itValid++ )
	{
		if( *itScan < min_distance && *itScan > max_distance )
			*itValid = false;
	}
}

/*---------------------------------------------------------------
  Implements the reading from a CStream capability of CSerializable objects
 ---------------------------------------------------------------*/
void  CObservation3DRangeScan::readFromStream(CStream &in, int version)
{
	switch(version)
	{
	case 0:
		{
			uint32_t		N;

			CMatrix covSensorPose;

			in >> maxRange >> sensorPose;

			in >> N;
			scan_x.resize(N);
			in.ReadBuffer( &scan_x[0], sizeof(scan_x[0])*N);
			scan_y.resize(N);
			in.ReadBuffer( &scan_y[0], sizeof(scan_y[0])*N);
			scan_z.resize(N);
			in.ReadBuffer( &scan_z[0], sizeof(scan_z[0])*N);
			validRange.resize(N);
			in.ReadBuffer( &validRange[0], sizeof(validRange[0])*N );
			in >> stdError;
			in.ReadBuffer( &timestamp, sizeof(timestamp));

			in >> sensorLabel;

		} break;
	default:
		MRPT_THROW_UNKNOWN_SERIALIZATION_VERSION(version)

	};

}

/*---------------------------------------------------------------
 Inserts a pure virtual method for finding the likelihood between this
   and another observation, probably only of the same derived class. The operator
   may be asymmetric.

 \param anotherObs The other observation to compute likelihood with.
 \param anotherObsPose If known, the belief about the robot pose when the other observation was taken can be supplied here, or NULL if it is unknown.

 \return Returns a likelihood measurement, in the range [0,1].
 \exception std::exception On any error, as another observation being of an invalid class.
  ---------------------------------------------------------------*/
float  CObservation3DRangeScan::likelihoodWith(
	const CObservation		*anotherObs,
	const CPosePDF			*anotherObsPose ) const
{
	CSimplePointsMap						map1,map2;
	const CObservation3DRangeScan			*obs;
	float									maxDisForCorrespondence = 0.04f; // XXX Will need adjustment for 3D range camera
	CMetricMap::TMatchingPairList			correspondences;
	float									correspondencesRatio;
	CPose2D									otherObsPose(0,0,0);
	static CPose2D							nullPose(0,0,0);


	if ( this->GetRuntimeClass() != anotherObs->GetRuntimeClass() )
		THROW_EXCEPTION("anotherPose is not a CObservation3DRangeScan object")

	// Now we know it is safe to make the casting:
	obs = static_cast<const CObservation3DRangeScan*>(anotherObs);

	// Build a 3D points map for each scan:
	map1.insertionOptions.minDistBetweenLaserPoints = maxDisForCorrespondence;
	map2.insertionOptions.minDistBetweenLaserPoints = maxDisForCorrespondence;

	map1.loadFromRangeScan( *this );
	map2.loadFromRangeScan( *obs );

	// if PDF is available, get "mean" value as an estimation:
	if (anotherObsPose)	anotherObsPose->getMean(otherObsPose);

	map1.computeMatchingWith2D(
			&map2,
			otherObsPose,
			maxDisForCorrespondence,
			0,
			nullPose,
			correspondences,
			correspondencesRatio);

	return correspondencesRatio;
}

/*---------------------------------------------------------------
						buildAuxPointsMap
 ---------------------------------------------------------------*/
const CPointsMap	*CObservation3DRangeScan::buildAuxPointsMap( CPointsMap::TInsertionOptions *options ) const
{
	if (!m_auxMap.get())
	{
		// Create:
		CPointsMap *m = new CSimplePointsMap();
		if (options)
			m->insertionOptions = *options;
		m->insertObservation( this );
		m_auxMap.set(m);
	}

	return m_auxMap.get();
}

/*---------------------------------------------------------------
						filterByExclusionAreas
 ---------------------------------------------------------------*/
//#include <mrpt/utils/CFileOutputStream.h>

//void CObservation3DRangeScan::filterByExclusionAreas( const std::vector<mrpt::math::CPolygon>  &areas )
//{
//	if (areas.empty()) return;
//
//	MRPT_START
//
//	size_t  sizeRangeScan = scan_x.size();
//
//	ASSERT_(validRange.size() == scan_x.size() && scan_x.size() == scan_y.size() && scan_y.size() == scan_z.size() );
//
//	if (!sizeRangeScan) return;
//
//	std::vector<char>::iterator   valid_it;
//	vector_float::const_iterator  scan_x_it, scan_y_it, scan_z_it;
//
//	for (scan_x_it=scan_x.begin(), scan_y_it=scan_y.begin(), scan_z_it=scan_z.begin(), valid_it=validRange.begin(); 
//		scan_x_it!=scan_x.end(); 
//		scan_x_it++, scan_y_it++, scan_z_it++, valid_it++)
//	{
//		if (! *valid_it)
//			continue; // Already it's invalid
//
//		// Compute point in 2D, local to the laser center:
//		const double Lx = *scan_x_it;
//		const double Ly = *scan_y_it;
//		const double Lz = *scan_z_it;
//
//		// To real 3D pose:
//		double Gx,Gy,Gz;
//		this->sensorPose.composePoint(Lx,Ly,Lz, Gx,Gy,Gz);
//
//		// Filter by X,Y:
//		for (std::vector<mrpt::math::CPolygon>::const_iterator i=areas.begin();i!=areas.end();++i)
//		{
//			if ( i->PointIntoPolygon(Gx,Gy) )
//			{
//				*valid_it = false;
//				break; // Go for next point
//			}
//		} // for each area
//	} // for each point
//
//	MRPT_END
//}
