/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/*----------------------------------------------------------------------*
 * Originally written:
 *    2002      Alexis <materm@tele2.fr>
 * 
 *----------------------------------------------------------------------*/

/* Resource management
 *
 * $Id: xresources.c,v 1.9 2004/08/15 16:20:38 alexis Exp $
 */

#include "../config.h"
#include "rxvt.h"		/* NECESSARY */
#include "xresources.h"

#ifdef DEBUG
#define DEBUG_XRSCR_ALEXIS 1
#else 
#define DEBUG_XRSCR_ALEXIS 0
#endif

#if DEBUG_XRSCR_ALEXIS
#define DXR_ALEXIS(d,x) if(d <= DEBUG_XRSCR_ALEXIS) fprintf x
#else
#define DXR_ALEXIS(d,x)
#endif

/* local variables */
static const char *rs_loginShell = NULL;
static const char *rs_utmpInhibit = NULL;
static const char *rs_scrollBar = NULL;
static const char *rs_scrollBar_right = NULL;
static const char *rs_scrollBar_floating = NULL;
static const char *rs_scrollTtyOutputInh = NULL;
static const char *rs_scrollKeypress = NULL;

#ifdef TRANSPARENT
static const char *rs_transparent = NULL;
static const char *rs_transparent_sb = NULL;
#endif

#ifdef DONT_TILE_PIXMAP_OPTION
static const char *rs_dontTilePixmap = NULL;
static const char *rs_dontTilePixmapH = NULL;
static const char *rs_dontTilePixmapV = NULL;
#endif

#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
static const char *rs_bigfont_key = NULL;
static const char *rs_smallfont_key = NULL;
#endif

#ifndef NO_MAPALERT
# ifdef MAPALERT_OPTION
static const char *rs_mapAlert = NULL;
# endif
#endif
static const char *rs_visualBell = NULL;
static const char *rs_reverseVideo = NULL;

#ifdef META8_OPTION
static const char *rs_meta8 = NULL;
#endif
#ifdef MULTICHAR_SET
static const char *rs_multichar_encoding = NULL;
#endif
#ifdef GREEK_SUPPORT
static const char *rs_greek_keyboard = NULL;
#endif

/* monolithic option/resource structure: */
/*
 * `string' options MUST have a usage argument
 * `switch' and `boolean' options have no argument
 *
 * if there's no desc(ription), it won't appear in usage()
 */
static const struct
{
	unsigned long flag;
	const char **dp;	/* data pointer */
	const char *const kw;	/* keyword */
	const char *const opt;	/* option */
	const char *const arg;	/* argument */
	const char *const desc;	/* description */
}
optList[] =
{

/*
 * INFO() - descriptive information only
 * STRG() - command-line option, with/without resource
 * RSTRG() - resource/long-option
 * BOOL() - regular boolean `-/+' flag
 * SWCH() - `-' flag
 */
#define INFO(opt, arg, desc)         {0, NULL, NULL, opt, arg, desc}
#define STRG(p, kw, opt, arg, desc)	 {0, &p, kw, opt, arg, desc}
#define RSTRG(p, kw, arg)	           {0, &p, kw, NULL, arg, NULL}
#define BOOL(p, kw, opt, flag, desc) {(Opt_Boolean|flag), &p, kw, opt, NULL, desc}
#define SWCH(opt, flag, desc)	       {(flag), NULL, NULL, opt, NULL, desc}

/* convenient macros */
#define optList_strlen(i)	(optList[i].flag ? 0 : (optList[i].arg ? strlen (optList[i].arg) : 1))
#define optList_isBool(i)	(optList[i].flag & Opt_Boolean)
#define optList_size()		(sizeof(optList) / sizeof(optList[0]))

  STRG (display_name, NULL, "display", "string", "X server to contact"), 
  STRG (rs_term_name, "termName", "tn", "string", "value of the TERM environment variable"), 
  STRG (rs_geometry, "geometry", "geometry", "geometry", "size (in characters) and position"), 
  STRG (display_name, NULL, "d", NULL, NULL),	/* short form */
  STRG (rs_geometry, NULL, "g", NULL, NULL),	/* short form */
  BOOL (rs_reverseVideo, "reverseVideo", "rv", Opt_reverseVideo,"reverse video"),
#ifdef _MYSTYLE_
  STRG (rs_mystyle, "MyStyle", "mst", "name", "MyStyle"),
#endif
  STRG (rs_color[Color_bg], "background", "bg", "color","background color"),
//  STRG (rs_tab_title, "tab_title", "tt", "string","tab titles"),
  STRG (rs_term_num, "tab_num", "nt", "number","numbers of tab at startup"),
  STRG (rs_color[Color_fg], "foreground", "fg", "color","foreground color"),
  /* colors: command-line long-option = resource name */
  RSTRG (rs_color[minCOLOR + 0], "color0", "color"),
  RSTRG (rs_color[minCOLOR + 1], "color1", "color"),
  RSTRG (rs_color[minCOLOR + 2], "color2", "color"),
  RSTRG (rs_color[minCOLOR + 3], "color3", "color"),
  RSTRG (rs_color[minCOLOR + 4], "color4", "color"),
  RSTRG (rs_color[minCOLOR + 5], "color5", "color"),
  RSTRG (rs_color[minCOLOR + 6], "color6", "color"),
  RSTRG (rs_color[minCOLOR + 7], "color7", "color"),
#ifndef NO_BRIGHTCOLOR
  RSTRG (rs_color[minBrightCOLOR + 0], "color8", "color"),
  RSTRG (rs_color[minBrightCOLOR + 1], "color9", "color"),
  RSTRG (rs_color[minBrightCOLOR + 2], "color10", "color"),
  RSTRG (rs_color[minBrightCOLOR + 3], "color11", "color"),
  RSTRG (rs_color[minBrightCOLOR + 4], "color12", "color"),
  RSTRG (rs_color[minBrightCOLOR + 5], "color13", "color"),
  RSTRG (rs_color[minBrightCOLOR + 6], "color14", "color"),
  RSTRG (rs_color[minBrightCOLOR + 7], "color15", "color"),
#endif /* NO_BRIGHTCOLOR */
#ifndef NO_BOLDUNDERLINE
  RSTRG (rs_color[Color_BD], "colorBD", "color"),
  RSTRG (rs_color[Color_UL], "colorUL", "color"),
#endif /* NO_BOLDUNDERLINE */
#ifdef KEEP_SCROLLCOLOR
  RSTRG (rs_color[Color_scroll], "scrollColor", "color"),
  RSTRG (rs_color[Color_trough], "troughColor", "color"),
#endif /* KEEP_SCROLLCOLOR */
#if defined (BACKGROUND_IMAGE) || (MENUBAR_MAX)
  RSTRG (rs_path, "path", "search path"),
#endif /* defined (BACKGROUND_IMAGE) || (MENUBAR_MAX) */
#ifdef BACKGROUND_IMAGE
  STRG (rs_backgroundPixmap,"backgroundPixmap","pixmap", "file[;geom]", "background pixmap"),
#endif /* BACKGROUND_IMAGE */
#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT)
  STRG (rs_backgroundType,"backgroundType","bgtype", BGT_ALL,
    "type of the background pixmap transformation"),
  STRG (rs_tintType, "tintingType", "tinttype", TINT_TYPE_ALL,
    "defines function to be used for background tinting"),
  STRG (rs_shade, "shading", "sh", "%",
    "make transparent background x% darker"),
  STRG (rs_color[Color_tint], "tinting", "tint", "color",
    "tinted transparency color"),
#endif
  STRG (rs_textType,"textType","txttype", GC_TYPE_ALL,
    "defines function to be used for text drawing"),
#ifdef OFF_FOCUS_FADING
  STRG (rs_fade,"fading","fade", "%",
    "make colors x% darker when aterm is loosing focus."),
#endif
#ifdef BOOL
#ifdef TRANSPARENT
  BOOL (rs_transparent, "transparent", "tr", Opt_transparent,
    "transparent background"),
  BOOL (rs_transparent_sb, "transpscrollbar", "trsb",
    Opt_transparent_sb, "transparent scrollbar"),
#endif
#endif
#if (MENUBAR_MAX)
  RSTRG (rs_menu, "menu", "name[;tag]"),
#endif
#ifndef NO_BOLDFONT
    STRG (rs_boldFont, "boldFont", "fb", "fontname","bold text font"),
#endif
  STRG (rs_font[0], "font", "fn", "fontname","normal text font"),
/* fonts: command-line option = resource name */
#if NFONTS > 1
  RSTRG (rs_font[1], "font1", "fontname"),
#endif
#if NFONTS > 2
  RSTRG (rs_font[2], "font2", "fontname"),
#endif
#if NFONTS > 3
  RSTRG (rs_font[3], "font3", "fontname"),
#endif
#if NFONTS > 4
  RSTRG (rs_font[4], "font4", "fontname"),
#endif
#if NFONTS > 5
  RSTRG (rs_font[5], "font5", "fontname"),
#endif
#if NFONTS > 6
  RSTRG (rs_font[6], "font6", "fontname"),
#endif
#if NFONTS > 7
  RSTRG (rs_font[7], "font7", "fontname"),
#endif
#ifdef MULTICHAR_SET
  STRG (rs_mfont[0], "mfont", "fm", "fontname","multichar font"),
/* fonts: command-line option = resource name */
# if NFONTS > 1
  RSTRG (rs_mfont[1], "mfont1", "fontname"),
# endif
# if NFONTS > 2
  RSTRG (rs_mfont[2], "mfont2", "fontname"),
# endif
# if NFONTS > 3
  RSTRG (rs_mfont[3], "mfont3", "fontname"),
# endif
# if NFONTS > 4
  RSTRG (rs_mfont[4], "mfont4", "fontname"),
# endif
# if NFONTS > 5
  RSTRG (rs_mfont[5], "mfont5", "fontname"),
# endif
# if NFONTS > 6
  RSTRG (rs_mfont[6], "mfont6", "fontname"),
# endif
# if NFONTS > 7
  RSTRG (rs_mfont[7], "mfont7", "fontname"),
# endif
#endif /* MULTICHAR_SET */
#ifdef MULTICHAR_SET
  STRG (rs_multichar_encoding, "multichar_encoding", "km","mode",
    "multiple-character font encoding; mode = eucj | sjis | big5"),
#endif /* MULTICHAR_SET */
#ifdef GREEK_SUPPORT
  STRG (rs_greek_keyboard, "greek_keyboard", "grk", "mode",
    "greek keyboard mapping; mode = iso | ibm"),
#endif
  SWCH ("iconic", Opt_iconic, "start iconic"), SWCH ("ic", Opt_iconic, NULL),	/* short form */
  STRG (rs_name, NULL, "name", "string", "client instance, icon, and title strings"),
  STRG (rs_title, "title", "title", "string", "title name for window"),
  STRG (rs_title, NULL, "T", NULL, NULL),	/* short form */
  STRG (rs_iconName, "iconName", "n", "string","icon name for window"),
#ifndef NO_CURSORCOLOR
  STRG (rs_color[Color_cursor], "cursorColor", "cr", "color","cursor color"),
/* command-line option = resource name */
  RSTRG (rs_color[Color_cursor2], "cursorColor2", "color"),
#endif /* NO_CURSORCOLOR */
  STRG (rs_color[Color_pointer], "pointerColor", "pr", "color","pointer color"),
  STRG (rs_borderWidth, "borderWidth", "bw", "number","width of border"),
//  STRG (rs_configFile, "configFile", "cf", "file","configuration file"),
  STRG (rs_color[Color_border], "borderColor", "bd", "color","border color"),
  BOOL (rs_loginShell, "loginShell", "ls", Opt_loginShell,"login shell"), 
  BOOL (rs_scrollBar, "scrollBar", "sb",Opt_scrollBar, "scrollbar"),
  BOOL (rs_scrollBar_right, "scrollBar_right", "sr",Opt_scrollBar_right, "scrollbar right"),
  BOOL (rs_scrollBar_floating, "scrollBar_floating", "st",Opt_scrollBar_floating,
    "scrollbar without a trough"),
  BOOL (rs_scrollTtyOutputInh, "scrollTtyOutput", "si",Opt_scrollTtyOutputInh,
    "scroll-on-tty-output inhibit"),
  BOOL (rs_scrollKeypress, "scrollTtyKeypress", "sk",Opt_scrollKeypress, "scroll-on-keypress"),
  STRG (rs_minBufferWidth, "minBufferWidth", "mbw", "number",
    "minimum number of columns stored in buffer"),
  STRG (rs_saveLines, "saveLines", "sl", "number","number of scrolled lines to save"),
  BOOL (rs_utmpInhibit, "utmpInhibit", "ut", Opt_utmpInhibit,"utmp inhibit"), 
  BOOL (rs_visualBell, "visualBell","vb", Opt_visualBell,"visual bell"),
#ifndef NO_MAPALERT
# ifdef MAPALERT_OPTION
  BOOL (rs_mapAlert, "mapAlert", NULL, Opt_mapAlert, NULL),
# endif
#endif
#ifdef META8_OPTION
  BOOL (rs_meta8, "meta8", NULL, Opt_meta8, NULL),
  RSTRG (rs_modifier, "modifier", "string"),
#endif
#ifndef NO_BACKSPACE_KEY
  RSTRG (rs_backspace_key, "backspacekey", "string"),
#endif
#ifndef NO_DELETE_KEY
  RSTRG (rs_delete_key, "deletekey", "string"),
#endif
#ifdef PRINTPIPE
  RSTRG (rs_print_pipe, "print-pipe", "string"),
#endif
#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
  RSTRG (rs_bigfont_key, "bigfont_key", "keysym"),
  RSTRG (rs_smallfont_key, "smallfont_key", "keysym"),
#endif
#ifdef CUTCHAR_RESOURCE
  RSTRG (rs_cutchars, "cutchars", "string"),
#endif /* CUTCHAR_RESOURCE */
  SWCH ("C", Opt_console, "intercept console messages"),
  INFO ("e", "command arg ...", "command to execute"),
//  STRG (rs_shading,".VT.Shading","sh2", "%","Shading percentage")
};

#undef INFO
#undef STRG
#undef RSTRG
#undef SWCH
#undef BOOL

#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT)
static const char *transparencies[MAX_PAGES];
static const char *shadings[MAX_PAGES];
static char *tintings[MAX_PAGES];
static char *pixmaps[MAX_PAGES];
static char *bg_types[MAX_PAGES];
static char *tint_types[MAX_PAGES];
#endif

static char *save_lines[MAX_PAGES];
static char *titles[MAX_PAGES];
static char *rs_console;

/* Brand new resources management,
 * experimental !
 */
struct s_resources_list {
  const int multiple;    /* indicate if there's more than 1 resource for that class */
  const char *resource;  /* resource name */
  const char *class;     /* resource class */
  const char *kw;        /* type of resource : string, color, ... */
  const char *option;    /* command line option */
  int flag;              /* flag for switch option */
  const char **data_ptr; /* data pointer : could be scalar or table */
  const char *help;      /* help message for lost soul */
} resources_list [] = {
  {0,NULL,NULL,"string","display",0,&display_name,"X server to contact"},
  {0,NULL,NULL,"string","d",0,&display_name,"X server to contact"},
#ifdef TRANSPARENT
  {0,APL_NAME".vt%d.transparent",APL_CLASS".Vt.Transparent",NULL,"tr",Opt_transparent,&rs_transparent,"transparent background"},
  {0,APL_NAME".transpscrollbar",APL_CLASS".Transpscrollbar",NULL,"trsb",Opt_transparent_sb,&rs_transparent_sb,"transparent scrollbar"},
#endif
#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT)
  {1,APL_NAME".vt%d.tinting",APL_CLASS".Vt.Tinting","color","tint",0,tintings,"tinted transparency color"},
  {1,APL_NAME".vt%d.shading",APL_CLASS".Vt.Shading","number","sh",0,shadings,"make transparent background x% darker"},
  {1,APL_NAME".vt%d.bgType",APL_CLASS".Vt.BgType","bgtype","bgtype",0,bg_types,"transformation type for background pixmap"},
  //{1,APL_NAME".vt%d.tintingType",APL_CLASS".Vt.TintingType","tinttype",0,tint_types,"function applied for background tinting"},
  {0,APL_NAME".vt%d.tintingType",APL_CLASS".Vt.TintingType","toto","tinttype",0,&rs_tintType,"function applied for background tinting"},
#endif
  {0,APL_NAME".textType",APL_CLASS".TextType","toto","txttype",0,&rs_textType,"function applied for text drawing"},
#ifdef BACKGROUND_IMAGE
  {1,APL_NAME".vt%d.pixmap",APL_CLASS".Vt.Pixmap","file[;geometry]","pixmap",0,pixmaps,"pixmap background"},
#endif
  {0,APL_NAME".name",APL_CLASS".Name","string","name",0,&rs_name,"client instance, icon, and title strings"},
  {0,APL_NAME".iconName",APL_CLASS".IconName","string","n",0,&rs_iconName,"icon name for window"},
  {0,NULL,NULL,NULL,"iconic",Opt_iconic,&rs_iconic,"start iconic"},
  {1,APL_NAME".vt%d.title",APL_CLASS".Vt.Title","string",NULL,0,titles,"title name for notebooks"},
  {1,APL_NAME".vt%d.saveLines",APL_CLASS".Vt.SaveLines","number","sl",0,save_lines,"number of scrolled lines to save"},
  {0,APL_NAME".termName",APL_CLASS".TermName","string","tn",0,&rs_term_name,"value of the TERM environment variable"},
  {0,APL_NAME".geometry",APL_CLASS".Geometry","geometry","geometry",0,&rs_geometry,"size (in characters) and position"},
  {0,NULL,NULL,"geometry","g",0,&rs_geometry,"size (in characters) and position"},
  {0,APL_NAME".reverseVideo",APL_CLASS".ReverseVideo",NULL,"rv",Opt_reverseVideo,&rs_reverseVideo,"reverse video"},
  {0,APL_NAME".foreground",APL_CLASS".Foreground","color","fg",0,&rs_color[Color_fg],"foreground color"},
  {0,APL_NAME".background",APL_CLASS".Background","color","bg",0,&rs_color[Color_bg],"background color"},
  {0,APL_NAME".loginShell",APL_CLASS".LoginShell",NULL,"ls",Opt_loginShell,&rs_loginShell,"login shell"},
  {0,APL_NAME".borderWidth",APL_CLASS".BorderWidth","number","bw",0,&rs_borderWidth,"border width"},
  {0,APL_NAME".scrollbar",APL_CLASS".Scrollbar",NULL,"sb",Opt_scrollBar,&rs_scrollBar,"scroll bar"},
  {0,APL_NAME".scrollbarRight",APL_CLASS".ScrollbarRight",NULL,"sr",Opt_scrollBar_right,&rs_scrollBar_right,"scrollbar on right"},
  {0,APL_NAME".scrollbarFloating",APL_CLASS".ScrollbarFloating",NULL,"st",Opt_scrollBar_floating,&rs_scrollBar_floating,"scrollbar without a trough"},
  {0,APL_NAME".scrollbarTtyOutput",APL_CLASS".ScrollbarTtyOutput",NULL,"si",Opt_scrollTtyOutputInh,&rs_scrollTtyOutputInh,"scroll-on-tty-output inhibit"},
  {0,APL_NAME".scrollbarTtyKeypress",APL_CLASS".ScrollbarTtyKeypress",NULL,"sk",Opt_scrollKeypress,&rs_scrollKeypress,"scroll-on-keypress"},
  {0,APL_NAME".minBufferWidth",APL_CLASS".MinBufferWidth","number","mbw",0,&rs_minBufferWidth,"minimum number of columns stored in buffer"},
  {0,APL_NAME".utmpInhibit",APL_CLASS".UtmpInhibit",NULL,"ut",Opt_utmpInhibit,&rs_utmpInhibit,"utmp inhibit"},
  {0,APL_NAME".visualBell",APL_CLASS".VisualBell",NULL,"vb",Opt_visualBell,&rs_visualBell,"visual bell"},
  {0,APL_NAME".borderColor",APL_CLASS".BorderColor","color","bd",0,&rs_color[Color_border],"border color"},
  {0,APL_NAME".nbTerm",APL_CLASS".NbTerm","number","nt",0,&rs_term_num,"numbers of tab at startup"},
#ifdef OFF_FOCUS_FADING
  {0,APL_NAME".fading",APL_CLASS".Fading","number","fade",0,&rs_fade,"make colors x% darker when aterm is loosing focus"},
#endif
  {0,APL_NAME".color0",APL_CLASS".Color0","color",NULL,0,&rs_color[minCOLOR + 0],NULL},
  {0,APL_NAME".color1",APL_CLASS".Color1","color",NULL,0,&rs_color[minCOLOR + 1],NULL},
  {0,APL_NAME".color2",APL_CLASS".Color2","color",NULL,0,&rs_color[minCOLOR + 2],NULL},
  {0,APL_NAME".color3",APL_CLASS".Color3","color",NULL,0,&rs_color[minCOLOR + 3],NULL},
  {0,APL_NAME".color4",APL_CLASS".Color4","color",NULL,0,&rs_color[minCOLOR + 4],NULL},
  {0,APL_NAME".color5",APL_CLASS".Color5","color",NULL,0,&rs_color[minCOLOR + 5],NULL},
  {0,APL_NAME".color6",APL_CLASS".Color6","color",NULL,0,&rs_color[minCOLOR + 6],NULL},
  {0,APL_NAME".color7",APL_CLASS".Color7","color",NULL,0,&rs_color[minCOLOR + 7],NULL},
#ifndef NO_BRIGHTCOLOR
  {0,APL_NAME".color8",APL_CLASS".Color8","color",NULL,0,&rs_color[minBrightCOLOR + 0],NULL},
  {0,APL_NAME".color9",APL_CLASS".Color9","color",NULL,0,&rs_color[minBrightCOLOR + 1],NULL},
  {0,APL_NAME".color10",APL_CLASS".Color10","color",NULL,0,&rs_color[minBrightCOLOR + 2],NULL},
  {0,APL_NAME".color11",APL_CLASS".Color11","color",NULL,0,&rs_color[minBrightCOLOR + 3],NULL},
  {0,APL_NAME".color12",APL_CLASS".Color12","color",NULL,0,&rs_color[minBrightCOLOR + 4],NULL},
  {0,APL_NAME".color13",APL_CLASS".Color13","color",NULL,0,&rs_color[minBrightCOLOR + 5],NULL},
  {0,APL_NAME".color14",APL_CLASS".Color14","color",NULL,0,&rs_color[minBrightCOLOR + 6],NULL},
  {0,APL_NAME".color15",APL_CLASS".Color15","color",NULL,0,&rs_color[minBrightCOLOR + 7],NULL},
#endif /* NO_BRIGHTCOLOR */
#ifndef NO_BOLDUNDERLINE
  {0,APL_NAME".colorBD",APL_CLASS".ColorBD","color",NULL,0,&rs_color[Color_BD],NULL},
  {0,APL_NAME".colorUL",APL_CLASS".ColorUL","color",NULL,0,&rs_color[Color_UL],NULL},
#endif /* NO_BOLDUNDERLINE */
#ifdef KEEP_SCROLLCOLOR
  {0,APL_NAME".scrollcolor",APL_CLASS".ScrollColor","color",NULL,0,&rs_color[Color_scroll],NULL},
  {0,APL_NAME".troughcolor",APL_CLASS".TroughColor","color",NULL,0,&rs_color[Color_trough],NULL},
#endif /* KEEP_SCROLLCOLOR */
#if defined (BACKGROUND_IMAGE) || (MENUBAR_MAX)
  {0,APL_NAME".path",APL_CLASS".Path","directory",NULL,0,&rs_path,"search path for pixmaps"},
#endif /* defined (BACKGROUND_IMAGE) || (MENUBAR_MAX) */
  {0,APL_NAME".fontName",APL_CLASS".FontName","font","font",0,&rs_font[0],"normal text font"},
#ifndef NO_BOLDFONT
  {0,APL_NAME".boldFont",APL_CLASS".BoldFont","font","fb",0,&rs_boldFont,"bold text font"},
#endif
#if NFONTS > 1
  {0,APL_NAME".font1",APL_CLASS".Font1","font",NULL,0,&rs_font[1],NULL},
#endif
#if NFONTS > 2
  {0,APL_NAME".font2",APL_CLASS".Font2","font",NULL,0,&rs_font[2],NULL},
#endif
#if NFONTS > 3
  {0,APL_NAME".font3",APL_CLASS".Font3","font",NULL,0,&rs_font[3],NULL},
#endif
#if NFONTS > 4
  {0,APL_NAME".font4",APL_CLASS".Font4","font",NULL,0,&rs_font[4],NULL},
#endif
#if NFONTS > 5
  {0,APL_NAME".font5",APL_CLASS".Font5","font",NULL,0,&rs_font[5],NULL},
#endif
#if NFONTS > 6
  {0,APL_NAME".font6",APL_CLASS".Font6","font",NULL,0,&rs_font[6],NULL},
#endif
#if NFONTS > 7
  {0,APL_NAME".font7",APL_CLASS".Font7","font",NULL,0,&rs_font[7],NULL},
#endif
#ifdef MULTICHAR_SET
  {0,APL_NAME".mfont",APL_CLASS".MFont","font",NULL,0,&rs_mfont[0],NULL},
# if NFONTS > 1
  {0,APL_NAME".mfont1",APL_CLASS".MFont1","font",NULL,0,&rs_mfont[1],NULL},
# endif
# if NFONTS > 2
  {0,APL_NAME".mfont2",APL_CLASS".MFont2","font",NULL,0,&rs_mfont[2],NULL},
# endif
# if NFONTS > 3
  {0,APL_NAME".mfont3",APL_CLASS".MFont3","font",NULL,0,&rs_mfont[3],NULL},
# endif
# if NFONTS > 4
  {0,APL_NAME".mfont4",APL_CLASS".MFont4","font",NULL,0,&rs_mfont[4],NULL},
# endif
# if NFONTS > 5
  {0,APL_NAME".mfont5",APL_CLASS".MFont5","font",NULL,0,&rs_mfont[5],NULL},
# endif
# if NFONTS > 6
  {0,APL_NAME".mfont6",APL_CLASS".MFont6","font",NULL,0,&rs_mfont[6],NULL},
# endif
# if NFONTS > 7
  {0,APL_NAME".mfont7",APL_CLASS".MFont7","font",NULL,0,&rs_mfont[7],NULL},
# endif
#endif /* MULTICHAR_SET */
  {0,APL_NAME".pointerColor",APL_CLASS".PointerColor",NULL,NULL,0,&rs_color[Color_pointer],NULL},
#ifndef NO_CURSORCOLOR
  {0,APL_NAME".cursorColor",APL_CLASS".CursorColor","color",NULL,0,&rs_color[Color_cursor],NULL},
  {0,APL_NAME".cursorColor2",APL_CLASS".CursorColor2","color",NULL,0,&rs_color[Color_cursor2],NULL},
#endif /* NO_CURSORCOLOR */
#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
  {0,APL_NAME".bigfont_key",APL_CLASS".Bigfont_key","font",NULL,0,&rs_bigfont_key,NULL},
  {0,APL_NAME".smallfont_key",APL_CLASS".Smallfont_key","font",NULL,0,&rs_smallfont_key,NULL},
#endif
#ifdef CUTCHAR_RESOURCE
  {0,APL_NAME".cutchars",APL_CLASS".Cutchars","string",NULL,0,&rs_cutchars,NULL},
#endif /* CUTCHAR_RESOURCE */
#ifdef META8_OPTION
  {0,APL_NAME".meta8",APL_CLASS".Meta8",NULL,NULL,0,&rs_meta8,NULL},
  {0,APL_NAME".modifier",APL_CLASS".Modifier",NULL,NULL,0,&rs_modifier,NULL},
#endif
#ifndef NO_BACKSPACE_KEY
  {0,APL_NAME".backspacekey",APL_CLASS".Backspacekey",NULL,NULL,0,&rs_backspace_key,NULL},
#endif
#ifndef NO_DELETE_KEY
  {0,APL_NAME".deletekey",APL_CLASS".Deletekey",NULL,NULL,0,&rs_delete_key,NULL},
#endif
#ifndef NO_MAPALERT
# ifdef MAPALERT_OPTION
  {0,APL_NAME".mapAlert",APL_CLASS".MapAlert",NULL,NULL,Opt_mapAlert,&rs_mapAlert,NULL},
# endif
#endif
#ifdef MULTICHAR_SET
  {0,APL_NAME".multichar_encoding",APL_CLASS".Multichar_encoding","eucj|sjis|big5","km",0,&rs_multichar_encoding,NULL},
#endif /* MULTICHAR_SET */
#ifdef GREEK_SUPPORT
  {0,APL_NAME".greek_keyboard",APL_CLASS".Greek_keyboard","iso | ibm","grk",0,&rs_greek_keyboard,NULL},
#endif
#ifdef PRINTPIPE
  {0,APL_NAME".Print-pipe",APL_CLASS".print-pipe",NULL,"string",0,&rs_print_pipe,NULL},
#endif
  {0,NULL,NULL,"file","cf",0,&rs_config_file,"resource configuration file"},
  {0,NULL,NULL,NULL,"C",Opt_console,&rs_console,"intercept console messages"}
};

/* for debug purpose only !
 */
#if DEBUG_XRSCR_ALEXIS

void display_options() {
  int i,l;
  
  fprintf(stderr,"Display option:\n");
  l = sizeof(resources_list) / sizeof(struct s_resources_list);
  for(i=0; i < l; i++) {
    if( resources_list[i].option ) {
      printf("%s",resources_list[i].option);
      if( resources_list[i].multiple ) {
        printf(": %s",resources_list[i].data_ptr ? resources_list[i].data_ptr[0] : "");
      } else {
        printf(": %s",resources_list[i].data_ptr ? *(resources_list[i].data_ptr) : "");
      }
      printf("\n");
    }
  }
}

#endif

/* my usage function
 */
static void howto_use(int type) {
  int i,l;
  
	fprintf (stderr, "\nUsage v%s :", VERSION);
	list_options ();
	fprintf (stderr, "\n%s", APL_NAME);

  l = sizeof(resources_list) / sizeof(struct s_resources_list);
  switch (type) {
    case 0 : /* brief listing */
      for(i=0; i < l; i++) {
        if( resources_list[i].option ) {
          fprintf(stderr," [-");
          if( resources_list[i].flag) {
            fprintf(stderr,"/+");
          }
          fprintf(stderr,"%s]",resources_list[i].option);
        }
      }
      fprintf(stderr,"\n");
      break;
    case 1 : /* long listening */
      fprintf(stderr," [options]\nwhere options might be :\n");
      for(i=0; i < l; i++) {
        int t,u;
        if( resources_list[i].option ) {
          t = strlen(resources_list[i].option);
          fprintf(stderr,"-");
          if( resources_list[i].flag) {
            fprintf(stderr,"/+");
          }
          fprintf(stderr,"%s",resources_list[i].option);
          u=1;
          if( resources_list[i].kw ) {
            fprintf(stderr," %s",resources_list[i].kw);
            u=strlen(resources_list[i].kw);
            
          }
          fprintf(stderr,"%-*s",25-u-t," ");
          if( resources_list[i].help ) {
            fprintf(stderr," %s",resources_list[i].help);
          }
          fprintf(stderr,"\n");
        }
      }
      break;
    case 2 : /* resources listening */
      fprintf(stderr,"\nwhere resources might be :\n");
      for(i=0; i < l; i++) {
        if( resources_list[i].class ) {
          fprintf(stderr,"%-30s",resources_list[i].class);
          if( resources_list[i].help ) {
            fprintf(stderr," %50s",resources_list[i].help);
          }
          fprintf(stderr,"\n");
        }
      }
      break;
  }
  fprintf(stderr,"\n");
  exit(EXIT_FAILURE);
}


void display_resources() {
  
}

void XrmValueToString(XrmValue *value,char **string) {
  *string = (char *) malloc(sizeof(char)*(value->size+1));
  memcpy(*string,value->addr,value->size + 1);
  (*string)[value->size] = 0;
}

void load_resources_from_db(XrmDatabase db) {
  XrmValue value;
  char resource[100];
  char *type,*val;
  int i,j,l;
  
  l = sizeof(resources_list) / sizeof(struct s_resources_list);
  for(i=0; i < l; i++) {
    if( resources_list[i].multiple ) {
      for(j=0; j < MAX_PAGES; j++) {
        if( resources_list[i].resource && !resources_list[i].data_ptr[j]) {
          sprintf(resource,resources_list[i].resource,j);
          if( XrmGetResource(db,resource,resources_list[i].class,&type,&value) ) {
            XrmValueToString(&value,&val);
            resources_list[i].data_ptr[j] = val;
          }
        }
      }
    } else {
      if( resources_list[i].resource && !*(resources_list[i].data_ptr) && XrmGetResource(db,resources_list[i].resource,resources_list[i].class,&type,&value) ) {
        XrmValueToString(&value,&val);
//        printf("%s : %s\n",resource,val);
        *(resources_list[i].data_ptr) = val;
        if( resources_list[i].flag ) {
          if( ! strcasecmp(*(resources_list[i].data_ptr),"True") ) {
            Options |= resources_list[i].flag;
          } else {
            Options &= ~(resources_list[i].flag);
          }
        }
      }
    }
  }
  
}


void get_resources() {
  XrmDatabase db=NULL,tmp_db;
  char *home;
  char *home_Xdefaults=NULL;
  char *home_Xresources=NULL;
  
  /* init files */
  if ((home = getenv ("HOME")) != NULL) {
    
    int len_home = strlen(home);
    home_Xdefaults = malloc(sizeof(char)*(len_home+12)); /* 12 = strlen("/.Xdefaults")+1 */
    home_Xresources = malloc(sizeof(char)*(len_home+13)); /* 13 = strlen("/.Xresources")+1 */
    
    strcpy(home_Xdefaults,home);
    strncpy(home_Xdefaults+len_home,"/.Xdefaults",11);
    home_Xdefaults[len_home+11] = 0;
    
    strcpy(home_Xresources,home);
    strncpy(home_Xresources+len_home,"/.Xresources",12);
    home_Xresources[len_home+12] = 0;

  }
  
  XrmInitialize();
  if( rs_config_file ) {
    db = XrmGetFileDatabase(rs_config_file);
    if( ! db ) {
      print_error("Cannot open %s",rs_config_file);
    }
  }
  tmp_db = XrmGetFileDatabase(home_Xdefaults);
  if( !tmp_db ) {
    /* cannot read ~/.Xdefaults
     * so try ~/.Xresources
     */
    tmp_db = XrmGetFileDatabase(home_Xresources);
  }
  free(home_Xdefaults);
  free(home_Xresources);

  if( tmp_db ) {
    XrmCombineDatabase(tmp_db,&db,False);
    //XrmDestroyDatabase(tmp_db);
  }
#ifdef XAPPLOADDIR
  tmp_db = XrmGetFileDatabase(XAPPLOADDIR "/" APL_CLASS);
  if( tmp_db) {
    XrmCombineDatabase(tmp_db,&db,False);
    //XrmDestroyDatabase(tmp_db);
  }
#endif
  if( !db ) {
    /* we'll use defaults values since no resource
     * file were found
     */
    return;
  }
  
  /* start of resources reading */
  load_resources_from_db(db);
  //XrmDestroyDatabase(db);
  
/*
 * even without resources, at least do this setup for command-line
 * options and command-line long options
 */
#ifdef MULTICHAR_SET
	set_multichar_encoding (rs_multichar_encoding);
#endif
#ifdef GREEK_SUPPORT
/* this could be a function in grkelot.c */
/* void set_greek_keyboard (const char * str); */
	if (rs_greek_keyboard)
	{
		if (!strcmp (rs_greek_keyboard, "iso"))
			greek_setmode (GREEK_ELOT928);	/* former -grk9 */
		else if (!strcmp (rs_greek_keyboard, "ibm"))
			greek_setmode (GREEK_IBM437);	/* former -grk4 */
	}
#endif /* GREEK_SUPPORT */

#define to_keysym(pks,str) do { KeySym sym;\
if (str && ((sym = XStringToKeysym(str)) != 0)) *pks = sym; } while (0)

#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
	to_keysym (&ks_bigfont, rs_bigfont_key);
	to_keysym (&ks_smallfont, rs_smallfont_key);
#endif
#undef to_keysym
}


/* load resources from XRM, .Xdefaults and .Xresources
 *
 */
void load_resources() {
}

void put_resources() {
}

/* my parse command-line function
 */
void parse_command_line(int argc, char *argv[]) {
  int i,j,k;
  int l;
  
  l = sizeof(resources_list) / sizeof(struct s_resources_list);
  DXR_ALEXIS(2,(stderr,"Command line options :\n"));
  for(i=1; i < argc; i++) {
    char *opt=argv[i];
    int long_opt=0;
    int opt_on=1;
    
    DXR_ALEXIS(2,(stderr,"option: %s\n",argv[i]));
    if( *opt == '-') {
      if(*++opt == '-') {
        long_opt=1;
        opt++;
      }
    } else if( *opt++ == '+' ) {
      opt_on=0;
    } else {
      print_error("Bad option %s\n",argv[i]);
      howto_use(0);
    }
    if( ! strcmp(opt,"help") ) {
      howto_use (long_opt ? 2 : 1);
    }
    if( ! strcmp(opt,"h") ) {
      howto_use(0);
    }
    if( ! strcmp(opt,"version") ) {
      version (long_opt ? 2 : 1);
    }
    if( ! strcmp(opt,"V") ) {
      version(0);
    }
    
    for(j=0; j < l; j++) {
      if( resources_list[j].option && !strcmp(opt,resources_list[j].option) ) {
        break;
      }
    }
    if( j < l) {
      DXR_ALEXIS(2,(stderr,"opt : %s",opt));
      if( resources_list[j].flag ) {
        
        /* this is a flag */
        DXR_ALEXIS(2,(stderr,"(switch) "));
        if( resources_list[j].multiple ) {
          for(k=0; k < MAX_PAGES; k++) {
            resources_list[j].data_ptr[k] = malloc(sizeof(char)*2);
            sprintf(resources_list[j].data_ptr[k],"%c",(opt_on ? 1 : 0));
          }
        } else {
          if( opt_on ) {
            Options |= resources_list[j].flag;
            *(resources_list[j].data_ptr) = "True";
            
          } else {
            Options &= ~(resources_list[j].flag);
            *(resources_list[j].data_ptr) = "False";
          }
        }
        
      } else {
        /* argv[i+1] should be an option */
        if( i+1 < argc) {
          if( ! resources_list[j].multiple ) {
            *(resources_list[j].data_ptr) = strdup(argv[++i]);
            DXR_ALEXIS(2,(stderr,": %s",*(resources_list[j].data_ptr)));
          } else {
            for(k=0; k < MAX_PAGES; k++) {
              resources_list[j].data_ptr[k] = strdup(argv[i+1]); /* points to argv -> do not free it */
            }
            i++;
          }
        } else {
          print_error("Missing argument for %s\n",opt);
          usage(0);
        }
      }
      DXR_ALEXIS(2,(stderr,"\n"));
    }
#ifdef KEYSYM_RESOURCE
		else if (Str_match (opt, "keysym.")) {
			char *str = argv[++i];

			/*
			 * '7' is strlen("keysym.")
			 */
			if (str != NULL)
				parse_keysym (opt + 7, str);
		}
#endif
    else {
      print_error("Unknown option %s\n",opt);
      howto_use(0);
    }
  }
//  display_options();
}

void set_vt_resources() {
  int i;
  
  for(i=0; i < MAX_PAGES; i++) {
    if( titles[i] ) {
      TermWin.vts[i].tab_title = titles[i];
    } else {
      TermWin.vts[i].tab_title = strdup(rs_default_tab_title); /* this is the default value */
    }
    if( save_lines[i] ) {
      TermWin.vts[i].saveLines = atoi(save_lines[i]);
      free(save_lines[i]);
    } else {
      TermWin.vts[i].saveLines = SAVELINES; /* this is the default value */
    }
    TermWin.vts[i].min_bcol = 1;  /* this is the default value */
  }
}

#if defined(BACKGROUND_IMAGE) || defined(TRANSPARENT)
void set_background() {
  int i;
  
  for(i=0; i < MAX_PAGES; i++) {
    if( shadings[i] ) {
      TermWin.vts[i].bg.Shading.shading = atoi(shadings[i]);
      free(shadings[i]);
    }
    if( tintings[i] ) {
      TermWin.vts[i].bg.color = tintings[i];
    }
    if( transparencies[i] ) {
      TermWin.vts[i].bg.transparent = 
        (transparencies[i][0] == '1' || !strcasecmp("true",transparencies[i]) )? Opt_transparent : ~Opt_transparent;
    }
    if( tint_types[i] ) {
      // to do ...
      //TermWin.vts[i].bg.trgType = ParseGCType(tint_types[i],GXand);
    }

    TermWin.vts[i].bg.trgType = BGT_Tile;

#ifdef TRANSPARENT
      if ((Options & Opt_transparent) && (!bg_types[i] || !strcasecmp("None",bg_types[i])) ) {
  //if( (Options & TermWin.vts[0].bg.transparent) && !type) {
        if (TermWin.vts[i].bg.Shading.shading != 100) {
          TermWin.vts[i].bg.trgType = BGT_Cut;
        }	else {
          TermWin.vts[i].bg.trgType = BGT_None;
        }
      }
#endif
    if( bg_types[i] ) {
      if (strcmp(bg_types[i], BGT_CENTER) == 0)
        TermWin.vts[i].bg.trgType = BGT_Center;
      else if (strcmp(bg_types[i], BGT_SCALE) == 0)
        TermWin.vts[i].bg.trgType = BGT_Scale;
      else if (strcmp(bg_types[i], BGT_SCALEH) == 0)
        TermWin.vts[i].bg.trgType = BGT_ScaleH;
      else if (strcmp(bg_types[i], BGT_SCALEV) == 0)
        TermWin.vts[i].bg.trgType = BGT_ScaleV;
      else if (strcmp(bg_types[i], BGT_NO_TILE) == 0)
        TermWin.vts[i].bg.trgType = BGT_NoTile;
      else if (strcmp(bg_types[i], BGT_NO_TILE_H) == 0)
        TermWin.vts[i].bg.trgType = BGT_NoTileH;
      else if (strcmp(bg_types[i], BGT_NO_TILE_V) == 0)
        TermWin.vts[i].bg.trgType = BGT_NoTileV;
      else if (strcmp(bg_types[i], BGT_CUT) == 0)
        TermWin.vts[i].bg.trgType = BGT_Cut;
    }
    if( pixmaps[i] ) {
      TermWin.vts[i].bg.pixmap_file = pixmaps[i];
    }
  }
}
#endif

extern const char *def_colorName[];

/* list_options 
 */
void list_options ()
{
#define INDENT 30
	fprintf (stderr, "(");
#ifdef BACKGROUND_IMAGE
	fprintf (stderr, "background image,");
#endif
#ifdef HAVE_LIBXPM
	fprintf (stderr, "XPM,");
#endif
#ifdef USE_LIBASIMAGE
#ifdef JPEG
	fprintf (stderr, "JPEG,");
#endif
#ifdef PNG
	fprintf (stderr, "PNG,");
#endif
	fprintf (stderr, "AfterStep integration,");
#ifdef _MYSTYLE_
	fprintf (stderr, "AfterStep MyStyles,");
#else
	fprintf (stderr, "no AfterStep MyStyles,");
#endif
#endif
#ifdef UTMP_SUPPORT
	fprintf (stderr, "utmp,");
#endif
#ifdef MENUBAR
	fprintf (stderr, "menubar,");
#endif
#ifdef KANJI
	fprintf (stderr, "Kanji,");
#endif
#ifdef ZH
	fprintf (stderr, "Chinese,");
#endif
#ifdef THAI
	fprintf (stderr, "Thai,");
#endif
#ifdef XTERM_SCROLLBAR
	fprintf (stderr, "XTerm-scrollbar,");
#endif
#ifdef GREEK_SUPPORT
	fprintf (stderr, "Greek,");
#endif
#ifdef NO_BACKSPACE_KEY
	fprintf (stderr, "no backspace,");
#endif
#ifdef NO_DELETE_KEY
	fprintf (stderr, "no delete,");
#endif
#ifdef TRANSPARENT
	fprintf (stderr, "transparency,");
#else
	fprintf (stderr, "no transparency,");
# endif
#ifdef OFF_FOCUS_FADING
	fprintf (stderr, "fading,");
#else
	fprintf (stderr, "no fading,");
# endif
#ifdef NEXT_SCROLLBAR
	fprintf (stderr, "NeXT scrollbar,");
# endif
#ifdef NO_RESOURCES
	fprintf (stderr, "NoResources");
#else
# ifdef USE_XGETDEFAULT
	fprintf (stderr, "XGetDefaults");
# else
	fprintf (stderr, ".Xdefaults");
# endif
#endif

	fprintf (stderr, ")");

}

/* version
 */
void version (int type) {
	switch (type)
	{
	case 0:		/* brief listing */
		fprintf (stderr, "%s version %s\n", APL_NAME, VERSION);
		break;
	case 1:		/* full command-line listing */
		fprintf (stderr, "%s version %s from %s\n", APL_NAME, VERSION,DATE);
		list_options ();
		fprintf (stderr, "\n");
		break;
	case 2:		/* full resource listing */
		fprintf (stderr, "%s\n", VERSION);
		break;
	}
	exit (EXIT_FAILURE);
}

/* usage
 */
void usage (int type) {
	int i, col;

	fprintf (stderr, "\nUsage v%s :", VERSION);
	list_options ();
	fprintf (stderr, "\n%s", APL_NAME);

	switch (type)
	{
	case 0:		/* brief listing */
		fprintf (stderr, " [-help][-V]\n");
		col = 3;
		for (i = 0; i < optList_size (); i++)
		{
			if (optList[i].desc != NULL)
			{
				int len = 2;

				if (!optList_isBool (i))
				{
					len = optList_strlen (i);
					if (len > 0)
						len++;	/* account for space */
				}

				len += 4 + strlen (optList[i].opt);

				col += len;
				if (col > 79)
				{	/* assume regular width */
					fprintf (stderr, "\n");
					col = 3 + len;
				}
				fprintf (stderr, " [-");
				if (optList_isBool (i))
					fprintf (stderr, "/+");

				fprintf (stderr, "%s", optList[i].opt);
				if (optList_strlen (i))
					fprintf (stderr, " %s]",
						 optList[i].arg);
				else
					fprintf (stderr, "]");
			}
		}
		fprintf (stderr, "\n\n");
		break;

	case 1:		/* full command-line listing */
		fprintf (stderr,
			 " [options] [-e command args]\n\n"
			 "where options include:\n");

		for (i = 0; i < optList_size (); i++)
		{
			if (optList[i].desc != NULL)
				fprintf (stderr, "    %s%s %-*s%s%s\n",
					 (optList_isBool (i) ? "-/+" : "-"),
					 optList[i].opt,
					 (INDENT - strlen (optList[i].opt) + (optList_isBool (i) ? 0 : 2)),
					 (optList[i].arg ? optList[i].arg : ""),
					 (optList_isBool (i) ? "turn on/off " : ""), optList[i].desc);
		}
		fprintf (stderr,
			 "\n    --help to list long-options\n    --version for the version information\n\n");
		break;

	case 2:		/* full resource listing */
		fprintf (stderr,
			 " [options] [-e command args]\n\n"
			 "where resources (long-options) include:\n");

		for (i = 0; i < optList_size (); i++)
			if (optList[i].kw != NULL)
				fprintf (stderr, "    %s: %*s\n",
					 optList[i].kw,
					 (INDENT - strlen (optList[i].kw)),
					 (optList_isBool (i) ? "boolean" :
					  optList[i].arg));

#ifdef KEYSYM_RESOURCE
		fprintf (stderr, "    " "keysym.sym" ": %*s\n",
			 (INDENT - strlen ("keysym.sym")), "keysym");
#endif
		fprintf (stderr,
			 "\n    -help to list options\n    -version for the version information with options list\n\n");
		break;
	}
	exit (EXIT_FAILURE);
}


/* get command-line options before getting resources
 */
void get_options (int argc, char *argv[]) {
	int i, bad_option = 0;
	static const char *const On = "ON";
	static const char *const Off = "OFF";

	for (i = 1; i < argc; i++)
	{
		int entry, longopt = 0;
		const char *flag;
		char *opt = argv[i];

#ifdef DEBUG_RESOURCES
		fprintf (stderr, "argv[%d] = %s: ", i, argv[i]);
#endif
		if (*opt == '-')
		{
			flag = On;
			if (*++opt == '-')
				longopt = *opt++;	/* long option */
		}
		else if (*opt == '+')
		{
			flag = Off;
			if (*++opt == '+')
				longopt = *opt++;	/* long option */
		}
		else
		{
			bad_option = 1;
			print_error ("bad option \"%s\"", opt);
			continue;
		}

		if (!strcmp (opt, "help"))
			usage (longopt ? 2 : 1);
		if (!strcmp (opt, "h"))
			usage (0);
		if (!strcmp (opt, "version"))
			version (longopt ? 2 : 1);
		if (!strcmp (opt, "V"))
			version (0);

		/* feature: always try to match long-options */
		for (entry = 0; entry < optList_size (); entry++)
		{
			if ( (optList[entry].kw && !strcmp (opt, optList[entry].kw)) || 
        (!longopt && optList[entry].opt && !strcmp(opt,optList[entry].opt)) )
				break;
		}

		if (entry < optList_size ())
		{
			if (optList_strlen (entry))
			{	/* string value */
				char *str = argv[++i];

#ifdef DEBUG_RESOURCES
				fprintf (stderr, "string (%s,%s) = ",
					 optList[entry].opt ? optList[entry].opt : "nil",
					 optList[entry].kw ? optList[entry].kw : "nil");
#endif
				if (flag == On && str && optList[entry].dp)
				{
#ifdef DEBUG_RESOURCES
					fprintf (stderr, "\"%s\"\n", str);
#endif
					*(optList[entry].dp) = str;

					/* special cases are handled in main.c:main() to allow
					 * X resources to set these values before we settle for
					 * default values
					 */
				}
#ifdef DEBUG_RESOURCES
				else
					fprintf (stderr, "???\n");
#endif
			}
			else
			{	/* boolean value */
#ifdef DEBUG_RESOURCES
				fprintf (stderr, "boolean (%s,%s) = %s\n",
					 optList[entry].opt,
					 optList[entry].kw, flag);
#endif
				if (flag == On)
					Options |= (optList[entry].flag);
				else
					Options &= ~(optList[entry].flag);

				if (optList[entry].dp)
					*(optList[entry].dp) = flag;
			}
		}
		else
#ifdef KEYSYM_RESOURCE
			/* if (!strncmp (opt, "keysym.", strlen ("keysym."))) */
		if (Str_match (opt, "keysym."))
		{
			char *str = argv[++i];

			/*
			 * '7' is strlen("keysym.")
			 */
			if (str != NULL)
				parse_keysym (opt + 7, str);
		}
		else
#endif
		{
			/* various old-style options, just ignore
			 * Obsolete since about Jan 96,
			 * so they can probably eventually be removed
			 */
			const char *msg = "bad";

			if (longopt)
			{
				opt--;
				bad_option = 1;
			}
			else if (!strcmp (opt, "7") || !strcmp (opt, "8")
#ifdef GREEK_SUPPORT
				 /* obsolete 12 May 1996 (v2.17) */
				 || !Str_match (opt, "grk")
#endif
				)
				msg = "obsolete";
			else
				bad_option = 1;

			print_error ("%s option \"%s\"", msg, --opt);
		}
	}

	if (bad_option)
		usage (0);
}



#ifndef NO_RESOURCES
/*
 * a replacement for strcasecmp() to avoid linking an entire library
 */
int my_strcasecmp (const char *s1, const char *s2) {
	for ( /*nil */ ; (*s1 && *s2); s1++, s2++)
	{
		register int c1 = toupper (*s1);
		register int c2 = toupper (*s2);

		if (c1 != c2)
			return (c1 - c2);
	}
	return (int) (*s1 - *s2);
}

# ifdef KEYSYM_RESOURCE

/*
 * Define key from XrmEnumerateDatabase.
 *   quarks will be something like
 *      "rxvt" "keysym" "0xFF01"
 *   value will be a string
 */
Bool define_key (XrmDatabase * database, XrmBindingList bindings,
  XrmQuarkList quarks, XrmRepresentation * type, XrmValue * value, XPointer closure) {
	int last;

	for (last = 0; quarks[last] != NULLQUARK; last++)	/* look for last quark in list */
		;
	last--;
	parse_keysym (XrmQuarkToString (quarks[last]), (char *) value->addr);
	return False;
}

/*
 * look for something like this (XK_Delete)
 * rxvt*keysym.0xFFFF: "\177"
 *
 * arg will be
 *      NULL for ~/.Xdefaults and
 *      non-NULL for command-line options (need to allocate)
 */
int parse_keysym (char *str, char *arg) {
	char *key_string;
	int n, sym;

	if (arg == NULL)
	{
		if ((n = Str_match (str, "keysym.")) == 0)
			return 0;
		str += n;	/* skip `keysym.' */
	}
/* some scanf() have trouble with a 0x prefix */
	if (isdigit (str[0]))
	{
		if (str[0] == '0' && toupper (str[1]) == 'X')
			str += 2;
		if (arg)
		{
			if (sscanf(str, (strchr (str, ':') ? "%x:" : "%x"),&sym) != 1)
				return -1;
		}
		else
		{
			if (sscanf (str, "%x:", &sym) != 1)
				return -1;

			/* cue to ':', it's there since sscanf() worked */
			str = strchr (str, ':');
			str++;
			arg = Str_trim (str);
			if (arg == NULL)
				return -1;
		}
	}
	else
	{
		/*
		 * convert keysym name to keysym number
		 */
		if (arg == NULL)
		{
			arg = str;

			arg = strchr (str, ':');
			if (arg == NULL)
				return -1;

			*arg++ = '\0';
			arg = Str_trim (arg);
			if (arg == NULL)
				return -1;
		}
		sym = XStringToKeysym (str);

		if (sym == None)
			return -1;
	}

	if (sym < 0xFF00 || sym > 0xFFFF)	/* we only do extended keys */
		return -1;
	sym -= 0xFF00;

	if (KeySym_map[sym] != NULL)	/* already set ? */
		return -1;

	if ((n = strlen (arg)) == 0)
		return -1;

	key_string = MALLOC ((n + 2) * sizeof (char));

	STRCPY (key_string + 1, arg);

	n = Str_escaped (key_string + 1);
	if (n)
	{
		key_string[0] = min (n, 255);
		KeySym_map[sym] = (unsigned char *) key_string;
	}
	else
	{
		FREE (key_string);
		return -1;
	}

	return 1;
}
# endif	/* KEYSYM_RESOURCE */

# ifndef USE_XGETDEFAULT
/* get_xdefaults() 
 *
 * the matching algorithm used for memory-save fake resources
 */

void
get_xdefaults (FILE * stream, const char *name)
{
	unsigned int len;
	char *str, buffer[256];

	if (stream == NULL)
		return;
	len = strlen (name);
	while ((str = fgets (buffer, sizeof (buffer), stream)) != NULL)
	{
		unsigned int entry, n;

		while (*str && isspace (*str))
			str++;	/* leading whitespace */

		if ((str[len] != '*' && str[len] != '.') || (len && strncmp (str, name, len)))
			continue;
		str += (len + 1);	/* skip `name*' or `name.' */

# ifdef KEYSYM_RESOURCE
		if (!parse_keysym (str, NULL))
# endif	/* KEYSYM_RESOURCE */
			for (entry = 0; entry < optList_size (); entry++)
			{
				const char *const kw = optList[entry].kw;

				if (kw == NULL)
					continue;
				n = strlen (kw);
				if (str[n] == ':' && Str_match (str, kw))
				{
					/* skip `keyword:' */
					str += (n + 1);
					str = Str_skip_space (str);
					str = Str_trim (str);
					n = (str ? strlen (str) : 0);
					if (n && *(optList[entry].dp) == NULL)
					{
						/* not already set */
						char *p =	MALLOC ((n +1) *sizeof(char));
						STRCPY (p, str);
						*(optList[entry].dp) = p;
						if (optList_isBool (entry))
						{
							if (!my_strcasecmp(str, "TRUE"))
                Options |= (optList[entry].flag);
							else
								Options &= ~(optList[entry].flag);
						}
					}
					break;
				}
			}
	}
	rewind (stream);
}


# endif	/* ! USE_XGETDEFAULT */

#endif /* NO_RESOURCES */

/* read the resources files */
/*
 * using XGetDefault() or the hand-rolled replacement
 */
void extract_resources (Display * display, const char *name) {
#ifndef NO_RESOURCES
# ifdef USE_XGETDEFAULT
/*
 * get resources using the X library function
 */
	int entry;

#ifdef XrmEnumOneLevel
	XrmName name_prefix[3];
	XrmClass class_prefix[3];
	char *displayResource;
	XrmDatabase database;
	char *screenResource;
	XrmDatabase screenDatabase;

/*
 * Get screen-specific resources (X11R5) and merge into common resources.
 */
//	DXR_ALEXIS (1, (stderr, "extract_resources\n"));
	database = NULL;
	screenDatabase = NULL;
	displayResource = XResourceManagerString (display);
	if (displayResource != NULL)
		database = XrmGetStringDatabase (displayResource);
	screenResource = XScreenResourceString (DefaultScreenOfDisplay (display));

	if (screenResource != NULL)
		screenDatabase = XrmGetStringDatabase (screenResource);

	XrmMergeDatabases (screenDatabase, &database);
	XrmSetDatabase (display, database);
#endif

	for (entry = 0; entry < optList_size (); entry++)
	{
		char *p;
		const char *kw = optList[entry].kw;

		if (kw == NULL || *(optList[entry].dp) != NULL)
			continue;	/* previously set */
		if ((p = XGetDefault (display, name, kw)) != NULL ||
		    (p = XGetDefault (display, APL_SUBCLASS, kw)) != NULL ||
		    (p = XGetDefault (display, APL_CLASS, kw)) != NULL)
		{
			*optList[entry].dp = p;

			if (optList_isBool (entry))
			{
				if (!my_strcasecmp (p, "TRUE"))
					Options |= (optList[entry].flag);
				else
					Options &= ~(optList[entry].flag);
			}
		}
	}

/*
 * [R5 or later]: enumerate the resource database
 */
#ifdef XrmEnumOneLevel
#ifdef KEYSYM_RESOURCE
	name_prefix[0] = XrmStringToName (name);
	name_prefix[1] = XrmStringToName ("keysym");
	name_prefix[2] = NULLQUARK;
	class_prefix[0] = XrmStringToName (APL_SUBCLASS);
	class_prefix[1] = XrmStringToName ("Keysym");
	class_prefix[2] = NULLQUARK;
	XrmEnumerateDatabase (XrmGetDatabase (display),
    name_prefix,
    class_prefix, XrmEnumOneLevel, define_key,
    NULL);
	name_prefix[0] = XrmStringToName (APL_CLASS);
	name_prefix[1] = XrmStringToName ("keysym");
	class_prefix[0] = XrmStringToName (APL_CLASS);
	class_prefix[1] = XrmStringToName ("Keysym");
	XrmEnumerateDatabase (XrmGetDatabase (display),
			      name_prefix,
			      class_prefix, XrmEnumOneLevel, define_key,
			      NULL);
#   endif
#  endif

# else /* USE_XGETDEFAULT */
/* get resources the hard way, but save lots of memory */
	const char *fname[] = { rs_configFile,".Xdefaults", ".Xresources" };
	FILE *fd = NULL;
	char *home;

	if ((home = getenv ("HOME")) != NULL)
	{
		int i, len = strlen (home) + 2;
		char *f = NULL;

		for (i = 0; i < (sizeof (fname) / sizeof (fname[0])); i++)
		{
      if( !fname[i][0] )
        break;
        
      f = REALLOC (f,(len + strlen (fname[i])) * sizeof (char));

      sprintf (f, "%s/%s", home, fname[i]);

      if ((fd = fopen (f, "r")) != NULL)
        break;
		}
		FREE (f);
	}
/*
 * The normal order to match resources is the following:
 * @ global resources (partial match, ~/.Xdefaults)
 * @ application file resources (XAPPLOADDIR/Rxvt)
 * @ class resources (~/.Xdefaults)
 * @ private resources (~/.Xdefaults)
 *
 * However, for the hand-rolled resources, the matching algorithm
 * checks if a resource string value has already been allocated
 * and won't overwrite it with (in this case) a less specific
 * resource value.
 *
 * This avoids multiple allocation.  Also, when we've called this
 * routine command-line string options have already been applied so we
 * needn't to allocate for those resources.
 *
 * So, search in resources from most to least specific.
 *
 * Also, use a special sub-class so that we can use either or both of
 * "XTerm" and "Rxvt" as class names.
 */

	get_xdefaults (fd, name);
	get_xdefaults (fd, APL_SUBCLASS);

#  ifdef XAPPLOADDIR
	{
		FILE *ad = fopen (XAPPLOADDIR "/" APL_SUBCLASS, "r");

		if (ad != NULL)
		{
			get_xdefaults (ad, "");
			fclose (ad);
		}
	}
#  endif /* XAPPLOADDIR */

	get_xdefaults (fd, APL_CLASS);
	get_xdefaults (fd, "");	/* partial match */
	if (fd != NULL)
		fclose (fd);

# endif	/* USE_XGETDEFAULT */
#endif /* NO_RESOURCES */

/*
 * even without resources, at least do this setup for command-line
 * options and command-line long options
 */
#ifdef MULTICHAR_SET
	set_multichar_encoding (rs_multichar_encoding);
#endif
#ifdef GREEK_SUPPORT
/* this could be a function in grkelot.c */
/* void set_greek_keyboard (const char * str); */
	if (rs_greek_keyboard)
	{
		if (!strcmp (rs_greek_keyboard, "iso"))
			greek_setmode (GREEK_ELOT928);	/* former -grk9 */
		else if (!strcmp (rs_greek_keyboard, "ibm"))
			greek_setmode (GREEK_IBM437);	/* former -grk4 */
	}
#endif /* GREEK_SUPPORT */

#define to_keysym(pks,str) do { KeySym sym;\
if (str && ((sym = XStringToKeysym(str)) != 0)) *pks = sym; } while (0)

#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
	to_keysym (&ks_bigfont, rs_bigfont_key);
	to_keysym (&ks_smallfont, rs_smallfont_key);
#endif
#undef to_keysym
}
