/*
 * convcard - A tool for converting vcards
 * Copyright (C) 2004-2005  Armin Bauer <armin.bauer@opensync.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 * 
 */
 
#include <opensync/opensync.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <glib.h>

static void usage (char *name, int ecode)
{
	fprintf (stderr, "Usage: %s [options] <source>\n\n", name);
	fprintf (stderr, "Converts a vCard version 2.1 in a vCard version 3.0 and vice versa.\n");
	fprintf (stderr, "By default the source file format is autodetected, and is converted to\n");
	fprintf (stderr, "the other format.\n\n");
	fprintf (stderr, "One of the tools from OpenSync.\n\n");
	fprintf (stderr, "Options:\n");
	fprintf (stderr, "  <source>\t\tThe vcard to convert.\n");
	fprintf (stderr, "  --out <file>\t\tSave the output in <file>.  Default is stdout.\n");
	fprintf (stderr, "  --from-vcard2.1\tTreat <source> as 2.1 vcard.\n");
	fprintf (stderr, "  --from-vcard3.0\tTreat <source> as 3.0 vcard.\n");
	exit (ecode);
}

typedef enum conv_detection {
	DETECT_AUTO = 0,
	DETECT_VCARD_21 = 1,
	DETECT_VCARD_30 = 2,
	DETECT_VEVENT_10 = 3,	
	DETECT_VEVENT_20 = 4
} conv_detection;

OSyncObjFormat *conv_run_detection(OSyncFormatEnv *env, OSyncChange *change)
{
	OSyncObjFormat *sourceformat = NULL;
	OSyncError *error = NULL;

	osync_change_set_objformat_string(change, "plain");

	if (!(sourceformat = osync_change_detect_objformat(env, change, &error))) {
		fprintf(stderr, "Unable to detect format of the file\n");
		exit(1);
	}
	osync_change_set_objformat(change, sourceformat);
	osync_change_set_objtype(change, osync_objformat_get_objtype(sourceformat));
	OSyncObjFormat *targetformat = NULL;

	if (!strcmp(osync_objformat_get_name(sourceformat), "vcard21")) {
		targetformat = osync_conv_find_objformat(env, "vcard30");
		return targetformat;
	}
	
	if (!strcmp(osync_objformat_get_name(sourceformat), "vcard30")) {
		targetformat = osync_conv_find_objformat(env, "vcard21");
		return targetformat;
	}

	if (!strcmp(osync_objformat_get_name(sourceformat), "vevent10")) {
		targetformat = osync_conv_find_objformat(env, "vevent20");
		return targetformat;
	}
	
	if (!strcmp(osync_objformat_get_name(sourceformat), "vevent20")) {
		targetformat = osync_conv_find_objformat(env, "vevent10");
		return targetformat;
	}
	
	fprintf(stderr, "The file is no vcard/vcal. Try forcing conversion --from-xyz!\n");
	exit(1);
}

OSyncObjFormat *conv_type_to_targetformat(OSyncFormatEnv *osync, conv_detection type)
{
	OSyncObjFormat *targetformat = NULL;
	if (type == DETECT_VCARD_21) {
		targetformat = osync_conv_find_objformat(osync, "vcard30");
		return targetformat;
	}
	
	if (type == DETECT_VCARD_30) {
		targetformat = osync_conv_find_objformat(osync, "vcard21");
		return targetformat;
	}

	if (type == DETECT_VEVENT_10) {
		targetformat = osync_conv_find_objformat(osync, "vevent10");
		return targetformat;
	}
	
	if (type == DETECT_VEVENT_20) {
		targetformat = osync_conv_find_objformat(osync, "vevent20");
		return targetformat;
	}

	fprintf(stderr, "Did not recognize the file format?\n");
	exit(1);
}

OSyncObjFormat *conv_type_to_sourceformat(OSyncFormatEnv *osync, conv_detection type)
{
	OSyncObjFormat *sourceformat = NULL;
	if (type == DETECT_VCARD_21) {
		sourceformat = osync_conv_find_objformat(osync, "vcard21");
		return sourceformat;
	}
	
	if (type == DETECT_VCARD_30) {
		sourceformat = osync_conv_find_objformat(osync, "vcard30");
		return sourceformat;
	}

	if (type == DETECT_VEVENT_10) {
		sourceformat = osync_conv_find_objformat(osync, "vevent10");
		return sourceformat;
	}
	
	if (type == DETECT_VEVENT_20) {
		sourceformat = osync_conv_find_objformat(osync, "veven20");
		return sourceformat;
	}

	fprintf(stderr, "Did not recognize the file format?\n");
	exit(1);
}

int main (int argc, char *argv[])
{
	if (argc < 2) {
		usage (argv[0], 1);
	}

	char *filename = argv[1];
	char *output = NULL;
	conv_detection type = DETECT_AUTO;
	OSyncObjFormat *targetformat = NULL;
	
	int i;
	for (i = 1; i < argc; i++) {
		char *arg = argv[i];
		if (!strcmp (arg, "--from-vcard2.1")) {
			type = DETECT_VCARD_21;
		} else if (!strcmp (arg, "--from-vcard3.0")) {
			type = DETECT_VCARD_30;
		} else if (!strcmp (arg, "--from-vcal")) {
			type = DETECT_VEVENT_10;
		} else if (!strcmp (arg, "--from-ical")) {
			type = DETECT_VEVENT_20;
		} else if (!strcmp (arg, "--out")) {
			output = argv[i + 1];
			i++;
			if (!output)
				usage (argv[0], 1);
		} else if (!strcmp (arg, "--help")) {
			usage (argv[0], 0);
		} else if (!strcmp (arg, "--")) {
			break;
		} else if (arg[0] == '-') {
			usage (argv[0], 1);
		/*
		} else {
			usage (argv[0], 1);
		*/
		}
	}
	
	OSyncEnv *env = osync_env_new();
	OSyncError *error = NULL;
	if (!osync_env_initialize(env, &error)) {
		fprintf(stderr, "Unable to load environment: %s\n", osync_error_print(&error));
		exit(1);
	}
	
	OSyncFormatEnv *conv_env = osync_conv_env_new(env);
	if (!conv_env) {
		fprintf(stderr, "Unable to create format environment");
		exit(1);
	}
	
	char *buffer;
	int size;
	if (!osync_file_read(filename, &buffer, &size, &error)) {
		fprintf(stderr, "%s\n", osync_error_print(&error));
		osync_error_free(&error);
		return 1;
	}
	
	OSyncChange *change = osync_change_new();
	osync_change_set_uid(change, filename);		
	osync_change_set_data(change, buffer, size, TRUE);
	osync_change_set_conv_env(change, conv_env);

	OSyncObjFormat *sourceformat = NULL;
	if (type == DETECT_AUTO) {
		targetformat = conv_run_detection(conv_env, change);
		osync_debug("CONVCARD", 3, "Done detecting. %s", osync_objformat_get_name(osync_change_get_objformat(change)));
	} else if (type < DETECT_VEVENT_10) {
		OSyncObjType *objtype = osync_conv_find_objtype(conv_env, "contact");
		osync_change_set_objtype(change, objtype);
		sourceformat = conv_type_to_sourceformat(conv_env, type);
		osync_change_set_objformat(change, sourceformat);
		targetformat = conv_type_to_targetformat(conv_env, type);
	} else {
		OSyncObjType *objtype = osync_conv_find_objtype(conv_env, "event");
		osync_change_set_objtype(change, objtype);
		sourceformat = conv_type_to_sourceformat(conv_env, type);
		osync_change_set_objformat(change, sourceformat);
		targetformat = conv_type_to_targetformat(conv_env, type);
	}
	
	if (!osync_change_convert(conv_env, change, targetformat, &error)) {
		fprintf(stderr, "Unable to convert data: %s\n", osync_error_print(&error));
		exit(1);
	}
	
	buffer = osync_change_get_data(change);
	size = osync_change_get_datasize(change);
	
	if (output) {
		if (!osync_file_write(output, buffer, size, 0644, &error)) {
			fprintf(stderr, "Unable to write file %s: %s", output, osync_error_print(&error));
			osync_error_free(&error);
			return 1;
		}
	} else {
		puts(buffer);
		printf("\n");
	}
	
	osync_conv_env_free(conv_env);
	osync_env_finalize(env, NULL);
	osync_env_free(env);
	
	return 0;
}
