## -*- coding: utf-8 -*-
#
# «mysql_configuration» - MCC MysQL Related configuration plugin
#
# Copyright (C) 2009, Mario Limonciello, for Mythbuntu
#
#
# Mythbuntu is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this application; if not, write to the Free Software Foundation, Inc., 51
# Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##################################################################################

from MythbuntuControlCentre.plugin import MCCPlugin
import os
import re
import logging

from mythbuntu_common.mysql import MySQLHandler
from mythbuntu_common.dictionaries import get_tweak_dictionary

MYSQL_TXT='/etc/mythtv/mysql.txt'

MYTHTV_SETUP='/usr/bin/mythtv-setup'
OPTIMIZE_DATABASE="/usr/bin/x-terminal-emulator -e perl /usr/share/doc/mythtv-backend/contrib/optimize_mythdb.pl"

class MySQLConfigurationPlugin(MCCPlugin):
    """A configuration tool for MySQL Related Connectivity"""

    def __init__(self):
        #Initialize parent class
        information = {}
        information["name"] = "MySQL"
        information["icon"] = "gtk-network"
        information["ui"] = "tab_mysql_configuration"
        MCCPlugin.__init__(self,information)
        self.mysql=MySQLHandler()

    def captureState(self):
        """Determines the state of the items on managed by this plugin
           and stores it into the plugin's own internal structures"""

        #Roles
        self.frontend=False
        self.mysql_service=False
        if self.query_installed('mythtv-backend-master'):
            #enable master backend and backend, disable frontend
            self.master=True
            self.backend=True
            if os.path.exists("/etc/mysql/conf.d/mythtv.cnf"):
                in_f=open("/etc/mysql/conf.d/mythtv.cnf")
                for line in in_f:
                    if re.compile("^bind-address").search(line):
                        self.mysql_service=True
                        break
                in_f.close()
        else:
            self.master=False
            self.backend=self.query_installed('mythtv-backend')

            #common mysql.txt for remote backend & frontend
            self.mysql.read_mysql_txt(MYSQL_TXT)

        #Dictionaries
        self.dictionary_state={}
        list=get_tweak_dictionary(self)
        for item in list:
            self.dictionary_state[list[item]]=os.path.exists(item)

    def applyStateToGUI(self):
        """Takes the current state information and sets the GUI
           for this plugin"""

        if self.master:
            self.master_backend_vbox.show()
            self.not_master_backend_vbox.hide()
            if self.mysql_service:
                self.enablemysql.set_active(1)
            else:
                self.enablemysql.set_active(0)
        else:
            self.not_master_backend_vbox.show()
            self.master_backend_vbox.hide()
            config=self.mysql.get_config()
            self.mysql_user.set_text(config["user"])
            self.mysql_pass.set_text(config["password"])
            self.mysql_server.set_text(config["server"])
            self.mysql_database.set_text(config["database"])
            self.mysql_test_hbox.hide()

        if self.backend:
            self.backend_vbox.show()
        else:
            self.backend_vbox.hide()

        #Dictionaries
        for item in self.dictionary_state:
            item.set_active(self.dictionary_state[item])


        #In case we can't show accurate data, show a warning
        try:
            file=open(MYSQL_TXT)
            file.close()
            self.read_error.hide()
        except IOError, msg:
            if msg.errno == 13:
                self.read_error.show()

        self._incomplete=False

    def compareState(self):
        """Determines what items have been modified on this plugin"""
        #Prepare for state capturing
        MCCPlugin.clearParentState(self)

        list=get_tweak_dictionary(self)
        for item in list:
            if list[item].get_active() != self.dictionary_state[list[item]]:
                self._markReconfigureRoot(item,list[item].get_active())

        if self.master:
            if self.mysql_service and self.enablemysql.get_active_text() == "Disable":
                #disable service
                self._markReconfigureRoot("mysql_service",False)
            if not self.mysql_service and self.enablemysql.get_active_text() == "Enable":
                #enable service
                self._markReconfigureRoot("mysql_service",True)
        else:
            config=self.mysql.get_config()
            if self.mysql_user.get_text() != config["user"]:
                self._markReconfigureRoot("mysql_user",self.mysql_user.get_text())
            if self.mysql_pass.get_text() != config["password"]:
                self._markReconfigureRoot("mysql_password",self.mysql_pass.get_text())
            if self.mysql_server.get_text() != config["server"]:
                self._markReconfigureRoot("mysql_server",self.mysql_server.get_text())
            if self.mysql_database.get_text() != config["database"]:
                self._markReconfigureRoot("mysql_database",self.mysql_database.get_text())


    def root_scripted_changes(self,reconfigure):
        """System-wide changes that need root access to be applied.
           This function is ran by the dbus backend"""
        mysql_file = False
        config=self.mysql.get_config()
        for item in reconfigure:
            if item == "mysql_user":
                config["user"] = reconfigure[item]
                mysql_file = True
            elif item == "mysql_password":
                config["password"] = reconfigure[item]
                mysql_file = True
            elif item == "mysql_server":
                config["server"] = reconfigure[item]
                mysql_file = True
            elif item == "mysql_database":
                config["database"] = reconfigure[item]
                mysql_file = True
            elif item == "mysql_service":
                self.mysql.toggle_mysql_service_config(reconfigure[item])
                self.mysql.restart_mysql_service()
            elif item == "/etc/mysql/conf.d/mythtv-tweaks.cnf" or\
                 item == "/etc/cron.daily/optimize_mythdb":
                if reconfigure[item]:
                    import shutil
                    try:
                        if os.path.exists(item):
                            os.remove(item)
                        if item == "/etc/mysql/conf.d/mythtv-tweaks.cnf":
                            shutil.copy('/usr/share/mythbuntu/examples/mythtv-tweaks.dist',item)
                        else:
                            shutil.copy('/usr/share/doc/mythtv-backend/contrib/maintenance/optimize_mythdb.pl',item)
                            os.chmod(item,0755)
                    except Exception, msg:
                        logging.warning("Exception when enabling %s, %s" % (item,msg))
                else:
                    try:
                        os.remove(item)
                    except Exception, msg:
                        logging.warning("Exception when disabling item %s, %s" % (item,msg))
        if mysql_file:
            logging.debug("Updating MySQL config")
            self.mysql.update_config(config)
            logging.debug("Writing out mysql.txt")
            self.mysql.write_mysql_txt(MYSQL_TXT)

    #Callbacks
    def password_reset(self,widget):
        """Resets a user's MySQL password"""
        if widget is not None:
            if widget.get_name() == "mysql_reset_close_button":
                self.mysql_reset_window.hide()
            elif widget.get_name() == "mysql_reset_apply_button":
                self.mysql.reset_password(self.mysql_reset_admin_password.get_text(),
                                          self.mysql_reset_user_password.get_text())
            elif widget.get_name() == "reset_button":
                self.mysql_reset_window.show()
            elif widget.get_name() == "mysql_reset_admin_password":
                self.mysql_reset_apply_button.set_sensitive(
                   (len(self.mysql_reset_admin_password.get_text()) > 0 and
                    len(self.mysql_reset_user_password.get_text()) > 0))

    def do_connection_test(self,widget):
        """Performs a connectivity test to the backend's mysql server"""
        def update_config(old_config=None):
            if old_config is None:
                config={}
                config["user"]=self.mysql_user.get_text()
                config["password"]=self.mysql_pass.get_text()
                config["server"]=self.mysql_server.get_text()
                config["database"]=self.mysql_database.get_text()
            else:
                config=old_config
            self.mysql.update_config(config)

        if widget is not None:
            old_config=self.mysql.get_config()
            update_config()
            result=self.mysql.do_connection_test()
            self.connection_results.set_text(result)
            self.mysql_test_hbox.show()
            if result == "Successful":
                self.pass_mysql.show()
                self.fail_mysql.hide()
                self._incomplete=False
            else:
                self.pass_mysql.hide()
                self.fail_mysql.show()
                self._incomplete=True
            update_config(old_config)


    def launch_app(self,widget):
        """Launches an app defined in the glade file"""
        if widget.get_name() == 'mysql_tweak_button':
            MCCPlugin.launch_app(self,widget,OPTIMIZE_DATABASE)
        elif widget.get_name() == 'mythtv_setup_button':
            MCCPlugin.launch_app(self,widget,MYTHTV_SETUP)
