/*
 * Mausezahn - A fast versatile traffic generator
 * Copyright (C) 2008 Herbert Haas
 * 
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 as published by the 
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more 
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with 
 * this program; if not, see http://www.gnu.org/licenses/gpl-2.0.html
 * 
*/


///////////////////////////////////////////////////
// 
// Table of contents:
// 
// rcv_rtp_init()
// rcv_rtp()
// compare4B()
// got_rtp_packet() 


///////////////////////////////////////////////////
// 
// Documentation about RTP traffic analysis
// 
// See http://wiki.wireshark.org/RTP_statistics
//
// 

#include "mz.h"

// Initialize the rcv_rtp process: Read user parameters and initialize globals
int rcv_rtp_init()
{
   char argval[MAX_PAYLOAD_SIZE];
   char dummy[512];
   int len;
   u_int32_t port = 30000;  // 4-byte variable to catch errors, see below
   
   

   
   // Help text

   if (getarg(tx.arg_string,"help", NULL)==1)
     {
	fprintf(stderr,"\n"
		MAUSEZAHN_VERSION
		"\n"
		"| RTP reception for jitter measurements.\n"
		"|\n"
		"| Parameters:\n"
		"|\n"
		"|   log              ....... Write moving average also in a datafile (not only on terminal).\n"
		"|   logg             ....... Like log but additionally write detailed real-time statistics in a\n"
		"|                            datafile.\n"
		"|   path = <path>    ....... Path to directory where datafiles can be stored (default: local directory).\n"
		"|   num = <10-%d> ...... number of packets to be received for averaging (default: %d).\n"
		"|   port = <0-65535> ....... Change if RTP packets are sent to a different port than 30000 (default).\n"
		"|\n"
		"| Note:\n"
		"|\n"
		"|  Mausezahn can log actual realtime measurement data in data files (in the specified path or\n"
		"|  current directory) but always prints the moving average on the commandline (can be disabled using the\n"
		"|  'quiet' option (-q)).\n" 
		"|\n"
		"|  The realtime data file(s) consist of three columns:\n"
		"|\n"
		"|  1. relative timestamp in usec\n"
		"|  2. 'true' jitter in usec\n"
		"|  3. delta-RX in usec\n"
		"|\n"
		"|  where the 'true' jitter is calculated using the (relative) timestamps inside the received\n"
		"|  packets t(i) and the (relative) timestamps T(i) observed locally when packets are received using\n"
		"|  the formula:\n"
		"|\n"
		"|    jitter(i) = [T(i) - T(i-1)] - [t(i) - t(i-1)] + jitter(i-1)  .\n"
		"|\n"
		"|  This method has two advantages: (i) we do not need to synchronize the clocks of sender and\n"
		"|  receiver, and (ii) the TX-side jitter (possibly created by the OS-scheduler) is subtracted\n"
		"|  so that we only measure the jitter caused by the network. However we do not currently consider\n"
		"|  the jitter caused by the OS-scheduler on the receiver (which is marginal anyway).\n"
		"|  \n"
		"|  The delta-RX values in the third column can be used to analyze the inter-packet delay observed\n"
		"|  at the local station.\n"
		"|  \n"
		"|  The average data files consist of seven columns:\n"
		"|  \n"
		"|  1. relative timestamp in seconds\n" 
		"|  2. minimum jitter\n"
		"|  3. average jitter\n"
		"|  4. minimum jitter\n"
		"|  5. minimum interval\n"
		"|  6. average interval\n"
		"|  7. maximum interval\n"
		"|  \n"
		"|  All measurement values are in usec and refer to the current set of samples (see parameter 'num').\n"
		"|  \n"
		"| EXAMPLE USAGE:\n"
		"|\n"		
		"|  At the TX-station enter:\n"
		"|\n"
		"|    # mz eth0 -t rtp -B 10.3.3.42                     ...  optionally change rate via -d option\n"
		"|\n"
		"|  At the RX-station (10.3.3.42) enter:\n"
		"|\n"
		"|    # mz eth0 -T rtp \"logg, path=/tmp/mz/\"\n"
		"|\n"
		"\n", TIME_COUNT_MAX, TIME_COUNT);
	exit(0);
     }
   
   
   // check argstring for arguments
   
   
   if (getarg(tx.arg_string,"log", NULL)==1)
     {
	rtp_log = 1;
     }

   if (getarg(tx.arg_string,"logg", NULL)==1)
     {
	rtp_log = 2;
     }

   
   if (getarg(tx.arg_string,"path", argval)==1)
     {
	len = strlen(argval);
	if (argval[len-1]!='/')
	  {
	     strncat(argval, "/",1); // ensure that all paths end with "/"
	  }
	strncpy(path, argval, 256);
     }


   if (getarg(tx.arg_string,"num", argval)==1)
     {
	gind_max = (u_int32_t) str2int(argval);
	if (gind_max > TIME_COUNT_MAX) 
	  {
	     gind_max = TIME_COUNT_MAX;
	     fprintf(stderr, " mz/Warning: num range is 10..%d. Will reset to %d.\n", 
		     TIME_COUNT_MAX, TIME_COUNT_MAX);
	  }
	else if (gind_max < 10)
	  {
	     gind_max = 10;
	     fprintf(stderr, " mz/Warning: num range is 10..%d. Will reset to 10.\n",
		     TIME_COUNT_MAX);
	  }
     }
   

   // initialize global filter string 
   strncpy (rtp_filter_str, "udp dst port 30000", 64);
   
   if (getarg(tx.arg_string,"port", argval)==1)
     {
	port = (u_int32_t) str2int(argval);
	if (port>65535)
	  {
	     port = 30000;
	     fprintf(stderr, " mz: Too large port number! Reset to default port (30000).\n");
	  }
	
	sprintf(rtp_filter_str, "udp dst port %u", (unsigned int) port);
     }
   
   

   // open file
   
   if (rtp_log)
     {
	// get a new filename
	timestamp_human(filename, "rtp_avg_");
	strncpy(dummy, path, 256);
	strncat(dummy, filename, 256);
	if (verbose) fprintf(stderr, " mz: Will open %s\n", dummy);
	
	fp = fopen (dummy, "w+");
	
	if (fp == NULL)
	  {
	     perror("fopen");
	     exit (-1);
	  }
	
	gtotal=0; // counts written data blocks
	fprintf(fp, "# Average jitter measurements made by Mausezahn " MAUSEZAHN_VERSION_SHORT ".\n");
	fprintf(fp, "# Timestamp is in seconds, all other values in microseconds.\n");
	fprintf(fp, "# Column values (from left to right):\n");
	fprintf(fp, "#  1. Timestamp\n"
		    "#  2. min_jitter\n"
		    "#  3. avg_jitter\n"
		    "#  4. max_jitter\n"
		    "#  5. min_deltaRX\n"
		    "#  6. avg_deltaRX\n"
		    "#  7. max_deltaRX\n"
		    "#  8. packet drop count (total)\n"
		    "#  9. packet disorder count (total)\n");

	
	if (rtp_log==2) ///////////// also detailed log required /////////////
	  {
	     // get a new filename
	     timestamp_human(filename, "rtp_rt_");
	     strncpy(dummy, path, 256);
	     strncat(dummy, filename, 256);
	     if (verbose) fprintf(stderr, " mz: Will open %s\n", dummy);
	     
	     fp2 = fopen (dummy, "w+");
	     
	     if (fp2 == NULL)
	       {
		  perror("fopen");
		  exit (-1);
	       }
	     
	     fprintf(fp2, "# Jitter measurements made by Mausezahn " MAUSEZAHN_VERSION_SHORT ".\n");
	     fprintf(fp2, "# Timestamp (usec) , true jitter (usec), delta-RX (usec)\n");
	  }
	
     }

   
   drop=0;
   dis=0;
   
   return 0;
}
   

   
   
   
   
   
////////////////////////////////////////////////////////////////////////////////////////////
//
// Defines the pcap handler and the callback function
int rcv_rtp()
{
   char   errbuf[PCAP_ERRBUF_SIZE];
   
   pcap_t     *p;
   
   struct bpf_program filter;

   
   
   p = pcap_open_live (tx.device,
		       MAXBYTES_TO_READ,   // max num of bytes to read
		       0,                  // 1 if promiscuous mode
		       -1,                 // read timeout 'until error' (-1 = indefinitely)
		       errbuf);
   
   if (p == NULL)
     {
	fprintf(stderr," mz/rcv_rtp: %s\n",errbuf);
	exit(1);
     }
   

   if ( pcap_compile(p,
		     &filter,         // the compiled version of the filter
		     rtp_filter_str,  // text version of filter
		     0,               // 1 = optimize
		     0)               // netmask
	== -1)
     {
	fprintf(stderr," mz/rcv_rtp: Error calling pcap_compile\n");
	exit(1);
     }
   
   
   
   if ( pcap_setfilter(p, &filter) == -1)
     {
	fprintf(stderr," mz/rcv_rtp: Error setting filter\n");
	pcap_geterr(p);
	exit(1);
     }

   again:


   pcap_loop (p,
	      1,                // number of packets to wait
	      got_rtp_packet,   // name of callback function
	      NULL);            // optional additional arguments for callback function
   
   
   goto again;
   
   
   // TODO: Currently we never reach this point!
   fprintf(stderr, " mz: receiving of RTP finished.\n");
   pcap_close(p);
   
   return 0;
}




// Compares two 4-byte variables byte by byte
// returns 0 if identical, 1 if different
int compare4B (u_int8_t *ip1, u_int8_t *ip2)
{
   if (*ip1 != *ip2) return 1;
   if (*(ip1+1) != *(ip2+1)) return 1;
   if (*(ip1+2) != *(ip2+2)) return 1;
   if (*(ip1+3) != *(ip2+3)) return 1;
   
   return 0;
}
                




// Handler function to do something when RTP messages are received
void got_rtp_packet(u_char *args,
		    const struct pcap_pkthdr *header, // statistics about the packet (see 'struct pcap_pkthdr')
		    const u_char *packet)             // the bytestring sniffed
{
   const struct struct_ethernet *ethernet;
   const struct struct_ip       *ip;
   const struct struct_udp      *udp;
   const struct struct_rtp      *rtp;
   
   int size_ethernet = sizeof(struct struct_ethernet);
   int size_ip = sizeof(struct struct_ip);
   int size_udp = sizeof(struct struct_udp);
// int size_rtp = sizeof(struct struct_rtp);
   
   ethernet = (struct struct_ethernet*)(packet);
   ip       = (struct struct_ip*)(packet+size_ethernet);
   udp      = (struct struct_udp*)(packet+size_ethernet+size_ip);
   rtp      = (struct struct_rtp*)(packet+size_ethernet+size_ip+size_udp);
   
   u_int32_t mz_ssrc = 0xedfefeca; // cafefeed in big endian
   
   struct timeval 
     curtv,
     deltaTX,
     deltaRX;

   u_int32_t 
     i,
     jitter_abs,
     jitter_avg,
     jitter_max,
     jitter_min,
     deltaRXusec,
     deltaRX_avg,
     deltaRX_max,
     deltaRX_min;

   unsigned long int
     curtime=0;
   
   u_int8_t *x,*y;
   
   char dummy[256];
   unsigned char *dum;
   
     
   

   // check if the RTP packet is really from a Mausezahn instance:
   if (compare4B((u_int8_t*) &rtp->ssrc, (u_int8_t*) &mz_ssrc))
     {
	if (verbose)
	  {
	     fprintf(stderr, " mz, FYI: got another packet on port 30000 (ignored)\n");
	  }
     }
   else    // we got a valid RTP packet from a Mausezahn instance
     {
	// Get current SQNR and store it in 'sqnr_cur' in normal byte order
	x  = (u_int8_t*) &rtp->sqnr;
	y  = (u_int8_t*) &sqnr_cur;
	
	*y = *(x+1);
	y++;
	*y = *x;

	/////////////////////////////////////////////////////////////////////
	// Packet drop and disorder detection:
	if (sqnr0_flag)
	  {
	     if (sqnr_next==sqnr_cur)  // correct SQNR received
	       {
		  sqnr_next++;
		  sqnr_last++;
	       }
	     else if (sqnr_last>sqnr_cur) // disordered sequence
	       {
		  dis++;
		  drop--;
	       }
	     else // packet drop
	       {
		  drop += (sqnr_cur-sqnr_next);
		  sqnr_last = sqnr_cur;
		  sqnr_next = (++sqnr_last);
	       }
	  }
	else 
	  {
	     // initial synchronization with observed SQNR:
	     sqnr_last = sqnr_cur;
	     sqnr_next = (++sqnr_last);
	     sqnr0_flag++;
	  }
	//
	/////////////////////////////////////////////////////////////////////

	
	
	// Get current timestamp
	getcurtime(&curtv);
	timeRX[gind].tv_sec  = curtv.tv_sec;
	timeRX[gind].tv_usec = curtv.tv_usec;

	// Get timestamp of packet and convert to normal byte order (reverse all bytes)
	x  = (u_int8_t*) &rtp->time_sec;
	y  = (u_int8_t*) &timeTX[gind].tv_sec;
	
	*y = *(x+3);
	y++;
	*y = *(x+2);
	y++;
	*y = *(x+1);
	y++;
	*y = *x;
	
	x  = (u_int8_t*) &rtp->time_usec;
	y  = (u_int8_t*) &timeTX[gind].tv_usec;
	
	*y = *(x+3);
	y++;
	*y = *(x+2);
	y++;
	*y = *(x+1);
	y++;
	*y = *x;

	gind++;
	
	///////////////////////////////////////////////////////////////////////////////////////////////////////////////
	if (gind == gind_max) // array full, now calculate statistics *************************************************
	  {
	     gind=0;
	     gtotal++;
	     
	     jitter_avg = 0;
	     jitter_min = 0xffffffff;
	     jitter_max = 0;
	     
	     deltaRX_avg = 0;
	     deltaRX_min = 0xffffffff;
	     deltaRX_max = 0;
	     
	     /////////////////////////////////////////////////////////////////////////////////////////////////////////
	     // calculate deltas and jitters
	     for (i=2; i<gind_max; i++) // omit the first 2 data entries because of artificial high TX-delta!
	       {

		  ////////////////////////////////////////////////////////////////////////////////////////////////////
		  // calculate deltaTX and deltaRX
		  // 
		  timeval_subtract (&timeTX[i-1], &timeTX[i], &deltaTX);
		  timeval_subtract (&timeRX[i-1], &timeRX[i], &deltaRX);

		  // Then calculate the precise jitter by considering also TX-jitter:
		  // (pseudo)jitter = deltaRX - deltaTX, hence we have positive and negative jitter (delay deviations)
		  // jitter entries are in +/- microseconds
		  jitter[i] = (deltaRX.tv_sec*1000000 + deltaRX.tv_usec) - (deltaTX.tv_sec*1000000 + deltaTX.tv_usec);
		  // Add previous pseudojitter to get the true jitter (See Documentation!)
		  jitter[i] += jitter[i-1]; 
		  //
		  ////////////////////////////////////////////////////////////////////////////////////////////////////
		  
		  
		  
		  // Then do the same but consider only the RX timestamps:
		  deltaRXusec = deltaRX.tv_sec*1000000 + deltaRX.tv_usec;
		  
		  
		  
		  ///////////////////////////////////////////////////////////////////////////////////////////
		  // Calculate relative timestamp for column 1 of the datafile 
		  // 
		  curtime = timeRX[i].tv_sec*1000000+timeRX[i].tv_usec;
		  if (time0_flag)
		    {
		       curtime = curtime - time0;
		    }
		  else // this is only done once during the Mausezahn process
		    {
		       time0 = curtime;
		       time0_flag=1;
		       curtime = curtime - time0;
		    }
		  //
		  ///////////////////////////////////////////////////////////////////////////////////////////
		  

		  
		  //////////////////////////////////////////////////////////////////
		  // Determine avg, min, and max jitter within this time frame:
		  jitter_abs = labs(jitter[i]);
		  jitter_avg += jitter_abs;
		  if (jitter_abs < jitter_min) jitter_min = jitter_abs;
		  if (jitter_abs > jitter_max) jitter_max = jitter_abs;
		  //
		  ////////////////////////////////
		  // 
		  deltaRX_avg += deltaRXusec;
		  if (deltaRXusec < deltaRX_min) deltaRX_min = deltaRXusec;
		  if (deltaRXusec > deltaRX_max) deltaRX_max = deltaRXusec;
		  //
		  ///////////////////////////////////////////////////////////////////
		  
		  /// PRINT IN FILE_2: Detailed jitter data ///
		  if (rtp_log==2)
		    {
		       fprintf(fp2, "%lu, %li, %lu\n", curtime, jitter[i], (long unsigned int) deltaRXusec);
		       fflush(fp2); // save everything immediately (CHECK if fsync() is additionally needed)
		    }

	       } // end for (i=2; i<gind_max; i++) 
	     /////////////////////////////////////////////////////////////////////////////////////////////////////////
	     
	     
	     jitter_avg = jitter_avg / (gind_max-2);    // average true jitter, always positive
	     deltaRX_avg = deltaRX_avg / (gind_max-2);  // average RX deltas, always positive
	     
	     // PRINT ON CLI: statistics data
	     if (!quiet) 
	       {  dum =  (unsigned char*) &ip->src;
		  fprintf(stdout, 
			  "Jitter (min/avg/max) = %lu/%lu/%lu usec; "
			  "delta-RX (min/avg/max) = %lu/%lu/%lu usec (got %u packets from %u.%u.%u.%u, %u lost, %u out of order).\n",
			  (long unsigned int) jitter_min,
			  (long unsigned int) jitter_avg,
			  (long unsigned int) jitter_max,
			  (long unsigned int) deltaRX_min,
			  (long unsigned int) deltaRX_avg,
			  (long unsigned int) deltaRX_max,
			  gind_max,
			  *(dum),*(dum+1),*(dum+2),*(dum+3),
			  drop,
			  dis);
	       }

	     // Determine whether some packets got lost:
	     // 
	     // 
	     // 
	     // 
	     
	     
	     
	     /// PRINT IN FILE_1: statistics only ///
	     if (rtp_log)
	       {
		  getcurtime(&curtv);
		  fprintf(fp, 
			  "%f, %lu, %lu, %lu, %lu, %lu, %lu, %u, %u\n",
			  (float) curtime/1000000,
			  (long unsigned int) jitter_min,
			  (long unsigned int) jitter_avg,
			  (long unsigned int) jitter_max,
			  (long unsigned int) deltaRX_min,
			  (long unsigned int) deltaRX_avg,
			  (long unsigned int) deltaRX_max,
			  drop,
			  dis);
		  fflush(fp);
	       }
	     
	     
	     
	     // Open another file if current file reaches a limit
	     // 
	     if ((rtp_log==2) && (gtotal>MAX_DATA_BLOCKS)) // file big enough, 
	       {
		  gtotal=0;
		  
		  if (fclose(fp2) == EOF)
		    {
		       perror("fclose");
		       exit(1);
		    }
		  
		  if (verbose) fprintf(stderr, " mz: %s written.\n",filename);
		  
		  timestamp_human(filename, "rtp_");  // get a new filename
		  strncpy(dummy, path, 256);
		  strncat(dummy, filename, 256);
		  
		  if (verbose) fprintf(stderr, " mz: Will open %s\n", dummy);
		  
		  if  ( (fp2 = fopen (dummy, "w+")) == NULL)
		    {
		       if (errno != EAGAIN)
			 {
			    perror("fopen");
			    exit (-1);
			 }
		    }
		  fprintf(fp2, "# Jitter measurements made by Mausezahn " MAUSEZAHN_VERSION_SHORT ".\n");
		  fprintf(fp2, "# Timestamp (usec) , true jitter (usec), delta-RX (usec)\n");
	       }
	     
	     
	  } // statistics end *********************************************************************
     }
}
