/*
 * NASPRO - The NASPRO Architecture for Sound Processing
 * Core library
 *
 * Copyright (C) 2007-2012 NASPRO core development team
 *
 * See the COPYING file for license conditions.
 */

/*
   Title: Common data types

   Data types used throughout the library and related generic utilities and
   defintions.
 */

#ifndef _NASPRO_CORE_TYPES_H
#define _NASPRO_CORE_TYPES_H

#ifndef _NASPRO_CORE_LIB_H
# error Only <NASPRO/core/lib.h> can be included directly.
#endif

NACORE_BEGIN_C_DECLS

/*
   Macro: NACORE_ELOOP

   Error code indicating too many levels of symbolic links.

   It is equivalent to ELOOP on POSIX-compliant platforms.
 */
#ifdef _WIN32
# define NACORE_ELOOP	10062L	/* WSAELOOP */
#else
# define NACORE_ELOOP	ELOOP
#endif

/*
   Macro: NACORE_EUNKNOWN

   Error code for unknown errors.

   Its value should not overlap with errno.h error codes.
 */
#define NACORE_EUNKNOWN	0xec0de

/*
   Macro: NACORE_LIBC_SIZE_FORMAT_LM

   Length modifier for size_t to be used with printf()-like functions of the
   platform's standard C library.

   This is not needed for NASPRO core printf()-like functions, since those
   understand the z length modifier correctly.

   See also <NACORE_LIBC_SIZE_FORMAT_TYPE>.
 */

/*
   Macro: NACORE_LIBC_SIZE_FORMAT_TYPE

   Type from/to which size_t values can be safely casted.

   This is mainly meant to be used in conjuction with
   <NACORE_LIBC_SIZE_FORMAT_LM>.

   Example:

     > size_t val = 0x12341234;
     >
     > printf("The value of val is: %" NACORE_LIBC_FORMAT_LM "u\n",
     >        (NACORE_LIBC_SIZE_FORMAT_TYPE)val);
 */
#ifdef _WIN32
# ifdef _WIN64
#  define NACORE_LIBC_SIZE_FORMAT_LM	"I64"
#  define NACORE_LIBC_SIZE_FORMAT_TYPE	unsigned long long
# else
#  define NACORE_LIBC_SIZE_FORMAT_LM	"l"
#  define NACORE_LIBC_SIZE_FORMAT_TYPE	unsigned long
# endif
#else
# define NACORE_LIBC_SIZE_FORMAT_LM	"z"
# define NACORE_LIBC_SIZE_FORMAT_TYPE	size_t
#endif

/*
   Type: nacore_get_size_cb

   A function that returns the size of some value.

   Parameters:

     value	- Value pointer.
     opaque	- Extra opaque data pointer or NULL.

   Returns:

     Size of value as number of bytes.
 */
typedef size_t
(*nacore_get_size_cb)(const void *value, void *opaque);

/*
   Type: nacore_cmp_cb

   A function that compares two values.

   Parameters:

     v1		- First value.
     v2		- Second value.
     opaque	- Extra opaque data pointer or NULL.

   Returns:

     A strictly negative value if v1 < v2, 0 if v1 = v2, a strictly positive
     value if v1 > v2.
 */
typedef int
(*nacore_cmp_cb)(const void *v1, const void *v2, void *opaque);

/*
   Type: nacore_to_string_cb

   A function that retuns a textual description of some value.

   Parameters:

     value	- Value pointer.
     opaque	- Extra opaque data pointer or NULL.

   Returns:
 
     A malloc()-allocated string containing a textual description of value. The
     caller is in charge of free()ing such string. If there was not enough
     memory, the function should return NULL.
 */
typedef char *
(*nacore_to_string_cb)(const void *value, void *opaque);

/*
   Type: nacore_filter_cb

   A function that decides whether a certain value should be filtered out or
   not.

   Parameters:

     value	- Pointer to the value to be examined.
     opaque	- Extra opaque data pointer or NULL.

   Returns:

     0 if the value has to be filtered out, non-0 otherwise.
 */  
typedef char
(*nacore_filter_cb)(const void *value, void *opaque);

/*
   Type: nacore_op_cb

   A function that performs some operation on a value.

   Usually, it is employed when performing context-dependent operations on
   values belonging to a certain set of data.

   Paramters:

     value	- Pointer to the value to operate on.
     opaque	- Extra opaque data pointer or NULL.
 */
typedef void
(*nacore_op_cb)(void *value, void *opaque);

/*
   Function: nacore_ptr_cmp()

   Pointer comparison function.

   opaque is used as a flag that, if NULL, makes the function usable as a
   <nacore_cmp_cb> callback for ascending ordering, otherwise for descending
   ordering. The opaque pointer is never dereferenced.

   Parameters:

     v1		- First pointer.
     v2		- Second pointer.
     opaque	- Ordering direction flag.

   Returns:

     If opaque is (not) NULL, a strictly negative (positive) value if v1 < v2, 0
     if v1 == v2, a strictly positive (negative) value if v1 > v2.
 */
_NACORE_DEF int
nacore_ptr_cmp(const void *v1, const void *v2, void *opaque);

NACORE_END_C_DECLS

#endif
