/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-pixbuf-cache.h"

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>

G_DEFINE_TYPE (NlPixbufCache, nl_pixbuf_cache, G_TYPE_OBJECT);

#define NL_PIXBUF_CACHE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_PIXBUF_CACHE, \
  NlPixbufCachePrivate))


#define HASH_KEY "%s:%d" //Where %s is the filename/icon_name and %d is size

struct _NlPixbufCachePrivate
{
  GHashTable   *pixbuf_table;
};

/* Forwards */


/* GObject stuff */
static void
nl_pixbuf_cache_finalize (GObject *object)
{
  NlPixbufCachePrivate *priv;

  priv = NL_PIXBUF_CACHE_GET_PRIVATE (object);

  if (priv->pixbuf_table)
    {
      g_hash_table_destroy (priv->pixbuf_table);
      priv->pixbuf_table = NULL;
    }

  G_OBJECT_CLASS (nl_pixbuf_cache_parent_class)->finalize (object);
}

static void
nl_pixbuf_cache_class_init (NlPixbufCacheClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize     = nl_pixbuf_cache_finalize;

  g_type_class_add_private (obj_class, sizeof (NlPixbufCachePrivate));
}

static void
nl_pixbuf_cache_init (NlPixbufCache *cache)
{
  NlPixbufCachePrivate *priv;

  priv = cache->priv = NL_PIXBUF_CACHE_GET_PRIVATE (cache);

  priv->pixbuf_table = g_hash_table_new_full (g_str_hash, g_str_equal,
                       g_free, g_object_unref);
}

/*
 * Public methods
 */
NlPixbufCache *
nl_pixbuf_cache_get_default (void)
{
  static NlPixbufCache *pixbuf_cache = NULL;

  if (G_UNLIKELY (pixbuf_cache == NULL))
    {
      pixbuf_cache = g_object_new (NL_TYPE_PIXBUF_CACHE,
                                   NULL);
      return pixbuf_cache;
    }
  return g_object_ref (pixbuf_cache);
}

/*
 * Private methods
 */
/* From matchbox-desktop */
static char *
strip_extension (const char *file)
{
  char *stripped, *p;

  stripped = g_strdup (file);

  p = strrchr (stripped, '.');
  if (p &&
      (!strcmp (p, ".png") ||
       !strcmp (p, ".svg") ||
       !strcmp (p, ".xpm")))
    *p = 0;

  return stripped;
}

/* Gets the pixbuf from a desktop file's icon name. Based on the same function
 * from matchbox-desktop
 */
static GdkPixbuf *
get_icon (const gchar *name, guint size)
{
  static GtkIconTheme *theme = NULL;
  GdkPixbuf *pixbuf = NULL;
  gchar *stripped = NULL;

  if (theme == NULL)
    theme = gtk_icon_theme_get_default ();

  if (name == NULL)
    return NULL;

  /* Check to see if it's a URI being passed in */
  if (g_str_has_prefix (name, "file://"))
    {
      gchar *filename;

      filename = g_filename_from_uri (name, NULL, NULL);

      if (filename)
        {
          pixbuf = gdk_pixbuf_new_from_file_at_scale (name, size, size,
                   TRUE, NULL);
        }
      g_free (filename);

      if (GDK_IS_PIXBUF (pixbuf))
        return pixbuf;
    }

  /* See if it's an filename */
  if (g_path_is_absolute (name))
    {
      if (g_file_test (name, G_FILE_TEST_EXISTS))
        {
          pixbuf = gdk_pixbuf_new_from_file_at_scale (name, size, size,
                   TRUE, NULL);
          if (GDK_IS_PIXBUF (pixbuf))
            return pixbuf;
        }
    }

  /* Try loading it directly from disk */
  GtkIconInfo *info;
  if ((info = gtk_icon_theme_lookup_icon (theme, name, size, 0)))
    {
      const gchar *filename;

      filename = gtk_icon_info_get_filename (info);
      pixbuf = gdk_pixbuf_new_from_file_at_scale (filename, size, size, TRUE,
               NULL);
      if (GDK_IS_PIXBUF (pixbuf))
        {
          gtk_icon_info_free (info);
          return pixbuf;
        }
      gtk_icon_info_free (info);
    }

  /* Finally try loading it from the icon theme */
  stripped = strip_extension (name);

  pixbuf = gtk_icon_theme_load_icon (theme,
                                     stripped,
                                     size,
                                     GTK_ICON_LOOKUP_FORCE_SVG, NULL);
  g_free (stripped);

  if (pixbuf &&
      (gdk_pixbuf_get_width (pixbuf) != size
       || gdk_pixbuf_get_height (pixbuf) != size))
    {
      GdkPixbuf *temp = pixbuf;

      pixbuf = gdk_pixbuf_scale_simple (temp, size, size, GDK_INTERP_HYPER);
      g_object_unref (temp);
    }

  return pixbuf;
}



/*
 * Public Methods
 */
GdkPixbuf *
nl_pixbuf_cache_icon_for_name (NlPixbufCache *cache,
                               const gchar         *icon_name,
                               guint                size)
{
  NlPixbufCachePrivate *priv;
  GdkPixbuf *ret = NULL;
  gchar *key;

  g_return_val_if_fail (NL_IS_PIXBUF_CACHE (cache), NULL);
  g_return_val_if_fail (icon_name, NULL);
  priv = cache->priv;

  key = g_strdup_printf (HASH_KEY, icon_name, size);

  /* Check to see if we've already the pixbuf in memory */
  ret = g_hash_table_lookup (priv->pixbuf_table, key);

  /* If not, load it and add it to the hash table */
  if (!GDK_IS_PIXBUF (ret))
    {
      ret = get_icon (icon_name, size);
      if (ret)
        g_hash_table_insert (priv->pixbuf_table, key, ret);
      else
        g_free (key);
    }
  else
    {
      g_free (key);
    }

  return GDK_IS_PIXBUF (ret) ? g_object_ref (ret) : NULL;
}

/* Taken from nautilus::panel-util.c, with some modifications */
GdkPixbuf *
nl_pixbuf_cache_icon_for_gicon(NlPixbufCache *cache,
                               GIcon               *icon,
                               guint                size)
{
  const gchar * const *names;
  GtkIconTheme *icon_theme;
  int i;

  g_return_val_if_fail (NL_IS_PIXBUF_CACHE (cache), NULL);
  g_return_val_if_fail (icon, NULL);

  if (!G_IS_THEMED_ICON (icon))
    return nl_pixbuf_cache_icon_for_name (cache, "folder", size);

  names = g_themed_icon_get_names (G_THEMED_ICON (icon));
  icon_theme = gtk_icon_theme_get_default ();

  for (i = 0; names[i] != NULL; i++)
    {
      if (gtk_icon_theme_has_icon (icon_theme, names[i]))
        return nl_pixbuf_cache_icon_for_name (cache, names[i], size);
    }

  return NULL;
}
