/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// Connection.cpp: implementation of the Connection class.
//
//////////////////////////////////////////////////////////////////////

#include "Connection.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Connection::Connection()
{
	m_socket = 0;
}

Connection::~Connection()
{

}

void Connection::set_host(const mString & host)
{
	m_host = host;
}

void Connection::set_port(unsigned short port)
{
	m_port = port;
}

bool Connection::do_connection()
{
	struct linger ling;
	sockaddr_in sockin;
	struct hostent * hostinf;
	int i;


	hostinf = gethostbyname(m_host.c_str());
	if(!hostinf)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONNECT_FAILED);
		return false;
	}

	memset(&sockin, 0, sizeof(sockin));
	sockin.sin_family = AF_INET;
	sockin.sin_port = htons (m_port);
	
	m_socket = ::socket(PF_INET, SOCK_STREAM, 0);
	if(m_socket == SOCKET_ERROR)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_SOCKET);
		return false;
	}

	ling.l_onoff=0;
	ling.l_linger=0;
	if(::setsockopt(m_socket, SOL_SOCKET, SO_LINGER, (const char*) &ling, sizeof(ling)) == SOCKET_ERROR)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_SOCKET);
		do_close();
		return false;
	}
		
	for(i=0 ; hostinf->h_addr_list[i] ; i++)
	{
		memcpy(&sockin.sin_addr, hostinf->h_addr_list[i], sizeof(struct in_addr));
		if(::connect(m_socket,(sockaddr *)&sockin, sizeof(sockin)) != SOCKET_ERROR)
		{
			return true;
		}
	}
	NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONNECT_FAILED);
	do_close();
	return false;
}

void Connection::do_close()
{
	if(m_socket)
	{
		::closesocket(m_socket);
		m_socket = 0;
	}
	m_entity_cert.Clear();
}

const mString & Connection::get_host() const
{
	return m_host;
}

unsigned short Connection::get_port() const
{
	return m_port;
}

SOCKET Connection::get_socket() const
{
	return m_socket;
}

bool Connection::do_send(BIO *connection, const AdminRequest &request, AdminResponse &response, int ReadTimeout)
{
	ADMIN_RESPONSE * lresponse;
	ADMIN_REQUEST * lrequest;
	fd_set rfds;
	fd_set efds;
	struct timeval tv;
	

	lrequest = NULL;
	if(!request.give_Datas(&lrequest))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!i2d_ADMIN_REQUEST_bio(connection, lrequest))
	{
		ERR_clear_error();
		ASN1_item_free((ASN1_VALUE*)lrequest, AdminRequest::get_ASN1_ITEM());
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONNECTION_LOST);
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)lrequest, AdminRequest::get_ASN1_ITEM());

	//Get response
	if(ReadTimeout)
	{
		FD_ZERO(&rfds);
		FD_ZERO(&efds);
		FD_SET(m_socket, &rfds);
		FD_SET(m_socket, &efds);

		tv.tv_sec = ReadTimeout;
		tv.tv_usec = 0;
		
		if(select(m_socket + 1, &rfds, NULL, &efds, &tv) <= 0)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONNECTION_LOST);
			return false;
		}
		// Did we get an error on the socket ?
		if(FD_ISSET(m_socket, &efds))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONNECTION_LOST);
			return false;
		}
	}
	
	if( !(lresponse = d2i_ADMIN_RESPONSE_bio(connection, NULL)) )
	{
#ifdef _DEBUG
		ERR_print_errors_fp(stdout);
#endif
		ERR_clear_error();
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONNECTION_LOST);
		return false;
	}

	if(!response.load_Datas(lresponse))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)lresponse, AdminResponse::get_ASN1_ITEM());
		response.Clear();
		return false;
	}

	//Do we have the lresponse signer cert yet ?
	if(!response.get_signingCert())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_RESPONSE_SIGNATURE);
		response.Clear();
		ASN1_item_free((ASN1_VALUE*)lresponse, AdminResponse::get_ASN1_ITEM());
		return false;
	}
	if(!m_entity_cert)
	{
		m_entity_cert = response.get_signingCert();
	}
	else
	{
		// Is the signing cert the same ?
		if(!(response.get_signingCert() == m_entity_cert))
		{
			//NO: that's an error, the cert can't change during a session !
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_RESPONSE_SIGNATURE);
			response.Clear();
			ASN1_item_free((ASN1_VALUE*)lresponse, AdminResponse::get_ASN1_ITEM());
			return false;
		}
	}

	//Let's verify the response
	if(ASN1_item_verify(AdminResponseBody::get_ASN1_ITEM(), lresponse->sig_algo, lresponse->signature, (char*)lresponse->body, (EVP_PKEY*)m_entity_cert.GetPublicKey()) <= 0 )
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		response.Clear();
		ASN1_item_free((ASN1_VALUE*)lresponse, AdminResponse::get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)lresponse, AdminResponse::get_ASN1_ITEM());


	return true;
}

const PKI_CERT & Connection::get_EntityCert() const
{
	return m_entity_cert;
}
