/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_CSR.cpp: implementation of the PKI_CSR class.
//
//////////////////////////////////////////////////////////////////////

#include "PKI_CSR.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PKI_CSR PKI_CSR::EmptyInstance;

PKI_CSR::PKI_CSR(const PKI_CSR & other)
{
	Reset();
	*this = other;
}

PKI_CSR::PKI_CSR()
{
	Reset();
}

PKI_CSR::PKI_CSR(const char * PEM_CSR)
{
	Reset();
	if(!SetCSR(PEM_CSR))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
}

PKI_CSR::PKI_CSR(const HashTable_Dn & RequestDN, const PKI_RSA & mRsaKey)
{
	Reset();
	if(!GenerateCSR(RequestDN, mRsaKey))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		Clear();
		throw ExceptionNewPKI();
	}
}

PKI_CSR::~PKI_CSR()
{
	Clear();
}

bool PKI_CSR::MakeRequest(const HashTable_Dn & RequestDN)
{
	X509_NAME * subj;
	if (!X509_REQ_set_version(m_csr,0L))  /* version 1 */
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GENERATE_CSR);
		return false;
	}

	subj = X509_REQ_get_subject_name(m_csr);
	if(!subj)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GENERATE_CSR);
		return false;
	}

	if(!RequestDN.To_X509_NAME(subj))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	return true;
}


bool PKI_CSR::X509ReqToString()
{

	int dsize=0;
	unsigned char *p,*data=NULL;
	int dataslen;
	PEM_DER pem_cvrt;
	char * tmpPEM;
	
	if ((dsize=i2d_X509_REQ(m_csr,NULL)) < 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_X509REQ_CSR);
		return false;
	}

	/* dzise + 8 bytes are needed */
	data=(unsigned char *)malloc(dsize+20);
	if (!data)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	p=data;
	if((dsize=i2d_X509_REQ(m_csr,&p))<0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_X509REQ_CSR);
		return false;
	}
	if(!pem_cvrt.Der2Pem((char *)data, dsize, &tmpPEM, &dataslen))
	{
		free(data);
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_pemCsr = tmpPEM;
	free(tmpPEM);
	free(data);
	return true;
}

bool PKI_CSR::StringToX509Req(const char * pem)
{

	int dsize=0;
	unsigned char *data=NULL;
	unsigned char *tmp_data=NULL;
	PEM_DER der_cvrt;
	BIO * sout;

	if(*pem == '-')
	{
		sout = BIO_new_mem_buf((char*)pem, strlen(pem));
		if(!sout)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
		if(!PEM_read_bio_X509_REQ(sout, &m_csr, NULL, NULL))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_CSR_X509REQ);
			BIO_free_all(sout);
			return false;
		}
		BIO_free_all(sout);
		return true;
	}

	if(!der_cvrt.Pem2Der(pem, strlen(pem), (char **)&data, &dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	tmp_data=data;
	if(!d2i_X509_REQ(&m_csr,&data,dsize))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_CONVERT_CSR_X509REQ);
		free(tmp_data);
		return false;
	}

	free(tmp_data);
	return true;
}

const mString & PKI_CSR::GetPemCSR() const
{
	return m_pemCsr;
}

bool PKI_CSR::LoadDN()
{
	X509_NAME * name;
	name=X509_REQ_get_subject_name(m_csr);
	if(!name)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!m_dnCsr.From_X509_NAME(name))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	return true;
}

const PKI_RSA & PKI_CSR::GetRsaKey() const
{
	return m_keyCsr;
}

const HashTable_Dn & PKI_CSR::GetRequestDN() const
{
	return m_dnCsr;
}

X509_REQ * PKI_CSR::GetX509_REQ(bool duplicate) const
{
	if(duplicate)
	{
		CRYPTO_add(&m_csr->references, 1, CRYPTO_LOCK_X509_REQ);
	}
	return m_csr;
}

bool PKI_CSR::CheckSignature() const
{
	if(!m_csr)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(X509_REQ_verify(m_csr, m_pubKeyCsr) < 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	else
	{
		return true;
	}
}

bool PKI_CSR::SetCSR(const char *PEM_CSR)
{
	Clear();
	
	//We test the parameters
	if(!PEM_CSR)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	m_csr=X509_REQ_new();
	if(!m_csr)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!StringToX509Req(PEM_CSR))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!LoadAll())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool PKI_CSR::GenerateCSR(const HashTable_Dn & RequestDN, const PKI_RSA & mRsaKey)
{
	Clear();

	const EVP_MD * digest;

	//We test the parameters
	if(!mRsaKey)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	
	m_csr=X509_REQ_new();
	if(!m_csr)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!MakeRequest(RequestDN))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!X509_REQ_set_pubkey(m_csr, (EVP_PKEY*)mRsaKey.GetRsaKey()))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GENERATE_CSR);
		return false;
	}
	digest=(EVP_MD *)EVP_md5();
	if(!digest)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GENERATE_CSR);
		return false;
	}
	if(!X509_REQ_sign(m_csr, (EVP_PKEY*)mRsaKey.GetRsaKey(), digest))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_GENERATE_CSR);
		return false;
	}
	m_keyCsr = mRsaKey;

	if(!LoadAll())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

void PKI_CSR::Reset()
{
	m_pubKeyCsr = NULL;
	m_csr = NULL;
}

void PKI_CSR::Clear()
{
	m_keyCsr.Clear();
	if(m_csr)
		X509_REQ_free(m_csr);
	if(m_pubKeyCsr)
		EVP_PKEY_free(m_pubKeyCsr);
	m_extsCsr.Clear();
	Reset();
}

bool PKI_CSR::SetCSR(const X509_REQ *x509csr)
{
	//We test the parameters
	CRYPTO_w_lock(CRYPTO_LOCK_X509_REQ);
	if(!x509csr)
	{
		CRYPTO_w_unlock(CRYPTO_LOCK_X509_REQ);
		Clear();
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	// We ignore the same certificate
	if(m_csr == x509csr)
	{
		CRYPTO_w_unlock(CRYPTO_LOCK_X509_REQ);
		return true;
	}
	((X509_REQ *)x509csr)->references++;
	CRYPTO_w_unlock(CRYPTO_LOCK_X509_REQ);

	Clear();

	m_csr = (X509_REQ *)x509csr;
	if(!LoadAll())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const HashTable_String & PKI_CSR::GetExtensions() const
{
	return m_extsCsr.GetExts();
}

const EVP_PKEY * PKI_CSR::GetPublicKey() const
{
	return m_pubKeyCsr;
}

const X509_PUBKEY * PKI_CSR::GetX509_PUBKEY() const
{
	return m_csr->req_info->pubkey;
}

const char * PKI_CSR::GetSignatureAlg() const
{
	int nid;
	nid = OBJ_obj2nid(m_csr->sig_alg->algorithm);
	if(nid == NID_undef) return NULL;
	return (char*)OBJ_nid2sn(nid);
}

unsigned long PKI_CSR::GetVersion() const
{
	return X509_REQ_get_version(m_csr);
}

unsigned long PKI_CSR::GetKeyLen() const
{
	if(!m_pubKeyCsr) return 0;
	return BN_num_bits(m_pubKeyCsr->pkey.rsa->n);;
}

bool PKI_CSR::load_Datas(const X509_REQ *x509csr)
{
	return SetCSR(x509csr);
}

bool PKI_CSR::give_Datas(X509_REQ **Csr) const
{
	if(*Csr)
	{
		X509_REQ_free(*Csr);
	}
	if(!m_csr)
	{
		*Csr = NULL;
	}
	else
	{
		*Csr = GetX509_REQ(true);
		if(!*Csr)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_PARAM);
			return false;
		}
	}
	return true;
}

bool PKI_CSR::operator=( const PKI_CSR & other)
{
	// Trying to copy myself on me
	if(&other.m_csr == &m_csr)
		return true;
	Clear();
	if(!other.m_csr)
		return false;

	CRYPTO_add(&other.m_csr->references, 1, CRYPTO_LOCK_X509_REQ);
	m_csr = other.m_csr;

	if( !(m_pubKeyCsr = X509_REQ_get_pubkey(m_csr)) )
	{
		Clear();
		return false;
	}

	m_pemCsr = other.m_pemCsr;
	m_dnCsr = other.m_dnCsr;
	m_keyCsr = other.m_keyCsr;
	m_extsCsr = other.m_extsCsr;
	return true;
}

PKI_CSR::operator int() const
{
	if(m_csr)
		return 1;
	else
		return 0;
}

bool PKI_CSR::LoadAll()
{
	STACK_OF(X509_EXTENSION) * exts;

	if(!X509ReqToString())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!LoadDN())
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	m_pubKeyCsr = X509_PUBKEY_get(m_csr->req_info->pubkey);
	if(!m_pubKeyCsr)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	exts = X509_REQ_get_extensions(m_csr);
	if(exts)
	{
		if(!m_extsCsr.Load(exts))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
			return false;
		}
		sk_X509_EXTENSION_pop_free(exts, X509_EXTENSION_free);
	}
	return true;
}
