/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

#ifndef SESSIONSCACHE_H
#define SESSIONSCACHE_H

#include "NewPKI.h"
#include "CriticalSection.h"
#include <openssl/ssl.h>
#include <map>
#include <mString.h>


/*!
	This class represents a SSL session
*/
class SslSession
{
public:
	/*! \brief This is the constructor.
	 *  \param other [IN] The other session to copy from.
	 */
	SslSession(const SslSession & other);

	/*! \brief This is the constructor.
	 */
	SslSession();

	/*! \brief This is the destructor.
	 */
	~SslSession();

	/*! \brief This is the constructor.
	 *  \param other [IN] The other session to copy from.
	 */
	bool operator=(const SslSession & other);

	/*! \brief This function sets the session.
	 *  \param sess [IN] The session.
	 */
	void set_session(SSL_SESSION * sess);

	/*! \brief This function returns the session.
	 *  \return The session.
	 */
	SSL_SESSION * get_session() const;

	/*! \brief This function returns the session ID in its string form.
	 *  \param sess [IN] The session.
	 *  \return The session ID.
	 */
	static const char * id2sz(SSL_SESSION * sess);

private:
	SSL_SESSION * m_sess;
};

/*!
	This class represents a SSL sessions cache for SSL connections
*/
class SessionsCache
{
public:
	/*! \brief This is the constructor.
	 */
	SessionsCache();

	/*! \brief This is the constructor.
	 */
	~SessionsCache();

	/*! \brief This function adds a session to the cache.
	 *  \param key [IN] The key.
	 *  \param sess [IN] The session to add.
	 */
	void add_session(const mString & key, SSL_SESSION * sess);

	/*! \brief This function gets a session from the cache.
	 *  \param key [IN] The key.
	 *  \return The session.
	 */
	SSL_SESSION * get_session(const mString & key);
	
	/*! \brief This function deletes a session from the cache.
	 *  \param key [IN] The key.
	 */
	void delete_session(const mString & key);
	
	/*! \brief This function generates a session key.
	 *  \param id [OUT] The key.
	 *  \param Server [IN] The server the session comes from.
	 *  \param dn [IN] The optional dn from the client connection.
	 *  \param Port [IN] The port connected to.
	 */
	void generate_id(mString & id, const mString & Server, const mString & dn, unsigned short Port);
	
	/*! \brief This function returns the total number of sessions that were in the cache.
	 *  \return The number of session.
	 */
	unsigned long get_numberOfSessions();
	
	/*! \brief This function empties the cache.
	 */
	void clean();
private:
	map<mString, SslSession> m_cache;
	CriticalSection m_cacheLock;
	unsigned long m_sessionsCtr;
};

#endif
