/*
 *  This file is part of the nfdump project.
 *
 *  Copyright (c) 2004, SWITCH - Teleinformatikdienste fuer Lehre und Forschung
 *  All rights reserved.
 *  
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions are met:
 *  
 *   * Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright notice, 
 *     this list of conditions and the following disclaimer in the documentation 
 *     and/or other materials provided with the distribution.
 *   * Neither the name of SWITCH nor the names of its contributors may be 
 *     used to endorse or promote products derived from this software without 
 *     specific prior written permission.
 *  
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 *  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 *  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 *  POSSIBILITY OF SUCH DAMAGE.
 *  
 *  $Author: peter $
 *
 *  $Id: nfx.c 101 2008-08-15 11:38:08Z peter $
 *
 *  $LastChangedRevision: 101 $
 *	
 */

#include "config.h"

#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif

#ifndef DEVEL
#   define dbg_printf(...) /* printf(__VA_ARGS__) */
#else
#   define dbg_printf(...) printf(__VA_ARGS__)
#endif

#include "nffile.h"
#include "nfx.h"

/* global vars */

/*
 * see nffile.h for detailed extension description
 */
extension_descriptor_t extension_descriptor[] = {
	// fill indices 0 - 3
	{ 0,	0,	 0, 1,   "Required extension"},
	{ 1,	0,	 0, 1,   "Required extension"},
	{ 2,	0,	 0, 1,   "Required extension"},
	{ 3,	0,	 0, 1,   "Required extension"},

	// the optional extension
	{ 4, 	4, 	 1, 0,   "2 byte input/output interface index"},
	{ 5, 	8, 	 1, 0,   "4 byte input/output interface index"},
	{ 6, 	4, 	 2, 0,   "2 byte src/dst AS number"},
	{ 7, 	8, 	 2, 0,   "4 byte src/dst AS number"},
	{ 8, 	4, 	 3, 0,   "dst tos, direction, src/dst mask"}, 
	{ 9,	4,	 4, 0,   "IPv4 next hop"},
	{ 10,	16,	 4, 0,   "IPv6 next hop"},
	{ 11,	4,	 5, 0,   "IPv4 BGP next IP"},
	{ 12,	16,	 5, 0,   "IPv6 BGP next IP"},
	{ 13,	4,	 6, 0,   "src/dst vlan id"},
	{ 14,	4,	 7, 0,   "4 byte output packets"},
	{ 15,	8,	 7, 0,   "8 byte output packets"},
	{ 16,	4,	 8, 0,   "4 byte output bytes"},
	{ 17,	8,	 8, 0,   "8 byte output bytes"},
	{ 18,	4,	 9, 0,   "4 byte aggregated flows"},
	{ 19,	8,	 9, 0,   "8 byte aggregated flows"},
	{ 20,	16,	10, 0,   "in src/out dst mac address"},
	{ 21,	16,	11, 0,   "in dst/out src mac address"},

	// last entry
	{ 0,	0,	0, 0,	NULL }
};

uint32_t Max_num_extensions;

int Insert_Extension_Map(extension_map_list_t *extension_map_list, extension_map_t *map) {
uint32_t next_free = extension_map_list->next_free;
uint16_t map_id;

	map_id = map->map_id == INIT_ID ? 0 : map->map_id & EXTENSION_MASK;
	map->map_id = map_id;
dbg_printf("Insert Extension Map:\n");
	// is this slot free
	if ( extension_map_list->slot[map_id] ) {
dbg_printf("Map %d already exists\n", map_id);
		// no - check if same map already in slot
		if ( extension_map_list->slot[map_id]->map->size == map->size ) {
			// existing map and new map have the same size 
dbg_printf("New map same size:\n");

			// we must compare the maps
			int i = 0;
			while ( extension_map_list->slot[map_id]->map->ex_id[i] && (extension_map_list->slot[map_id]->map->ex_id[i] == map->ex_id[i]) ) 
				i++;

			// if last entry == 0 => last map entry => maps are the same
			if ( extension_map_list->slot[map_id]->map->ex_id[i] == 0 ) {
dbg_printf("Same map => nothing to do\n");
				// same map
				return 0;
			} else  { 
				// new map is different but has same id - move it to some other place, which is currently free
				if ( next_free < MAX_EXTENSIONS ) {
					extension_map_list->slot[next_free]   	= extension_map_list->slot[map_id];
					extension_map_list->slot[map_id] 		= NULL;
					extension_map_list->slot[next_free]->map->map_id = next_free;
dbg_printf("Move existing map from slot %d to slot %d\n",map_id, next_free);
				} else {
					fprintf(stderr, "Extension map list exchausted - too many extension maps ( > %d ) to process;\n", MAX_EXTENSIONS);
					exit(255);
				}
			}
		} else { // new map has different size => different map with same map_id - move it to some other place, which is currently free
			if ( next_free < MAX_EXTENSIONS ) {
dbg_printf("New map same ID: Move existing map from slot %d to slot %d\n",map_id, next_free);
				extension_map_list->slot[next_free] 	= extension_map_list->slot[map_id];
				extension_map_list->slot[map_id] 		= NULL;
				extension_map_list->slot[next_free]->map->map_id = next_free;
			} else {
				fprintf(stderr, "Extension map list exchausted - too many extension maps ( > %d ) to process;\n", MAX_EXTENSIONS);
				exit(255);
			}
		}
	}

	// add new entry to slot
	extension_map_list->slot[map_id]	= (extension_info_t *)calloc(1,sizeof(extension_info_t));
	if ( !extension_map_list->slot[map_id] ) {
		fprintf(stderr, "malloc() error in %s line %d: %s\n", __FILE__, __LINE__, strerror(errno) );
		exit(255);
	}
	extension_map_list->slot[map_id]->map   = (extension_map_t *)malloc((ssize_t)map->size);
	if ( !extension_map_list->slot[map_id]->map ) {
		fprintf(stderr, "malloc() error in %s line %d: %s\n", __FILE__, __LINE__, strerror(errno) );
		exit(255);
	}
	memcpy((void *)extension_map_list->slot[map->map_id]->map, (void *)map, map->size);
	BuildExtensionCache(extension_map_list->slot[map->map_id]);

	// Update next_free slot, if it's used now
	while ( extension_map_list->slot[next_free] && (next_free < MAX_EXTENSIONS))
		next_free++;
	extension_map_list->next_free = next_free;

	// if all slots are exhausted next_free is now MAX_EXTENSIONS. The next time an empty slot is needed, it will properly fail.
dbg_printf("Installed map in slot %d.Next free: %d\n", map_id, next_free);
	
	//map changed
	return 1;

} // End of Insert_Extension_Map

void BuildExtensionCache(extension_info_t *extension) {
int i, id;

	memset((void *)extension->cache, 0, 256);

	// unused entry
	extension->cache[0] = 0;
	// required extensions always exists
	extension->cache[1] = extension->cache[2] = extension->cache[3] = 1;

	// fill optional extension ids into cache
	i = 0;
	while ( ( id = extension->map->ex_id[i++]) != 0  ) {
		extension->cache[id] = 1;
	}

} // End of BuildExtensionCache

void SetupExtensionDescriptors(char *options) {
int i, *mask;
char *p, *q, *s;

	Max_num_extensions = 0;
	i = 1;
	while ( extension_descriptor[i++].id ) 
		Max_num_extensions++;

	mask = (int *)calloc(Max_num_extensions+1, sizeof(int));
	if ( !mask ) {
		fprintf(stderr, "malloc() error in %s line %d: %s\n", __FILE__, __LINE__, strerror(errno) );
		exit(255);
	}

	s = (char *)malloc(strlen(options));
	if ( !s ) {
		fprintf(stderr, "malloc() error in %s line %d: %s\n", __FILE__, __LINE__, strerror(errno) );
		exit(255);
	}
	q = s;
	*q = '\0';
	p = options;
	while ( *p ) {
		if ( !isspace(*p) )
			*q++ = *p;
		p++;
	}
	*q = '\0';

	p = s;
	while ( p && *p ) {
		int sign;
		q = strchr(p, ',');
		if ( q )
			*q++ = '\0';
		
		// get possible sign
		sign = 1;
		if ( *p == '-' ) {
			sign = -1;
			p++;
		}
		if ( *p == '+' ) {
			sign = 1;
			p++;
		}

		switch ( *p ) {
			case '\0':
				fprintf(stderr, "Extension format error: Unexpected end of format.\n");
				exit(255);
				break;
			case '*': 
				for (i=4; i<= Max_num_extensions; i++ ) 
					extension_descriptor[i].enabled = sign == 1 ? : 0;
				break;
			default: {
				int i = strtol(p, NULL, 10);
				if ( i == 0 ) {
					fprintf(stderr, "Extension format error: Unexpected string: %s.\n", p);
					exit(255);
				}
				if ( i > Max_num_extensions ) {
					fprintf(stderr, "Extension format error: Invalid extension: %i\n", i);
					exit(255);
				}
				mask[i] = sign;
			}
				
		}
		p = q;
	}
	for (i=4; i<= Max_num_extensions; i++ ) {
		int ui = extension_descriptor[i].user_index;

		// mask[ui] == 0 means no input from user -> default behaviour or already overwritten by '*' 
		if ( mask[ui] < 0 )
			extension_descriptor[i].enabled = 0;
		if ( mask[ui] > 0 ) {
			extension_descriptor[i].enabled = 1;
		}
		if ( extension_descriptor[i].enabled ) 
			printf("Add extension: %s\n", extension_descriptor[i].description);
	}

	free(mask);

} // End of SetupExtensionDescriptors

void PrintExtensionMap(extension_map_t *map) {
int i;

	printf("Extension Map:\n");
	printf("  Map ID   = %u\n", map->map_id);
	printf("  Map Size = %u\n", map->size);
	printf("  Ext Size = %u\n", map->extension_size);
	i=0;
	while (map->ex_id[i]) {
		int id = map->ex_id[i++];
		printf("  Index %3i, ext %3u = %s\n", extension_descriptor[id].user_index, id, extension_descriptor[id].description );
	}
	printf("\n");

} // End of PrintExtensionMap


