/*
 ** Copyright (C) 2002-2008 INL
 ** Written by Eric Leblond <eric@regit.org>
 **            Vincent Deffontaines <vincent@gryzor.com>
 **            Pierre Chifflier <chifflier@inl.fr>
 ** INL http://www.inl.fr/
 **
 ** $Id: tls.c 5459 2008-12-22 13:48:45Z regit $
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, version 2 of the License.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "nufw.h"
#include <time.h>
#include <gnutls/x509.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

/**
 * \file nufw/tls.c
 * \brief Create a TLS connection to NuAuth
 *
 * Create a TLS connection to NuAuth using tls_connect().
 */

static unsigned int tls_cert_verify(gnutls_session *tls_session, char *nuauth_cert_dn);
static unsigned int check_nuauth_cert_dn(gnutls_session *tls_session);
static unsigned int check_nuauth_cert_hostname(gnutls_session *tls_session);
extern int init_x509_filenames(void);

/* read the file indicated (by na1me) in the fname parameter.  store
   its entire contents in a single datum. */
int set_datum_file(gnutls_datum_t* d, const char* fname) {
	struct stat sbuf;
	unsigned char* c = NULL;
	FILE* file = NULL;
	size_t x = 0;

	if (0 != stat(fname, &sbuf)) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "Unable to access '%s'\n", fname);
		return -1;
	}

	c = gnutls_realloc(d->data, sbuf.st_size);
	if (NULL == c) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "failed to allocate %ld bytes for '%s'\n", (long)sbuf.st_size, fname);
		return -1;
	}

	d->data = c;
	d->size = sbuf.st_size;

	file = fopen(fname, "r");
	if (NULL == file) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "failed to open '%s' for reading\n",  fname);
		return -1;
	}

	x = fread(d->data, d->size, 1, file);
	if (x != 1) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "tried to read %d bytes, read %d instead from '%s'\n", d->size, x, fname);
		fclose(file);
		return -1;
	}

	fclose(file);
	return 0;
}


static int check_crl_validity(const char *crl_file, const char *ca_file)
{
	gnutls_x509_crt_t ca;
	gnutls_x509_crl_t crl;
	gnutls_datum_t datum_crl;
	gnutls_datum_t datum_ca;
	time_t t, now;
	int ret;
	int return_value;
	char buffer[256];
	size_t s;


	log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "Checking CRL file %s against %s", crl_file, ca_file);
	if (!ca_file || !crl_file)
		return -1;

	datum_ca.data = NULL;
	datum_crl.data = NULL;

	ret = set_datum_file(&datum_crl, crl_file);
	if (ret) {
	        log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "TLS: Could not read CRL file %s", crl_file);
		return -1;
	}
	ret = set_datum_file(&datum_ca, ca_file);
	if (ret) {
	        log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "TLS: Could not read CA file %s", ca_file);
		gnutls_free(datum_crl.data);
		return -1;
	}

	gnutls_x509_crt_init(&ca);
	gnutls_x509_crl_init(&crl);

	ret = gnutls_x509_crl_import(crl, &datum_crl, GNUTLS_X509_FMT_PEM);
	if (ret) {
	        log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "TLS: Could not import CRL data");
		gnutls_free(datum_ca.data);
		gnutls_free(datum_crl.data);
		return -1;
	}

	ret = gnutls_x509_crt_import(ca, &datum_ca, GNUTLS_X509_FMT_PEM);
	if (ret) {
	        log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "TLS: Could not import CA data");
		gnutls_free(datum_ca.data);
		gnutls_free(datum_crl.data);
		return -1;
	}


	/* debug stuff */
	s = sizeof(buffer);
	ret = gnutls_x509_crl_get_issuer_dn(crl, buffer, &s);
	log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "TLS: CRL issuer DN: %s", buffer);

	/* Check if CRL was signed by configured CA */
	return_value = 0;
	ret = gnutls_x509_crl_check_issuer (crl, ca);
	if (ret != 1) {
	        log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "TLS: CRL issuer is NOT the configured certificate authority", ret);
		return_value--;
	}

	/* Check if CRL has expired */
	now = time(NULL);
	t = gnutls_x509_crl_get_next_update (crl);
	/* This field is optional in a CRL */
	if (t != (time_t)-1 ) {
		if (now > t) {
			log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING, "TLS: CRL has expired and should be re-issued");
			return_value--;
		}
	}

	gnutls_x509_crt_deinit(ca);
	gnutls_x509_crl_deinit(crl);
	gnutls_free(datum_ca.data);
	gnutls_free(datum_crl.data);

	return return_value;
}

void tls_maintenance()
{
	int ret;
	if (crl_file && ! invalid_tls_params) {
		if (check_crl_validity(crl_file, ca_file)) {
			log_area_printf(DEBUG_AREA_MAIN,
					DEBUG_LEVEL_CRITICAL,
					"TLS: invalid CRL file %s", crl_file);
			if (invalid_tls_params == 0) {
				pthread_mutex_lock(&tls.mutex);
				shutdown_tls();
				invalid_tls_params = 1;
				pthread_mutex_unlock(&tls.mutex);
			}
			return;
		} else {
			ret = gnutls_certificate_set_x509_crl_file(tls.xcred,
					crl_file,
					GNUTLS_X509_FMT_PEM);
			if (ret < 0) {
				log_area_printf(DEBUG_AREA_MAIN,
						DEBUG_LEVEL_CRITICAL,
						"TLS: can not set gnutls crl file: %s",
						gnutls_strerror(ret));
				if (invalid_tls_params == 0) {
					pthread_mutex_lock(&tls.mutex);
					shutdown_tls();
					invalid_tls_params = 1;
					pthread_mutex_unlock(&tls.mutex);
				}
			}
		}
	}
}


/**
 * Perform certificate checks: Logs for each check and
 * return the global check result.
 *
 * Returns 1 on error, 0 on success.
 */
unsigned int tls_cert_verify(gnutls_session *tls_session, char *nuauth_cert_dn)
{

	int ret;
	unsigned int status = 0;
	time_t now;

	/* we need to verify received certificates */
	ret = gnutls_certificate_verify_peers2(*tls_session, &status);
	if (ret < 0) {
		log_area_printf(DEBUG_AREA_GW,
				DEBUG_LEVEL_WARNING,
				"TLS: Certificate authority verification failed: %s",
				gnutls_strerror(ret));
		return 1;
	}
	if (status) {
		char buffer[200];

		buffer[0] = 0;
		if (status & GNUTLS_CERT_INVALID)
			SECURE_STRNCAT(buffer, " invalid.", sizeof(buffer));
		if (status & GNUTLS_CERT_REVOKED)
			SECURE_STRNCAT(buffer, " revoked.", sizeof(buffer));
		if (status & GNUTLS_CERT_SIGNER_NOT_FOUND)
			SECURE_STRNCAT(buffer, " signer not found.", sizeof(buffer));
		if (status & GNUTLS_CERT_SIGNER_NOT_CA)
			SECURE_STRNCAT(buffer, " signer not a CA.", sizeof(buffer));
#ifdef GNUTLS_CERT_INSECURE_ALGORITHM
		if (status & GNUTLS_CERT_INSECURE_ALGORITHM)
			SECURE_STRNCAT(buffer, " use of insecure algorithm.", sizeof(buffer));
#endif

		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"TLS: Invalid certificates received from nuauth server:%s",
				buffer);
		return 1;
	}

	now = time(NULL);
	if (gnutls_certificate_activation_time_peers(*tls_session) > now) {
		char buffer[200];
		buffer[0] = 0;

		SECURE_STRNCAT(buffer, " server certificate not yet activated", sizeof(buffer));
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"TLS: Invalid certificates received from nuauth server:%s",
				buffer);
		return 1;
	}
	if (gnutls_certificate_expiration_time_peers(*tls_session) < now) {
		char buffer[200];
		buffer[0] = 0;

		SECURE_STRNCAT(buffer, " server certificate expired", sizeof(buffer));
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"TLS: Invalid certificates received from nuauth server:%s",
				buffer);
		return 1;
	}

	if (nuauth_cert_dn) {
		if (!check_nuauth_cert_dn(tls_session))
			return 1;
	}

	if (check_nuauth_cert_hostname(tls_session))
		return 1;

	return 0;
}

/**
 * Check that nuauth hostname matches hostname from certificate
 *
 * Returns 1 on error, 0 if the domain name is valid.
 */
unsigned int check_nuauth_cert_hostname(gnutls_session *tls_session)
{
	/* we check that dn provided in nuauth certificate is valid */
	char certbuf[256];
	size_t size;
	int ret;

	const gnutls_datum *cert_list;
	unsigned int cert_list_size = 0;
	gnutls_x509_crt x509_cert;

	/* This function only works for X.509 certificates.
	*/
	if (gnutls_certificate_type_get(*tls_session) != GNUTLS_CRT_X509)
		return 1;

	cert_list = gnutls_certificate_get_peers(*tls_session, &cert_list_size);
	if (cert_list_size == 0) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"TLS: could not extract certificate from session\n");
		return 1;
	}

	gnutls_x509_crt_init(&x509_cert);
	gnutls_x509_crt_import(x509_cert, &cert_list[0], GNUTLS_X509_FMT_DER);

	size = sizeof(certbuf);
	ret = gnutls_x509_crt_get_dn_by_oid(x509_cert, GNUTLS_OID_X520_COMMON_NAME,
			0, /* the first and only one */
			0,
			certbuf,
			&size);

	if (ret) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"TLS: error fetching CN from cert:%s\n",
				gnutls_strerror(ret));
		gnutls_x509_crt_deinit(x509_cert);
		return 1;
	}

	/* This function will check if the given certificate's subject matches the
	 * given hostname. This is a basic implementation of the matching described
	 * in RFC2818 (HTTPS), which takes into account wildcards, and the subject
	 * alternative name PKIX extension. Returns non zero on success, and zero on
	 * failure. */
	ret = gnutls_x509_crt_check_hostname(x509_cert, authreq_addr);

	if (!ret) {
		if (nufw_fqdn_check) {
			log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"SSL: certificate subject name (%s) does not match target host name '%s'",
				certbuf, authreq_addr);
			gnutls_x509_crt_deinit(x509_cert);
			return 1;
		}
		else
			log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"SSL: certificate subject name (%s) does not match target host name '%s', but continuing",
				certbuf, authreq_addr);
	}
	else
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_INFO,
				"TLS common name: %s (ok)", certbuf);


	gnutls_x509_crt_deinit(x509_cert);
	return 0;
}

/**
 * Check nuauth certification domain name (DN).
 *
 * Returns 1 on error, 0 if the domain name is valid.
 */
unsigned int check_nuauth_cert_dn(gnutls_session *tls_session)
{
	/* we check that dn provided in nuauth certificate is valid */
	char dn[128];
	size_t size;
	int ret;

	const gnutls_datum *cert_list;
	unsigned int cert_list_size = 0;
	gnutls_x509_crt cert;

	/* This function only works for X.509 certificates.
	*/
	if (gnutls_certificate_type_get(*tls_session) != GNUTLS_CRT_X509)
		return 0;

	cert_list = gnutls_certificate_get_peers(*tls_session, &cert_list_size);
	if (cert_list_size == 0) {
		return 1;
	}

	/* we only print information about the first certificate */
	ret = gnutls_x509_crt_init(&cert);
	if (ret != 0) {
		log_area_printf	(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: cannot init x509 cert: %s",
				gnutls_strerror(ret));
		return 0;
	}

	ret = gnutls_x509_crt_import(cert, &cert_list[0], GNUTLS_X509_FMT_DER);
	if (ret != 0) {
		log_area_printf
			(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
			 "TLS: cannot import x509 cert: %s",
			 gnutls_strerror(ret));
		gnutls_x509_crt_deinit(cert);
		return 0;
	}

	size = sizeof(dn);
	ret = gnutls_x509_crt_get_dn(cert, dn, &size);
	if (ret != 0) {
		log_area_printf (DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
			 "TLS: cannot copy x509 cert name into buffer: %s",
			 gnutls_strerror(ret));
		gnutls_x509_crt_deinit(cert);
		return 0;
	}
	dn[sizeof(dn)-1] = 0;
	if (strcmp(dn, nuauth_cert_dn)) {
		log_area_printf (DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
			 "TLS: bad certificate DN received from nuauth server: %s",
			 dn);
		gnutls_x509_crt_deinit(cert);
		return 0;
	}

	gnutls_x509_crt_deinit(cert);

	return 1;
}


/**
 * Inialialize key_file and cert_file variables
 */
int init_x509_filenames()
{
#if USE_X509
	if (!key_file) {
		key_file = calloc(strlen(CONFIG_DIR) + strlen(KEYFILE) +
					2, sizeof(char));
		if (!key_file) {
			return 0;
		}
		strcat(key_file, CONFIG_DIR);
		strcat(key_file, "/");
		strcat(key_file, KEYFILE);
	}
	if (!cert_file) {
		cert_file = calloc(strlen(CONFIG_DIR) + strlen(CERTFILE) +
					2, sizeof(char));
		if (!cert_file) {
			return 0;
		}
		strcat(cert_file, CONFIG_DIR);
		strcat(cert_file, "/");
		strcat(cert_file, CERTFILE);
	}
#endif
	return 1;
}

int check_key_perms(const char* filename)
{
	struct stat infos;

	if (stat(filename, &infos) != 0) {
		return 0;
	}

	/* File should not be readable or writable by others */
	if (infos.st_mode & S_IROTH || infos.st_mode & S_IWOTH) {
		return 0;
	}

	return 1;
}

/**
 * Create auth server thread
 */

void create_authserver()
{
	pthread_attr_t attr;
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr,
			PTHREAD_CREATE_JOINABLE);

	/* create joinable thread for auth server */
	pthread_mutex_init(&tls.auth_server_mutex, NULL);
	if (pthread_create
			(&tls.auth_server, &attr, authsrv,
			 NULL) == EAGAIN) {
		exit(EXIT_FAILURE);
	}
	tls.auth_server_running = 1;

}

/**
 * Create a TLS connection to NuAuth: create a TCP socket and connect
 * to NuAuth using ::adr_srv.
 *
 * If x509 is enable (USE_X509 equals to 1), create credentials and check
 * NuAuth's one. This function modify the tls variable and in particular
 * set tls.session.
 *
 */
void tls_connect()
{
	gnutls_session *tls_session;
	int tls_socket, ret;
#if USE_X509
	const int cert_type_priority[3] = { GNUTLS_CRT_X509, 0 };

	tls.session = NULL;

	/* compute patch key_file */
	if (!init_x509_filenames()) {
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_DEBUG,
				"Couldn't malloc for key or cert filename!");
		return;
	}

	/* test if key exists */
	if (access(key_file, R_OK)) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"TLS: can not access key file \"%s\"!",
				key_file);
		return;
	}

	if (access(cert_file, R_OK)) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"TLS: can not access cert file \"%s\"!",
				cert_file);
		return;
	}

	/* X509 stuff */
	ret = gnutls_certificate_allocate_credentials(&tls.xcred);
	if (ret != 0) {
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: can not allocate gnutls credentials: %s",
				gnutls_strerror(ret));
		return;
	}

	/* sets the trusted cas file */
	if (ca_file) {
		ret = gnutls_certificate_set_x509_trust_file(tls.xcred,
							     ca_file,
							     GNUTLS_X509_FMT_PEM);
		if (ret < 0) {
			log_area_printf(DEBUG_AREA_MAIN,
					DEBUG_LEVEL_FATAL,
					"TLS: can not set gnutls trust file: %s",
					gnutls_strerror(ret));
			gnutls_certificate_free_credentials(tls.xcred);
			return;
		}
	}

	if (crl_file) {
		if (check_crl_validity(crl_file, ca_file)) {
			log_area_printf(DEBUG_AREA_MAIN,
					DEBUG_LEVEL_CRITICAL,
					"TLS : invalid CRL file %s", crl_file);

			gnutls_certificate_free_credentials(tls.xcred);
			return;
		}
		ret = gnutls_certificate_set_x509_crl_file(tls.xcred,
							   crl_file,
							   GNUTLS_X509_FMT_PEM);
		if (ret < 0) {
			log_area_printf(DEBUG_AREA_MAIN,
					DEBUG_LEVEL_CRITICAL,
					"TLS: can not set gnutls crl file: %s",
					gnutls_strerror(ret));
			gnutls_certificate_free_credentials(tls.xcred);
			return;
		}
	}

	if (!check_key_perms(key_file)) {
		log_area_printf(DEBUG_AREA_MAIN,
				DEBUG_LEVEL_FATAL,
				"ERROR Permissions on private key (%s) are not restrictive enough\n. It should be readable only by nuauth user or group.",
				key_file);
		if (ca_file || nufw_strict_tls) {
			gnutls_certificate_free_credentials(tls.xcred);
			return;
		}
	}

	ret = gnutls_certificate_set_x509_key_file(tls.xcred, cert_file,
						   key_file,
						   GNUTLS_X509_FMT_PEM);
	if (ret < 0) {
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: can not set cert/key file: %s",
				gnutls_strerror(ret));
		gnutls_certificate_free_credentials(tls.xcred);
		return;

	}

	invalid_tls_params = 0;

#endif

	/* Initialize TLS session */
	tls_session = (gnutls_session *) calloc(1, sizeof(gnutls_session));
	if (tls_session == NULL) {
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: can not calloc!");
		gnutls_certificate_free_credentials(tls.xcred);
		return;
	}
	ret = gnutls_init(tls_session, GNUTLS_CLIENT);
	if (ret != 0) {
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: init failed: %s",
				gnutls_strerror(ret));
		gnutls_certificate_free_credentials(tls.xcred);
		free(tls_session);
		return;
	}
	tls_socket = socket(adr_srv->ai_family, adr_srv->ai_socktype,
		   adr_srv->ai_protocol);

	/* connect */
	if (tls_socket <= 0) {
		gnutls_certificate_free_credentials(tls.xcred);
		gnutls_deinit(*tls_session);
		free(tls_session);
		return;
	}

	if (connect(tls_socket, adr_srv->ai_addr, adr_srv->ai_addrlen) == -1) {
		close(tls_socket);
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_CRITICAL,
				"[!] Unable to connect to nuauth: %s.",
				strerror(errno));
		gnutls_certificate_free_credentials(tls.xcred);
		gnutls_deinit(*tls_session);
		free(tls_session);
		return;
	}

	ret = gnutls_set_default_priority(*(tls_session));
	if (ret < 0) {
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: priority setting failed: %s",
				gnutls_strerror(ret));
		gnutls_certificate_free_credentials(tls.xcred);
		gnutls_deinit(*tls_session);
		free(tls_session);
		return;
	}
#if USE_X509
	ret = gnutls_certificate_type_set_priority(*(tls_session),
						   cert_type_priority);
	if (ret < 0) {
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: gnutls_certificate_type_set_priority() failed: %s",
				gnutls_strerror(ret));
		gnutls_certificate_free_credentials(tls.xcred);
		gnutls_deinit(*tls_session);
		free(tls_session);
		return;
	}

	/* put the x509 credentials to the current session */
	ret = gnutls_credentials_set(*(tls_session), GNUTLS_CRD_CERTIFICATE,
				   tls.xcred);
	if (ret < 0) {
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: Failed to configure credentials: %s",
				gnutls_strerror(ret));
		gnutls_certificate_free_credentials(tls.xcred);
		gnutls_deinit(*tls_session);
		free(tls_session);
		return;
	}
#endif

	/* This function returns void */
	gnutls_transport_set_ptr(*tls_session, (gnutls_transport_ptr) tls_socket);

	/* Perform the TLS handshake */
	ret = 0;
	do {
		if (ret != 0) {
			log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_INFO,
					"TLS: gnutls_handshake() ... (last error: %i)",
					ret);
		}
		ret = gnutls_handshake(*tls_session);
	} while (ret < 0 && !gnutls_error_is_fatal(ret));

	if (ret < 0) {
		/* Verify the certificate even if handshake failed, so that we have
		 * a good error message if the cause of the failure is an invalid
		 * certificate
		 */
		(void) tls_cert_verify(tls_session, nuauth_cert_dn);
		gnutls_perror(ret);
		log_area_printf(DEBUG_AREA_MAIN, DEBUG_LEVEL_WARNING,
				"TLS: gnutls_handshake() failed: %s",
				gnutls_strerror(ret));
		gnutls_certificate_free_credentials(tls.xcred);
		gnutls_deinit(*tls_session);
		free(tls_session);
		return;
	}
#if USE_X509

	ret = tls_cert_verify(tls_session, nuauth_cert_dn);
	if (ret && (ca_file || nufw_strict_tls)) {
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_FATAL,
				"TLS: Certificate verification failed, and either NuFW is running under the TLS strict mode, or you provided the CA file as parameter. Cannot authenticate the packet.");
		gnutls_certificate_free_credentials(tls.xcred);
		gnutls_deinit(*tls_session);
		free(tls_session);
		return;
	}

	if ( ! nufw_strict_tls )
		log_area_printf(DEBUG_AREA_GW, DEBUG_LEVEL_WARNING,
				"TLS: The strict mode will be activated by default in next major stable release.");

#endif

	tls.session = tls_session;
	create_authserver();

}
