/* $Id: nws_state.c,v 1.58 2005/05/24 19:38:25 graziano Exp $ */

#include "config_nws.h"

#include <stdio.h>       /* FILE, et al */
#include <errno.h>       /* errno */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>      /* close() unlink() */
#include <string.h>      /* memset() strlen() */
#include <stdlib.h>      /* malloc() */
#if HAVE_DIRENT_H
#	include <dirent.h>
#	define NAMLEN(dirent) strlen((dirent)->d_name)
#else
#	define dirent direct
#	define NAMLEN(dirent) (dirent)->d_namlen
#	if HAVE_SYS_NDIR_H
#		include <sys/ndir.h>
#	endif
#	if HAVE_SYS_DIR_H
#		include <sys/dir.h>
#	endif
#	if HAVE_NDIR_H
#		include <ndir.h>
#	endif
#endif

#include "diagnostic.h"  /* FAIL() LOG() WARN() */
#include "strutil.h"     /* vstrncpy() SAFESTRCPY() */
#include "osutil.h"      /* CurrentTime() */
#include "hosts.h"
#include "host_protocol.h"
#include "nws_state.h"
#ifdef WITH_NETLOGGER
#	include "logging.h"
#endif

#include "nws_cache.h"

/* global variables */
int useCache = -1;
char Dummy_buf[MAX_RECORD_SIZE];
char *memDir = NULL;
#ifndef MODMINUS
#	define MODMINUS(a,b,m) (((a) - (b) + (m)) % (m))
#	define MODPLUS(a,b,m) (((a) + (b)) % (m))
#endif

#define HEADER_FORMAT "%10d %10d %10d\n"
#define HEADER_LEN 33 /* Three 10-digit floats + 2 spaces and 1 new line */
#define OVERHEAD_LEN 33 /* Three 10-digit floats + 3 spaces */
#define STATE_FORMAT "%10.0f %10.0f %10.0f %s\n"

#ifdef WITH_NETLOGGER
#	define STATE_FORMAT_NL "SEQN=%.0f TIMEOUT=%.0f"
#endif

/* 
 * Constructs and returns a file path suitable for storing records for a
 * state named #stateName# in the state directory.
 */
static const char *
FileOfState(const char *stateName) {
	char *c;
	size_t dirLen = strlen(memDir);
	static char returnValue[255 + 1];
				                                                                                
	/* sanity check */
	if (memDir == NULL || stateName == NULL) {
		ERROR("FileOfState: NULL Parameter\n");
		return NULL;
	}

	vstrncpy(returnValue, sizeof(returnValue), 3, memDir, (memDir[dirLen - 1] == '/') ? "" : "/", stateName);
	/* Translate characters that are special to the file system. */
	for(c = returnValue + dirLen; *c != '\0'; c++) {
		if(*c == '/') {
			*c = '_';
		}
	}
	return returnValue;
}


int
InitStateModule(	int cacheNum,
			int memorySize,
			const char *dir) {
	if (cacheNum > 0) {
		RC_entries = cacheNum;
		RCMemorySize = memorySize;
		useCache = 1;
	} else {
		RC_entries = 0;
		RCMemorySize = 0;
		useCache = 0;
	}

	/* this is the default directory in where to store states */
	if (memDir != NULL) {
		FREE(memDir);
	}
	if (dir == NULL || strlen(dir) == 0) {
		WARN("InitStateModule: empty direcotry: using current\n");
		memDir = (char *)MALLOC(3, 1);
		memDir[0] = '.';
		memDir[1] = '\0';
	} else {
		memDir = (char *)MALLOC(strlen(dir) + 2, 1);
		memcpy(memDir, dir, strlen(dir));
		memDir[strlen(dir)] = '\0';
	}

	/* let's set/check our base directory */
	if(memDir[strlen(memDir) - 1] != '/') {
		strcat(memDir, "/");
	}
	if (!MakeDirectory(memDir, 0775, 1)) {
		ABORT1("Unable to use %s as directory\n", memDir);
	}

#ifdef WITH_NETLOGGER
	if (netlogger == 1) {
		useNetLogger = 1;
	}
#endif

	return 1;
}

/*
 * Reads and parses the header record of #fd#, which must already be
 * opened for reading.  Returns the index of the next record to write,
 * index of the oldest record, maximum records, and record size in
 * #head#, #tail#, #fileSize#, and #recSize#, respectively.  Returns 1 if
 * successful, else 0.
 */
static int
ReadHeaderRec(	FILE *fd,
		int *head,
		int *tail,
		int *fileSize,
		size_t *recSize) {
	char firstRecord[MAX_RECORD_SIZE + 1];
	char header[HEADER_LEN + 1];

	/* sanity check */
	if (useCache < 0) {
		ABORT("You need to call InitStateModule before using this function!\n");
	}

	/* let's  look for the header at the beginning of the file */
	if (fseek(fd, 0, SEEK_SET) < 0) {
		DDEBUG1("ReadHeaderRec: Seek failed (%d)\n", errno);
		return 0;
	}
	if(fgets(header, sizeof(header), fd) == NULL || sscanf(header, "%d %d %d", head, tail, fileSize) < 3) {
		DDEBUG("ReadHeaderRec: couldn't find header!\n");
		return 0;
	}

	/* let's get the first record */
	memset(firstRecord, 0, sizeof(firstRecord));
	if(fgets(firstRecord, sizeof(firstRecord), fd) == NULL) {
		WARN("ReadHeaderRec: Unable to determine record size.\n");
		*recSize = HEADER_LEN;
	} else {
		*recSize = strlen(firstRecord);
		if((*recSize <= 0) || (*recSize >= MAX_RECORD_SIZE)) {
			DDEBUG1("ReadHeaderRec: bad record size %d\n", *recSize);
			return 0;
		}
	}

	return(1);
}


/*
 * Reads the #readIndex#'th #recordSize#-long state record from #fd#,
 * which must be opened for reading.  Returns the time stamp, sequence
 * number, and time-out in #timeStamp#, #seqNo#, and #timeOut#,
 * respectively, and the client record in the #dataSize#-long array
 * #data#.  Returns 1 if successful, else 0.
 */
static int
ReadStateRec(	FILE *fd,
		size_t recSize,
		int readIndex,
		double *timeStamp,
		double *seqNo,
		double *timeOut,
		char *data,
		size_t dataSize) {
	char overhead[OVERHEAD_LEN + 1];

	if(fseek(fd, HEADER_LEN + recSize * readIndex, SEEK_SET) < 0) {
		INFO1("ReadStateRec: fseek failed on index %d\n", readIndex);
		return 0;
	}

	/* grab only the overhead; looks like fgets tries to null term it. */
	memset(overhead, 0, sizeof(overhead));
	if((fgets(overhead, sizeof(overhead), fd) == NULL) || (fgets(data, dataSize, fd) == NULL)) {
		INFO1("ReadStateRec: %s\n", feof(fd) ? "EOF" : "read failed");
		return 0;
	}

	sscanf(overhead, "%lf %lf %lf", timeStamp, seqNo, timeOut);
	return(1);
}


/* Takes whatever steps are necessary to close the file #fd#.  */
static void
Shut(FILE *fd) {
	if ((fflush(fd) != 0) || (fclose(fd) != 0)) {
		(void)close(fileno(fd));
	}
}


/*
 * Writes #head#, #tail#, #fileSize#, and #recSize# into the header of
 * #fd#, which must already be opened for writing.  Returns 1 if
 * successful, else 0.
 */
static int
WriteHeaderRec(FILE *fd,
               int head,
               int tail,
               int fileSize,
               size_t recSize) {
	/*
	** TBD: #recSize# is presently ignored, although we really should
	include it ** in the header information, rather than force a read
	of the first record in ** ReadHeaderRec() just to discover the
	record size.
	*/
	if (fseek(fd, 0, SEEK_SET) < 0) {
		FAIL1("WriteHeaderRec: Seek failed (%d)\n", errno);
	}
	fprintf(fd, HEADER_FORMAT, head, tail, fileSize);
	fflush(fd);

	return(1);
}


/*
 * Writes #timeStamp#, #seqNo#, #timeOut#, and #data# as the
 * #writeIndex#'th #recSize#-long record of #fd#, which must already be
 * opened for writing.  Returns 1 if successful, else 0.
 */
static int
WriteStateRec(	FILE *fd,
		size_t recSize,
		int writeIndex,
		double timeStamp,
		double seqNo,
		double timeOut,
		const char *data) {
	int clientSize;
	char padded[MAX_RECORD_SIZE + 1];

	if (fseek(fd, HEADER_LEN + recSize * writeIndex, SEEK_SET) < 0) {
		FAIL1("WriteStateRec: fseek failed on index %d\n", writeIndex);
	}

	clientSize = recSize - OVERHEAD_LEN - 1;
	if (clientSize <= 0) {
		FAIL("WriteStateRec: too short record!\n");
	}
	memcpy(padded, data, clientSize);
	if (recSize < clientSize) {
		memset(padded, ' ', clientSize - recSize);
	}
	padded[clientSize] = '\0';
	fprintf(fd, STATE_FORMAT, timeStamp, seqNo, timeOut, padded);
	fflush(fd);

	return(1);
}


int
ReadState(	const char *fname,
		char *state,
		int count,
		size_t max_size,
		double earliest_seq_no,
		double *out_time_out,
		double *out_seq_no,
		int *out_count,
		int *out_recordsize) {
	char *curr;
	char stateName[255+1];
	char *endOfBuffer;
	int eofIndex;
	FILE *fd;
	int fileHead;
	int fileSize;
	int fileTail;
	double headSeqNo;
	int i;
	double now;
	int readIndex;
	double recordSeqNo;
	size_t recordSize;
	double recordTimeOut;
	double recordTimeStamp;
	RCache r;

	/* sanity check */
	if (useCache < 0) {
		ABORT("You need to call InitStateModule before using this function!\n");
	}

	/* make the compiler happy */
	r = NULL;

	/* let's fet the full path */
	SAFESTRCPY(stateName, FileOfState(fname));

	if (useCache) {
		r = FindRCacheEntry(stateName);
		if(r != NULL) {
			(void)ReadStateCache(r,
				       state,
				       count,
				       max_size,
				       earliest_seq_no,
				       out_time_out,
				       out_seq_no,
				       out_count,
				       out_recordsize);
			return(1);
		} else {
			r = GetRCacheEntry(stateName);
		}
	}

	fd = fopen(stateName, "r");
	if(fd == NULL) {
		INFO2("ReadState cannot open file %s (%d)\n", stateName, errno);
		return 0;
	}

	if(!ReadHeaderRec(fd, &fileHead, &fileTail, &fileSize, &recordSize)) {
		Shut(fd);
		INFO1("ReadState error reading head for file %s\n", stateName);
		return 0;
	}

	memset(state, 0, max_size);
	now = CurrentTime();
	curr = state;
	endOfBuffer = state + max_size;
	headSeqNo = 0.0;
	eofIndex = MODMINUS(fileTail, 1, fileSize);

	for(readIndex = MODMINUS(fileHead, 1, fileSize), i = 0; (readIndex != eofIndex) && (i < count); readIndex = MODMINUS(readIndex, 1, fileSize), i++) {
		if((endOfBuffer - curr) < recordSize) {
			/* We've exhausted the return buffer. */
			break;
		}

		if(!ReadStateRec(fd,
				recordSize,
				readIndex,
				&recordTimeStamp,
				&recordSeqNo,
				&recordTimeOut,
				curr,
				endOfBuffer - curr)) {
			Shut(fd);
			INFO2("ReadState: read failed on file %s, index %d\n", stateName, readIndex);
			return 0;
		}

		if (useCache) {
			/* might be NULL if user chooses no caching */
			if(r != NULL) {
				WriteStateCache(r,
						recordTimeOut,
						recordSeqNo,
						curr,
						endOfBuffer - curr);
			}
		}

		if ((recordTimeStamp + recordTimeOut) < now || recordSeqNo <= earliest_seq_no) {
			/* Dead record or of no interest */
			break;
		}

		if(i == 0) {
			headSeqNo = recordSeqNo;
		}

		/* Drop the trailing newline added by WriteStateRec() 
		 * RICH:
		 * punch in a space so that sprintf and sscanf won't get
		 * confused by the 10 digit time stamp.  When the time
		 * stamp was 9 digits, a space was added automatically */
		curr += strlen(curr) - 1;
		*curr = ' ';
		/* now NULL terminate */
		curr++;
		*curr = '\0';
	}

	*out_count = i;
	*out_recordsize = recordSize - OVERHEAD_LEN;
	*out_seq_no = headSeqNo;
	*out_time_out = recordTimeOut;

	if (useCache) {
		/* might be NULL is user chooses no caching */
		if(r != NULL) {
			for( ; (readIndex != eofIndex); readIndex = MODMINUS(readIndex, 1, fileSize)) {
				if(!ReadStateRec(fd,
						recordSize,
						readIndex,
						&recordTimeStamp,
						&recordSeqNo,
						&recordTimeOut,
						Dummy_buf,
						sizeof(Dummy_buf))) {
					Shut(fd);
					INFO2("ReadState: extra read failed on file %s, index %d\n", stateName, readIndex);
					return 0;
				}
	      
				/* This will put things in timestamp
				 * and/or sequence number order */
				WriteStateCache(r,
					recordTimeOut,
					recordSeqNo,
					Dummy_buf,
					sizeof(Dummy_buf));
			}
		}
	}
	Shut(fd);

	return(1);
}


int
WriteState(	const char *fname,
		size_t flength,
		double time_out,
		double seq_no,
		const char *state,
		size_t stateSize) {
	FILE *fd;
	int eofIndex;
	int fileHead;
	int fileSize;
	int fileTail;
	double now;
	char record[MAX_RECORD_SIZE];
	double recordSeqNo;
	size_t recordSize;
	double recordTimeStamp;
	double recordTimeOut;
	int writeIndex;
	RCache r;
	char stateName[255 + 1];

	/* sanity check */
	if (useCache < 0) {
		ABORT("You need to call InitStateModule before using this function!\n");
	}

	/* let's fet the full path */
	SAFESTRCPY(stateName, FileOfState(fname));

	if(stateSize > MAX_RECORD_SIZE) {
		FAIL2("WriteState: State size %d too big for %s\n", stateSize, stateName);
	}

	fd = fopen(stateName, "r+");
	if (fd == NULL) {
		fd = fopen(stateName, "w+");
		if (fd == NULL) {
			FAIL2("WriteState: File create failed (%d) for %s\n", errno, stateName);
		}
		fileHead = fileTail = 0;
		fileSize = flength;
		recordSize = OVERHEAD_LEN + stateSize + 1;
		if (!WriteHeaderRec(fd, fileHead, fileTail, flength, recordSize)) {
			Shut(fd);
			FAIL1("WriteState: Initial header write failed for %s\n", stateName);
		}
		LOG1("WriteState: Opened new file %s\n", stateName);
	} else if(!ReadHeaderRec(fd, &fileHead, &fileTail, &fileSize, &recordSize)) {
		Shut(fd);
		FAIL1("WriteState cannot get header for file %s\n", stateName);
	}

	now = CurrentTime();
	if (now < seq_no) {
		seq_no = now;
	}

	if (useCache) {
		/* look for the cache entry and if found, add the current
		 * data to it */
		r = FindRCacheEntry(stateName);
		if(r != NULL) {
			WriteStateCache(r,
					time_out,
					seq_no,
					state,
					stateSize);
		}
	}

	eofIndex = MODMINUS(fileTail, 1, fileSize);

  /*
  ** To find where this record should be written, start at the head and walk
  ** back until we hit the tail, timed out data, or a seq_no less than the
  ** incoming one.
  */
  for(writeIndex = MODMINUS(fileHead, 1, fileSize);
      writeIndex != eofIndex;
      writeIndex = MODMINUS(writeIndex, 1, fileSize)) {

    if(!ReadStateRec(fd,
                     recordSize,
                     writeIndex,
                     &recordTimeStamp,
                     &recordSeqNo,
                     &recordTimeOut,
                     record,
                     sizeof(record))) {
      Shut(fd);
      WARN1("WriteState: ReadStateRec %s failed\n", stateName);
      if(unlink(stateName) == 0) {
        return(WriteState(stateName, flength, time_out, seq_no, state, stateSize));
      }
      else {
        FAIL2("WriteState: unlink %s failed with errno %d\n", stateName, errno);
      }
    }

    if(recordSeqNo <= seq_no) {
      break;
    }

    if((recordTimeStamp + recordTimeOut) < now) {
      fileTail = writeIndex;
      break;
    }

  }

  if(writeIndex == MODMINUS(fileHead, 1, fileSize)) {
    /* Appending. */
    writeIndex = fileHead;
    fileHead = MODPLUS(fileHead, 1, fileSize);
    if(fileHead == fileTail) {
      fileTail = MODPLUS(fileTail, 1, fileSize);
    }
  }
  else if(writeIndex == eofIndex) {
    Shut(fd);
    FAIL1("WriteState: old merge attempted on file %s\n",stateName);
  }

  if(!WriteStateRec(fd,
                    recordSize,
                    writeIndex,
                    now,
                    seq_no,
                    time_out,
                    state)) {
    Shut(fd);
    FAIL1("WriteState: write to %s failed\n", stateName);
  }

  if(!WriteHeaderRec(fd, fileHead, fileTail, fileSize, recordSize)) {
    Shut(fd);
    FAIL1("WriteState: header write to %s failed\n", stateName);
  }

  Shut(fd);
  return(1);

}

/* Returns a list from the state directory of files \n separated. It can
 * return NULL */
char *
ReadOldStates() {
#ifdef HAVE_DIRENT_H
	DIR *dir;
	struct dirent *entry;
	unsigned int len;
	double howMany;
	char *tmp;

	/* sanity check */
	if (memDir == NULL) {
		ERROR("ReadOldStates: NULL parameter\n");
		return NULL;
	}

	/* open the directory */
	dir = opendir(memDir);
	if (dir == NULL) {
		ERROR1("ReadOldStates: couldn't open %s\n", memDir);
		return NULL;
	}

	len = 0;
	tmp = MALLOC(1, 0);
	if (tmp == NULL) {
		ERROR("ReadOldStates: out of memory\n");
		closedir(dir);
		return NULL;
	}
	tmp[0] = '\0';

	/* get all entry one by one */
	howMany = 0;
	for (entry = readdir(dir); entry != NULL; entry = readdir(dir)) {
		/* we skip the obvioulsy wrong entries */
		if (entry->d_name[0] == '.') {
			continue;
		}

		/* let's add the name to the be processed list */
		if (tmp[0] != '\0') {
			tmp[len++] = '\n';	/* add a marker for new entry */
			tmp[len] = '\0';
		}
		tmp = (char*)REALLOC(tmp, len + strlen(entry->d_name) + 2, 1);
		memcpy(tmp + len, entry->d_name, strlen(entry->d_name) + 1);
		len += strlen(entry->d_name);
		howMany++;
	}
	INFO1("ReadOldStates: read %.0f old states\n", howMany);
	closedir(dir);

	return tmp;
#else
	/* we don't have a mean to access directory entries */
	ERROR("ReadOldStates: system doesn't have readdir\n");
	return NULL;
#endif
}


/*
 * this function check that file #name# is a good NWS state file. Returns
 * 1 on success, 0 otherwise 
 */
int
CheckFileName(	const char *name) {
	int uno, j, z, i;
	size_t recSize;
	FILE *fd;
	char *tmp;

	/* sanity check */
	if (name == NULL) {
		ERROR("CheckFileName: NULL parameter\n");
		return 0;
	}

	/* let's add the directory name before the name */
	if (memDir != NULL) {
		i = strlen(memDir);
	} else {
		i = 0;
	}
	j = strlen(name);

	tmp = MALLOC(i + j + 1, 0);
	if (tmp == NULL) {
		ERROR("CheckFileName: out of memory\n");
		return 0;
	}
	if (memDir != NULL) {
		memcpy(tmp, memDir, i);
	}
	memcpy(tmp + i, name, j);
	tmp[i + j] = '\0';

	/* well we should use ReadState but given that is too
	 * expensive we have to get our hands dirty */
	uno = 0;
	fd = fopen(tmp, "r");
	if (fd != NULL) {
		uno = ReadHeaderRec(fd, &i, &j, &z, &recSize);
		fclose(fd);
	}
	free(tmp);

	return (uno);
}

/* deletes all the states that have not been accessed within the past
 * #idle# seconds. Returns 1 if succesful.
 */
int
CleanLocalStates(	const char *series,
			unsigned long idle) {
	time_t expiration;
	char *filePath;
	struct stat fileStat;
	int ret;

	if (series == NULL || strlen(series) <= 1) {
		ERROR("CleanLocalStates: NULL parameter!\n");
		return 0;
	}

	ret = 0;
	expiration = (time_t)CurrentTime() - (time_t)idle;
	filePath = MALLOC(strlen(memDir) + strlen(series) + 1, 1);
	sprintf(filePath, "%s%s", memDir, series);
	if(stat(filePath, &fileStat) != 0) {
		WARN1("CleanLocalStates: unable to state file %s\n", filePath);
	} else if(fileStat.st_mtime < expiration) {
		LOG2("CleanLocalStates: deleting %s, last modified %d\n", filePath, fileStat.st_mtime);
		(void)unlink(filePath);
		ret = 1;
	}
	free(filePath);

	return (ret);
}


#ifdef WITH_NETLOGGER
int
WriteStateNL(const char* path,
			 const char* id,
			 size_t flength,
			 double time_out,
			 double seq_no,
			 const char *state,
			 size_t stateSize) {
	struct host_cookie host;
	unsigned short nsPort = 0;
	NLhandle * nlhandle;
	char nlmsg[MAX_NL_MSG_LEN];
	char buffer[MAX_RECORD_SIZE];
	int i,start,end,nfield;
	char c;
	/* decode host:socket and open connection */
	Host2Cookie(path, nsPort, &host);
	printf("WriteStateNL: host.host_name: .%s., host.port:%d\n", host.name, host.port);
	
	sprintf(nlmsg, STATE_FORMAT_NL, seq_no, time_out);
	printf("WriteStateNL: .%s.\n", nlmsg);
	nfield=0;
	i=0;
	while(i<stateSize)
	{
		/* ignore blanks */
		while(state[i]==' ') i++;
		
		start = i;
		/* find next blank or 0x00 */
		while(state[i]!=' ' && state[i]!=0) i++;

		end = i;
		printf("WriteStateNL: %d .%d.\n",end,start);
		/* insert filed */
		if(end-start>0)
		{
			sprintf(buffer," FIELD%d=",nfield);
			strcat(nlmsg,buffer);
			memcpy(buffer,state+start,end-start);
			buffer[end-start]=0;
			strcat(nlmsg,buffer);
			printf("WriteStateNL: %d .%s.\n",end-start,nlmsg);
			nfield++;
		}
	}
	nlhandle = NetLoggerOpen(NL_HOST, (char *)"NWS_MEMORY",
							 (char *)"nws_memory.log", host.name, host.port);
	
	if(nlhandle==NULL)
	{
		LOG2("WriteStateNL: NetLoggerOpen failed host:%s port:%d\n", host.name, host.port);
		return 0;
	}

	printf("WriteStateNL: NetLoggerOpen done\n");
	/* now convert all apha to upper case, and non-alpah-numeric to _ */
	strcpy(buffer,id);
	i=strlen(buffer)-1;
	while(i>=0)
	{
		c= buffer[i];
		if(c>='a' && c<='z') /* lower case */
			buffer[i]=c+'A'-'a';
		else if((c<'A' || c>'Z') && (c<'0' || c>'9')) /* non alpha numeric */
			buffer[i]='_';
		i--;
	}
	/* now format and write */
	NetLoggerWrite(nlhandle, buffer, (char *)"%s", nlmsg);
	
	/* close NL connection */
	NetLoggerClose(nlhandle);
	
	return (1);
}
#endif /* WITH_NETLOGGER */
