/*
 * Permafrost - Physical modelling framework
 *
 * Copyright (C) 2009, 2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <string.h>

#include "src/list.h"
#include "src/util.h"

struct list_elem
  {
	struct list_elem	*next;
	struct list_elem	*prev;
	void			*data;
  };

struct _list
  {
	struct list_elem	*head;
	struct list_elem	*tail;
  };

list_t
list_new()
{
	list_t ret;

	ret = xmalloc(sizeof(struct _list));
	ret->head = ret->tail = NULL;

	return ret;
}

void
list_append(list_t list, void *data)
{
	struct list_elem *elem;

	elem = xmalloc(sizeof(struct list_elem));
	elem->data = data;
	elem->next = NULL;
	if (list->head == NULL)
	  {
		list->head = list->tail = elem;
		elem->prev = NULL;
	  }
	else
	  {
		elem->prev = list->tail;
		list->tail = elem;
		elem->prev->next = elem;
	  }
}

void
list_append_copy(list_t list, void *data, size_t size)
{
	void *p;

	p = xmalloc(size);
	memcpy(p, data, size);
	list_append(list, p);
}

void *
list_pop(list_t list)
{
	void *p;

	if (list->tail == NULL)
		return NULL;

	p = list->tail->data;

	if (list->head == list->tail)
	  {
		free(list->tail);
		list->head = list->tail = NULL;
	  }
	else
	  {
		list->tail = list->tail->prev;
		free(list->tail->next);
		list->tail->next = NULL;
	  }

	return p;
}

void
list_for_each(list_t list, void (*callback)(void *data, void *context),
	      void *context)
{
	struct list_elem *elem;

	for (elem = list->head; elem != NULL; elem = elem->next)
		callback(elem->data, context);
}

void
list_for_each_rev(list_t list, void (*callback)(void *data, void *context),
		  void *context)
{
	struct list_elem *elem;

	for (elem = list->tail; elem != NULL; elem = elem->prev)
		callback(elem->data, context);
}

void *
list_find(list_t list, int (*cmp)(void *d1, void *d2), void *data)
{
	struct list_elem *elem;

	for (elem = list->head; elem != NULL; elem = elem->next)
		if (cmp(elem->data, data) == 0)
			return elem->data;

	return NULL;
}

void
list_merge(list_t list1, list_t list2)
{
	if (list2->head == NULL)
		/* do nothing */ ;
	else if (list1->head == NULL)
	  {
		list1->head = list2->head;
		list1->tail = list2->tail;
	  }
	else
	  {
		list1->tail->next = list2->head;
		list2->head->prev = list1->tail;
		list1->tail = list2->tail;
	  }

	free(list2);
}

void *
list_get_last_data(list_t list)
{
	return (list->tail == NULL) ? NULL : list->tail->data;
}

char
list_is_empty(list_t list)
{
	return list->head == NULL;
}

unsigned long
list_get_n_elems(list_t list)
{
	struct list_elem *elem;
	unsigned long i;

	i = 0;
	for (elem = list->head; elem != NULL; elem = elem->next)
		i++;

	return i;
}

static void
elem_copy(void *data, void *context)
{
	list_t list;

	list = (list_t)context;

	list_append(list, data);
}

list_t
list_copy(list_t list)
{
	list_t ret;

	ret = list_new();

	list_for_each(list, elem_copy, ret);

	return ret;
}

void
list_free(list_t list)
{
	struct list_elem *p;

	if (list->head == NULL)
		/* do nothing */ ;
	else if (list->head == list->tail)
		free(list->head);
	else
	  {
		for (p = list->head->next; p != NULL; p = p->next)
			free(p->prev);
		free(list->tail);
	  }

	free(list);
}

#include <stdio.h>

void
list_dump(list_t list)
{
	struct list_elem *p;

	fprintf(stderr, "list: %p\nhead: %p, tail: %p\n",
		(void *)list, (void *)list->head, (void *)list->tail);

	for (p = list->head; p != NULL; p = p->next)
		fprintf(stderr, "elem: %p, next: %p, prev: %p, data: %p\n",
			(void *)p, (void *)p->next, (void *)p->prev,
			(void *)p->data);
}
