/*
 * pebs_smpl.c - Intel Core PEBS support
 *
 * Copyright (c) 2006 Hewlett-Packard Development Company, L.P.
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"

#include <perfmon/perfmon_pebs_smpl.h>

static int
pebs_process_samples(pfmon_sdesc_t *sdesc)
{
	pfm_pebs_smpl_hdr_t *hdr;
	pfm_pebs_smpl_entry_t *ent;
	pfmon_smpl_desc_t *csmpl;
	FILE *fp;
	uint64_t count, entry;
	void *hash_desc;
	int ret;

	csmpl      = &sdesc->csmpl;
	hdr        = csmpl->smpl_hdr;
	fp         = csmpl->smpl_fp;
	hash_desc  = csmpl->sym_hash;

	entry      = csmpl->entry_count;

	count = (hdr->ds.pebs_index - hdr->ds.pebs_buf_base)/sizeof(*ent);
	/*
	 * the beginning of the buffer does not necessarily follow the header
	 * due to alignement.
	 */
	ent   = (pfm_pebs_smpl_entry_t *)((unsigned long)(hdr+1)+ hdr->start_offs);

	DPRINT(("hdr_count=%"PRIu64" hdr=%p\n", count, hdr));

	csmpl->last_count = count;
	csmpl->entry_count += count;
	csmpl->last_ovfl = hdr->overflows;

	while(count--) {
		ret = fprintf(fp, "entry %06"PRIu64" eflags:0x%08lx EAX:0x%08lx ESP:0x%08lx IIP:",
			entry,
			ent->eflags,
			ent->eax,
			ent->esp);
		pfmon_print_address(fp, hash_desc, &options.primary_syms, PFMON_TEXT_SYMBOL, ent->ip);
		fputc('\n', fp);
		if (ret == -1)
			goto error;
		ent++;
		entry++;
	}
	return 0;
error:
	warning("cannot write to sampling file: %s\n", strerror(errno));
	/* not reached */
	return -1;
}

pfmon_smpl_module_t pebs_smpl_module;
static void pebs_initialize_mask(void)
{
	/* support only Intel Core, not Pentium 4 */
	pfmon_bitmask_set(pebs_smpl_module.pmu_mask, PFMLIB_CORE_PMU);
}

static int
pebs_smpl_init_ctx_arg(pfmon_ctx_t *ctx, unsigned int max_pmds_sample)
{
#define ENTRY_SIZE(npmd,ez)	((ez)+((npmd)*sizeof(uint64_t)))
	pfm_pebs_smpl_arg_t *ctx_arg;
	size_t entry_size, pmu_max_entry_size, buf_size, slack;
	int ret;

	/*
	 * samples may have different size, max_pmds_samples represent the
	 * largest sample for the measurement.
	 */
	entry_size         = ENTRY_SIZE(0, sizeof(pfm_pebs_smpl_entry_t));
	pmu_max_entry_size = ENTRY_SIZE(0, sizeof(pfm_pebs_smpl_entry_t));
	
	/*
	 * The buffer is full if the space left after recording a sample is
	 * smaller than the maximum sample size possible. The maximum sample size
	 * is defined with an entry_header and all implemented PMDS in the body.
	 *
	 * The slack ensures that slightly less than pmu_max_entry_size is left
	 * after the kernel writes the last entry (doing -1 instead of -entry_size
	 * would also work).
	 *
	 * buffer size is rounded up to kernel page size.
	 *
	 * The kernel is testing for < pmu_max_entry_size and not <=.
	 *
	 * buf_size =  sizeof(pfm_default_smpl_hdr_t) + slack + options.smpl_entries*entry_size;
	 */
	slack = 256;

	ret = pfmon_compute_smpl_entries(sizeof(pfm_pebs_smpl_hdr_t), entry_size, slack);	

	buf_size = sizeof(pfm_pebs_smpl_hdr_t)
		 + slack
		 + options.smpl_entries*entry_size;

	vbprintf("sampling buffer #entries=%lu size=%zu, max_entry_size=%zu\n",
		 options.smpl_entries, buf_size, entry_size);

	/*
	 * ctx_arg is freed in pfmon_create_context().
	 */
	ctx_arg = calloc(1, sizeof(*ctx_arg));
	if (ctx_arg == NULL) {
		warning("cannot allocate format argument\n");
		return -1;
	}
	ctx->ctx_arg      = ctx_arg;
	ctx->ctx_arg_size = sizeof(pfm_pebs_smpl_arg_t);

	ctx_arg->buf_size = buf_size;
	ctx->ctx_map_size = buf_size;

	ctx_arg->intr_thres = (buf_size/sizeof(pfm_pebs_smpl_entry_t))*90/100;
	ctx_arg->cnt_reset = options.sets->long_rates[0].value;

	return 0;
}

pfmon_smpl_module_t pebs_smpl_module = {
	.name		    = "pebs",
	.description	    = "Intel Core PEBS sampling",
	.process_samples    = pebs_process_samples,
	.initialize_mask    = pebs_initialize_mask,
	.init_ctx_arg	    = pebs_smpl_init_ctx_arg,
	.flags		    = PFMON_SMPL_MOD_FL_PEBS,
	.fmt_name	    = PFM_PEBS_SMPL_NAME

};
