# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

# Embedded icon is from Imagemagick (http://www.imagemagick.org).

# Follows PEP8

from core import ct, models
from lib.reverse_translation import _t

COMMANDS = {

_t('3D Edge'):
    """%(convert)s file_in.tif  -fx A  +matte -blur 0x6  -shade 110x30  \
    -normalize \
    file_in.tif  -compose Overlay -composite \
    file_in.tif  -matte  -compose Dst_In  -composite \
    file_out.png""",

#http://www.imagemagick.org/Usage/convolve/
_t('Blur'): \
    """%(convert)s file_in.tif -blur %(blur_radius)sx%(blur_sigma)s \
    file_out.png""",

_t('Bullet'): r"""%(convert)s file_in.tif -matte \
    \( +clone -channel A -separate +channel -negate \
   -bordercolor black -border %(border)s  -blur 0x%(blur1)s \
   -shade %(shade)s \
   -normalize -blur 0x%(blur2)s -fill '%(color)s' -tint 100 \) \
    -gravity center -compose Atop -composite \
    file_out.png""",

_t('Charcoal'): \
    """%(convert)s file_in.tif -charcoal %(charcoal_radius)s file_out.png""",

_t('Motion Blur'): \
    """%(convert)s file_in.tif \
    -motion-blur %(blur_radius)sx%(blur_sigma)s+%(blur_angle)s file_out.png""",

_t('Pencil Sketch'): \
    """%(convert)s file_in.tif -colorspace gray -sketch \
    %(sketch_radius)sx%(sketch_sigma)s+%(sketch_angle)s file_out.png""",

_t('Paint'): \
    """%(convert)s file_in.tif -paint %(paint_radius)s file_out.png""",

_t('Polaroid'):
    """%(convert)s %(caption_cli)s file_in.tif \
    -bordercolor '%(border_color)s' \
    -background '%(shadow_color)s' +polaroid file_out.png""",

#http://www.imagemagick.org/Usage/convolve/
_t('Shadow'):
    r"""%(convert)s file_in.tif \( +clone  -background '%(shadow_color)s' \
    -shadow \
    %(blur_radius)sx%(blur_sigma)s+%(horizontal_offset)s+%(vertical_offset)s \
    \) +swap -background none   -layers merge  +repage file_out.png""",

_t('Sharpen'): \
    """%(convert)s file_in.tif -sharpen %(sharpen_radius)sx%(sharpen_sigma)s \
    file_out.png""",

_t('Sigmoidal Contrast'):\
    """%(convert)s file_in.tif  \
    -sigmoidal-contrast %(contrast_factor)s,%(contrast_treshold)s%% \
    file_out.png""",

_t('Unsharp'): \
    """%(convert)s file_in.tif -unsharp %(unsharp_radius)sx%(unsharp_sigma)s \
    file_out.png""",

_t('Wave'): \
    """%(convert)s file_in.tif -wave %(wave_height)sx%(wave_length)s \
    file_out.png""",
}

ACTIONS = COMMANDS.keys()
ACTIONS.sort()


class Action(models.Action):
    """Defined variables: <filename> <type> <folder> <width> <height>"""

    label = _t('Imagemagick')
    author = 'Stani'
    email = 'spe.stani.be@gmail.com'
    version = '0.1'
    tags = [_t('filter'), _t('plugin')]
    tags_hidden = ACTIONS
    __doc__ = _t('Blur, Polaroid, Shadow, Unsharp...')

    def init(self):
        self.find_exe('convert', 'Imagemagick')

    def interface(self, fields):
        fields[_t('Action')] = self.ChoiceField('Polaroid',
            choices=ACTIONS)
        fields[_t('Horizontal Offset')] = self.PixelField('2%',
                                        choices=self.SMALL_PIXELS)
        fields[_t('Vertical Offset')] = self.PixelField('2%',
                                        choices=self.SMALL_PIXELS)
        fields[_t('Color')] = self.ColorField('#FF0000')
        fields[_t('Border Color')] = self.ColorField('#FFFFFF')
        fields[_t('Shadow Color')] = self.ColorField('#000000')
        fields[_t('Caption')] = self.CharField(choices=self.STAMPS)
        fields[_t('Charcoal Radius')] = self.PixelField('0.5%')
        fields[_t('Contrast Factor')] = self.SliderField(100, 0, 100)
        fields[_t('Contrast Treshold')] = self.SliderField(50, 0, 100)
        fields[_t('Blur Radius')] = self.PixelField('80px')
        fields[_t('Blur Sigma')] = self.PixelField('3px')
        fields[_t('Blur Angle')] = self.SliderField(120, 0, 359)
        fields[_t('Paint Radius')] = self.PixelField('0.5%')
        fields[_t('Sharpen Radius')] = self.PixelField('0px')
        fields[_t('Sharpen Sigma')] = self.PixelField('3px')
        fields[_t('Sketch Radius')] = self.PixelField('0px')
        fields[_t('Sketch Sigma')] = self.PixelField('20px')
        fields[_t('Sketch Angle')] = self.SliderField(120, 0, 359)
        fields[_t('Unsharp Radius')] = self.PixelField('0px')
        fields[_t('Unsharp Sigma')] = self.PixelField('3px')
        fields[_t('Wave Height')] = self.PixelField('0px')
        fields[_t('Wave Length')] = self.PixelField('3px')

    def get_relevant_field_labels(self):
        """If this method is present, Phatch will only show relevant
        fields.

:returns: list of the field labels which are relevant
:rtype: list of strings

        .. note::

            It is very important that the list of labels has EXACTLY
            the same order as defined in the interface method.
        """
        relevant = ['Action']
        action = self.get_field_string('Action')
        if action == 'Blur':
            relevant.extend(['Blur Radius', 'Blur Sigma'])
        elif action == 'Bullet':
            relevant.extend(['Color'])
        elif action == 'Charcoal':
            relevant.extend(['Charcoal Radius'])
        if action == 'Motion Blur':
            relevant.extend(['Blur Radius', 'Blur Sigma', 'Blur Angle'])
        elif action == 'Paint':
            relevant.extend(['Paint Radius'])
        elif action == 'Polaroid':
            relevant.extend(['Border Color', 'Shadow Color', 'Caption'])
        elif action == 'Shadow':
            relevant.extend(['Horizontal Offset', 'Vertical Offset',
                'Shadow Color', 'Blur Radius', 'Blur Sigma'])
        elif action == 'Sharpen':
            relevant.extend(['Sharpen Radius', 'Sharpen Sigma'])
        elif action == 'Pencil Sketch':
            relevant.extend(['Sketch Radius', 'Sketch Sigma',
                'Sketch Angle'])
        elif action == 'Sigmoidal Contrast':
            relevant.extend(['Contrast Factor', 'Contrast Treshold'])
        elif action == 'Unsharp':
            relevant.extend(['Unsharp Radius', 'Unsharp Sigma'])
        elif action == 'Wave':
            relevant.extend(['Wave Height', 'Wave Length'])
        return relevant

    def apply(self, photo, setting, cache):
        info = photo.info
        action = self.get_field('Action', info)

        w, h = info['size']
        dia = (w+h)/2
        values = self.values(info, pixel_fields = {
            'Horizontal Offset': w,
            'Vertical Offset': h,
            'Blur Radius': dia,
            'Blur Sigma': dia,
            'Charcoal Radius': dia,
            'Paint Radius': dia,
            'Sharpen Radius': dia,
            'Sharpen Sigma': dia,
            'Sketch Radius': dia,
            'Sketch Sigma': dia,
            'Unsharp Radius': dia,
            'Unsharp Sigma': dia,
            'Wave Height': dia,
            'Wave Length': dia,
        })
        values['convert'] = self.exe['convert']
        if action == 'Bullet':
            #extra values
            values['border'] = dia/3
            values['blur1'] = dia/7
            values['shade'] = '%dx%d'%(dia+105, dia+15)
            values['blur2'] = dia/12
        elif action == 'Polaroid':
            caption = values['caption']
            if caption.strip():
                values['caption_cli'] = \
                    '-caption "%s" -gravity center'%caption
        elif action == 'Sigmoidal Contrast':
            values['contrast_factor'] /= 10.0

        command = COMMANDS[action]%values
        #print(command)  #use "python phatch.py -v" for verbose mode
        photo.call(command)
        return photo

    icon = \
'x\xda\x01l\x07\x93\xf8\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x000\x00\
\x00\x000\x08\x06\x00\x00\x00W\x02\xf9\x87\x00\x00\x00\x04sBIT\x08\x08\x08\
\x08|\x08d\x88\x00\x00\x07#IDATh\x81\xd5\x99\x7fh\x93\xf9\x1d\xc7_\x8d\x8f\
\xf2\x9d\xc6\xfap\x14/\xe7\x05\xc9Fv\x0b\xdc3\xad\xbbL\x8a\xcb4\xb8Pz\x90\
\x93 r\xf4\xa0w\x94\x12\xaaH\x19E\x82\xabG\x91(\xa1\xeb\xa4\x8c\xe2z^{\xb8Q\
\\\x90\xa3\x17\xa4\xbb\x95\xb6\xec\x82D)\xb7\xd2E/z\xe5\xc8\x8ep{\xe6\xc4\
\x05\xc9z\x0f]\xf4\x1e]\xae\xdf\xfdQ\xeb\xaam\xb5\xb5M\xea^\x90\x7f\x9e|\x9f\
\xcf\xf7\xf3~\xbe\x9f\x1f\xdf\xef\xf3@\x91\x19\x1a\x1a\x90\xc5\x9ecA|\xd5\
\xbe%O~\xfd\xda\xf8\x1e\xf7N\xa7\x0c6\xd6\xc9`\x83O\xfa\xf6\xda\xe5\xc8\xa7#\
\xc9b\xf8\x07\xa0,\xe4D\xd7{\x9d\x89m\xda0v\xbb_j\x9a\x97\x17T\x95\x86\x86`\
\xd9\xd3\x0cn\xdb\xae]\xf2\xfb5\xc2\xe1\x14\xaaU\xd0\xfeK\xd03\xfak+\xef\xfa\
\x02\x84\x8e\x86dM\xb5]\x0e\x0c:e~\xd2-3\x19M\x06\xdf\xb1\xcbH[\xc7\xa2V#\
\xd2\x16\x91\xad\xc7\xec\xf2\xe6-M\x16\nn\x19\x8f;\xa5o\xb7]F\xda"E\t%\xcb\
\xe3\x17:Nu\x94y\xbc\x87\xd03&\xa6i\x02\x90\xcb\xbbh}7\xf4\xd4\xa7\x0f \x14\
\x81\xb2\xb6\x9e\xe6C*\xbd\xe7\xb2\xf4\xf5y\t\xb7\xf7_i}\xb7uQ\xf7\xaf\x08CC\
\x03\xd2\xb7\xd7.\x83\x8d~\x19l\xf4K\xf7N\xed\x99\x9e\x9ew\x97{\xf5\x128\xfe\
\xa7\xf8\xeaM^J\x82\r\xc1U\x15:\'\x07\x96J\x81,\xd1sg\x8b&\xe2\xfa\xb5\xf1=\
\xc9\xcf\x92\x8d\x00\xc1\xfa\x032\xb0\xff\xd1\xd2\xbel\x01n\xad\x92T\xb2w\
\xb9f\x16\xe4\xf2\xa5D\xa2\xe5\x17\xde\x9e\xd6cv\xd9\x1cJ\xa3(:\x03\x83+\x18\
\xde\xb1\xbe\xa8\xec>\xe3\x94#\x97G\x96d49\x96\xecY\xecX\xbf_\x93\xb9\xdb\
\xd3ey|\\\x93\xbe\xbd\x152t4\xf4\xbf\xf9f\x1b\x0b\xec\xf7\xc9\xe8\xf9\xe8\
\x92\x9c\xe9\xfa\xb5&;\xda\x16\x9f\x0b\xc1\xc6:\xe9\xf7k\xf2\xfa\xb5\xf1=O\
\x1b{\xb6\xe7\xac\xf4U[e\xe4\xb8S&\xc74y`\x9fS\xce.0e\x00\xbej\x8ft9\xb3\x00\
x=\x90\x1c\x0f\xd0\xde\xd6\xb1\xe8\xba\x1d9^\'m\xf6\x14\x95?\xee=\xe8\xde\
\xe1\xfe`\xa1q\xa1#!i\x18)\x1cv\x9d\xc0\x9b\x82\xfe>\x93\xe1\x04\x1c\n\x86\
\xa9{\xa7n\xde\xf9F>M&\x01<\xbb\xdc\xee\xa6\xc3u2\x9b5\x88]\x18xtl\xf4|Tv\
\x9c\xb2\xcb\xfc\xa4[\xe6\'\xdd2r\xdc)\x03\xfb}\x0b>\xa1\xc2\xe4tR\xcd044 \
\xe3qMFN>y\x15\xce\xf6\x9c\x95M\x87\xb5\x87]z`\xd0)CG\x9b\x96\x15\xcf\x16\
\x80\xde\xdenTU0:f0:f\x90\xce8\xa8\xab;\xc4\xb6\xed\xda\xa59\xce\x7f\xe1\x93\
\xa4kz\n\xb7\xbb\x1eN\xfc\xfa\xeb\xfe\xb2\xe4\x98\x8a\xc31\xfa\xc4\xd8\x0e\
\x1e\x0c\x96\t\xe1%\x117i?e\x90\x1e7\x11\x8a\xba\x1c\xff\xa77s\xf5\xf5\x87\
\xb0\xaeW\x89_L0\x12\xef\xa5\xbd#\x8cg\xb7g\xfe\x10\x12.\x10\x0e\x10\x95Wf_\
\xb6\xaa\x01\\\xae^\x12#\xbd\x8d\xc0\xc1\x85\xa7\x14\xa4\xc6\x03\xd4\xbeU{0\
\xfdy\xba\x07\xb2\xcb\x120\x87\xfe\xfeg\xdf\xbfwu\xb9e\xf4\x9csQ\xc9\xb9R\
\xcc\xe9\x03\x81\x80\xff\xd97]\x96\x00\x95?\x12|\xfc\x87\xce\xc4r\x9cZU\xba\
\xba\xdc2vA+\xea!f6\xcb\xee\xc4s-\xd6\xe2t@"\xdeY\xbaC\xccJ\xd3\xde\xe6\x91\
\x03\x83N\xb9\xa2-\x7f\x01V~\x05\x00\xad\xb2\x19\xb1\x0eR\xc9\xceb\x98/\r\
\x91\x93u2\x1e\xd7d\xec\xc2\xd2\xb6%\xcf\x15\xddg\x9c\xb2\xa3\xad\xb8\xa7\
\xb2\xa2\x84\xd0\x0c\x8aRO\x95\xc7\xa0\xa3\xad\xf5\xffw\x15"m>\x19\xfd\x9d&G\
.\x17\'\xa1\x8b\xba\x02\x00^O\x0b(0:\xd0^\x14\xfbE\x17\xe0\xd9\xed+3r\x01\
\xaa<Y:N.o\xe7\xb9\xaaDN\xfa\xe5@\x9b&c}+[\x95\x8a\xbe\x023\xd4\xbc\x11>\x98\
\x05\xf4\x91vf\x0e\xe9+A\xc9\x04\xb8w\xb8?\xb0\xd8\x9aq\xd9 \xf1\xfb\xf0\xa2\
\xcf\xc3O\xa3d\x02\x00\x1a\x1a\x82e\xa9\xbc\x97\xaa\n\x9d\xc8\x91\x03\xf3\
\x86R\xf2\xb3d\xe3R\xb6\xe3\xa5{_9\x8b\xc8a\xbf\xac}E\xa7\xa5\xcf\xe4[\x13\
\xd6(`\x15\x82\x97\xca\xc5\xc31\xff\x9c4\xc9\x9b&z\x1e\x1c\xafV\x12\xfb(6\
\xaf\xaf\xab"\xc0\xbb\xcb-=\x15\x06\xf5{\x05\xbd\x17M*\xed\x0e\xcc\x02T9m8T\
\xf1\xc8\xd8l\xde\xa4\xb5?\xc5\xb8\x01\xc9\xeb\xe3s\xfc-\xb9\x80\xc6\xc6\xa0\
\xacVt\xaa\x9c6\xbaGRD\x13:\x15V\xc1\xcb\xaa\xe0\x92n\x10\xad\xf7R\xed\xb2ay\
\xf0\xe5b\xaa\x00\xf7\x0b&?\xeb\x1c\xe6\x8d`\xcb\x9c\xb7\xe4kJ-\xe0\xa55\xf7\
\xc3\xcd{5*\xd6+\xa4n\xe6\xb8g\x16\xf8\xa8\xc9\xcf\xdb;\x9d\x8cen\xf2\x9bx\
\x8a\x8d\xe5\x02m\xb3\x15\x0b\n\x16\x05,(\xe4\xee\x9a\x0c_\xcdp\xe3\x1f7N\
\x00\xc4/\xc6e\xf9\x86\xf2pI\x05\xc4\xfab\xf2\xdb\xcc(5?\xb0!\xa7\xe0W\x17u\
\xb6;\x04\xd5N;\x00^\xcdN\xfav\x9e\xf7\xe2\xe3D\xc72\\\xd5s\xa4o\x19\xect\
\xd8(\x00g\x86G1\xef\xdd?\x11:\x12\x92MM\x11\x92\x7f\xb3\xcd\xff\x89\xa9\x98\
|\xf2E\x96\xd7\xb6\xea(\n\x98\x13Y>\xc9A\x9dS\xc7n\xb7\xb1Y\x08\xce\xd7{\xf9\
r\xc2 }\xcb\xc0*\x04\x9a]e\xad\x02\xf6\x17\x04\xf9\xbc\x89g\x97Gv\x9e\x1e\
\x86)/\xe4D\xe9s \xd6\x17\x93\xc6\xd7\x06-\xc7Zh\xf1\xb9\x18\xbe\x9a\xa6r\
\x8b\xa0\xbe\xd2\x81\xaa\xaa\xa8\xe5*\xeb\xd6?\x9a\xc8\x16\x05\x0c\x13vF>\
\xe4\xc6\x84\nS5\x0f\xfe1K\xdb\x07\x00\x0e\xbcy\xa0\xcc\xf1]\x07?\xb4\tj=\
\x0e\xba\x1bjH\xdd2i\x1eL3\x9a\xd6\xc9|\x95\xe6+]\xe7~\xc1|\x98\xc8\x00\xe5\
\n\x1c\xf6\xb9`\xca\x00\xcc\xd5\x13\x00 \xac\xe2\xca]\xd3D \xd8\xaa\n*_\xb1a\
L\t\xdaGtzS\xd3/\xba\xcc\xbc\xc9=s\xfa\xe9O\x15\xa6\xef\xfb\xb9\xa7\x92\x9f\
\xbaT\xb0\xa4f,\xadN\x1f\x00\x88\x9e\x8b\xcaD|\x18,&\xc6\x1dh\xf9>\x0c\xa7u\
\xfe\x92\xc9r\xc2cc\x93\xaa\xb2\xd1\xaa"\xac\x82u\x8ax(\xe4\xcb\t\x83\xc0\
\xe9\x04z\xae\nX\xa5F\xf68]\xa7\xbb$\xc9\x0f1\xf2&\xfd\xa3i\xde\xf7\xbbX\xab\
\xc0&Ue\xad"P\xad\x8221\x9d\xcc\x16\x05R\xbaAMw\x02\xc3p\x94\xbe\x0f\xcc\xc7\
\xe0\xd0\xe0\x89\xef\xfdd_8\xf3\x8d\xc2\x95\xcf3l\xde\xa0\xb0\xc5*\xf8w\xde\
\xe4\x9b\xbby\xee\xdc51&rl\x14\n\x16E\xf0b\xb9`\xc7V\x95?\xa6\x92\xcf\xc7\n\
\xcc&|\xbcUvvv\x12\xd0l\xbc\xe5\x14lP\x04w\n\xd3I\xfb\xb2\xaa\xe2\xb0\xdbXgU\
\xb1(\xd0\x12\x1b}\xfe\x04\x00\xf8\xf7\xd5\xc9\xe1x\x82\xb5\x85,/Z\xa7\xeb\
\xbf\x10\x82\xdf\x06\x1c\x8f\x88h\x8c&\x9e\x8f\x10z\x9c\x89\xffl\r\x9b\x9bj\
\x99\xfaW\x05\x93\xa6\rs\xcaI\xfe\xbe\xce_\'M\xf6l\xb1B\xc1\xe4;\xc2\xca\xfb\
\x7f\xce\xacN\x19}\x12\xb1\xbe\xa84f\xca<\x02\xb0=\xf8UqE7\xe8Me\xf9:or\xf3v\
\x96\xbf\xe7\x0c\xfe\x0bB\x1b\x08z1\xab\x91\x11\x00\x00\x00\x00IEND\xaeB`\
\x82\x1e\x0c\xa8\x99'
