# -*- coding: utf-8 -*-
# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

# Embedded icon is designed by Alexandre Moore (http://nuovext.pwsp.net/).

# Follows PEP8

import time
from datetime import datetime

from core import ct, models
from lib.metadata import DateTime
from lib.reverse_translation import _t

OPTIONS = [_t('Exif time'), _t('File time'), _t('Exif and file')]


def init():
    global relativedelta
    from other.relativedelta import relativedelta


def get_date(info):
    return datetime(
        year=info[_t('year')],
        month=info[_t('month')],
        day=info[_t('day')],
        hour=info[_t('hour')],
        minute=info[_t('minute')],
        second=info[_t('second')])


class Action(models.Action):
    label = _t('Time Shift')
    author = 'Juho Vepsäläinen'
    email = 'bebraw@gmail.com'
    version = '0.2'
    init = staticmethod(init)
    tags = [_t('metadata')]
    __doc__ = _t('Shift time values of tags')
    metadata = ['Exif_Image_DateTime', 'year', 'month', 'day', 'hour',
        'minute', 'second'] #only specified for reading

    def interface(self, fields):
        fields[_t('Change')] = self.ChoiceField(OPTIONS[2], choices=OPTIONS)
        fields[_t('Use exif datetime')] = self.BooleanField(True)
        fields[_t('Seconds')] = self.IntegerField(0)
        fields[_t('Minutes')] = self.IntegerField(0)
        fields[_t('Hours')] = self.IntegerField(0)
        fields[_t('Days')] = self.IntegerField(0)
        fields[_t('Months')] = self.IntegerField(0)
        fields[_t('Years')] = self.IntegerField(0)

    def apply(self, photo, setting, cache):
        info = photo.info
        method = self.get_field('Change', info)
        date_delta = self._construct_date_delta(photo)
        if method == OPTIONS[1] or not self.get_field('Use exif datetime'):
            new_date = get_date(info) + date_delta
        else:
            try:
                new_date = info['Exif_Image_DateTime'].datetime + date_delta
            except KeyError:
                new_date = get_date(info) + date_delta
        if method in (OPTIONS[1], OPTIONS[2]):
            try:
                photo.modify_date = time.mktime(new_date.timetuple())
            except OverflowError:
                raise Exception(
                    _('Sorry, time shift can only handle '\
                      'dates before 2038-1-19'))
        new_date = DateTime(new_date)
        if method in (OPTIONS[0], OPTIONS[2]):
            info['Exif_Image_DateTime'] = new_date
            info['Exif_Photo_DateTimeOriginal'] = new_date
            info['Exif_Photo_DateTimeDigitized'] = new_date
        return photo

    def get_relevant_field_labels(self):
        relevant = ['Change', 'Seconds', 'Minutes', 'Hours', 'Days', 'Months',
        'Years']
        if self.get_field_string('Change') == OPTIONS[2]:
            relevant.append('Use exif datetime')
        return relevant

    def _construct_date_delta(self, photo):
        info = photo.info

        second_delta = self.get_field('Seconds', info)
        minute_delta = self.get_field('Minutes', info)
        hour_delta = self.get_field('Hours', info)
        day_delta = self.get_field('Days', info)
        month_delta = self.get_field('Months', info)
        year_delta = self.get_field('Years', info)

        return relativedelta(
            years=year_delta, months=month_delta, days=day_delta,
            hours=hour_delta, minutes=minute_delta, seconds=second_delta)

    icon = \
'x\xda\x01^\x0b\xa1\xf4\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x000\x00\
\x00\x000\x08\x06\x00\x00\x00W\x02\xf9\x87\x00\x00\x00\x04sBIT\x08\x08\x08\
\x08|\x08d\x88\x00\x00\x0b\x15IDATh\x81\xed\x9a{lS\xd7\x1d\xc7\xbf\xf7\xdc\
\x87_q\xeck;\xb1\xd3<\xec\xbc\x80\x8e\x04\x92\n\xe8\\\xa0*[!\xb4T\n\xd3\xd6\
\x0eU\xac\xd2\xaan\xeb\xfaO\xa9*\xb5\xaa\x86\xda?\x8a\xd4\x7f\xdaj\xffN\xea(\
\x9b\xa0\x12\x12]\xa7\t\xba\xc2DT\xda\x82(\nKKy\xd5$NB\x9c\xe1<\xec$\x8e\x1f\
\xd7\xf7q\xce\xfe\x88\x1d\x9c\xc4\t\x0e\x84I\x93\xfa\x95~\xbaW\xd7\xbf{\xfc\
\xfd\x9cs\xcf\xb9\xe7\x1c\x1b\xf8?\x17w\x9f\xca4\x01p\x02p\xe4\xaeM\x01\x98\
\x04\x90\x05\xc0V\xfa\xcbVB\xe5\x00\xb6\x03\xd8b6\x9b\xdbjjj\xeadY\xb6Z,\x16\
\x0e\x002\x99\x0c\x8b\xc7\xe3\xe9H$2\x98\xcdf\xbf\x01p\x16\xc0I\x00\xc9{\xfd\
\xe2{\x01\xe0\x00\xb4\x00x\xa1\xa9\xa9\xe9\xc9`0\xe8njj2UVV\x8a\x94R\x8e\xe3\
8p\xdcL\xf1\x8c10\xc6@\x08a\xa3\xa3\xa3Zooo\xf6\xec\xd9\xb3\xb1p8|\x02\xc0\
\x9f\x01\\\xc6]\xb6\xcc\xdd\x024\x11B\xfe\xb0a\xc3\x86\x9d;w\xee\xb4WVV\x9aD\
Q\xe4\\.\x17\xb1\xd9l\x9c\xd9l\x86$Is\x00TU\x85\xa2(H\xa5R,\x1e\x8f\xd3l6\
\xcbFFF\xb2\xa7N\x9d\x9a\xbex\xf1\xe2?)\xa5\x07\x00\x84\xef7\x00\x0f\xe0wuuu\
\xaf>\xfd\xf4\xd3^\xaf\xd7k\x92e\x99\xf8|>\xe2\xf1xPQQ\x01\x87\xc3\x01\x8b\
\xc5\x02\x93\xc9\x04A\x10\x00\x00\xba\xae#\x9b\xcd"\x93\xc9`rr\x12ccc\x18\
\x1f\x1fG4\x1a\xa5\xf1x\x9c\x8e\x8c\x8cd\x8f\x1d;6:44\xf4.\x80?\x010\xee\x07\
\x80\r\xc0\x1f\x1f}\xf4\xd1\x9f=\xf6\xd8c\xe5v\xbb\x9d\x0f\x04\x02$\x10\x08\
\xa0\xbe\xbe\x1e.\x97k\xb6\xc6\xf3\x8a\xc5b\x90$\tv\xbb}\xceu\xc6\x18\xe2\
\xf18\xc2\xe10\x06\x06\x06000@\x13\x89\x84\xd1\xd5\xd55u\xee\xdc\xb9\xbf\x03\
\xd8\x07 U\x8a)\xbeT\xf3\x1c\xc7\x1d\xee\xec\xec|\xb2\xad\xad\xad\xbc\xaa\
\xaaJX\xbdz5\xd9\xb8q#V\xadZ\x05\x8b\xc5R\xf4\xa63g\xce@UU\xf8|\xbe\x05\x9fY\
,\x16<\xf0\xc0\x03p:\x9d\xa0\x94r\x8a\xa2\x10\xa7\xd3ir\xb9\\M7n\xdch\x07p\
\x1c\x80\xb6\x12\x00&\x00\x7f\xe9\xec\xec\xdc\x1e\x08\x04l~\xbf_hmm\xe56o\
\xde\x0c\xa7\xd3\t\x00\xb35\x9f\xef\xb8\xf9\xe8\xeb\xeb\x83\xddnGUU\xd5\x9c\
\xeby1\xc6`\xb7\xdbQ]]\rM\xd3\xb8t:M\x0c\xc3\x10}>_M(\x14z\x10\xc0?\x00\xd0{\
\x058\xb0u\xeb\xd6_\xd6\xd7\xd7\xdb\x02\x81\x80\xd0\xd6\xd6\x86`0\x08Q\x14\
\x17\x18\x9a\xaf\xde\xde^\x94\x95\x95\x15m\x81Bp\x9e\xe7Q]]\x8dl6\x8bT*E\xb2\
\xd9,o\xb5Z\xeb\x86\x86\x86l\x00\xba\x962\'\xdc\xc1|\x87\xdf\xef\x7f\xbe\xb9\
\xb9\xb9\xac\xa2\xa2BX\xbbv-\x82\xc1 \x04A\x00!dI\xf3\xa5\x8a\xe38\x10B \x08\
\x02\x1a\x1b\x1b\xa1(\n\xa6\xa6\xa6\x04Ji\xd9\xe0\xe0\xe0\x0b\x91H\xe4s\x00\
\xffZ\xec\xfe\xa5Z\xc0,\x08\xc2\xa1\x1d;v\x04\xca\xca\xca\xc4\xd6\xd6V\xae\
\xa3\xa3\x03\x92$\x81\xe7K\xeb:\xb2,\xc3\xe3\xf1@\x92\xa4;\xe6^\xb8p\x01\x03\
\x03\x03\xd8\xb2e\x0b\xa2\xd1(n\xdd\xba\xc5y<\x1e1\x14\n=H)=\x02@_.\xc0o\x1e\
z\xe8\xa1=\x1e\x8f\xc7\xdc\xd0\xd0@:;;\xe1r\xb9J6\x0f\x00V\xab\x15&\x93\xe9\
\x8ey\'N\x9c@&\x93\xc1\xae]\xbb \x08\x02\\.\x17\xfa\xfb\xfb\xb9x<\xce\xf1<\
\xef\x88F\xa3#\x00\xfe]\xec^\xb2H\x99\xa2(\x8a\xcf\xd7\xd4\xd4X-\x16\x0b\xbf\
~\xfdzTUU\x81\x102\xfbV]\xa98x\xf0 \xccf3\x9ex\xe2\x89\xfc\xdb\x1a\xd5\xd5\
\xd5hmm\x85\xd9l\xe6kkk\xad\x82 \xfc\x16\x80\xb8\x1c\x80\x9f\x06\x02\x81f\
\x8e\xe3x\x8f\xc7\x83\x87\x1f~\x18<\xcf\xaf\xa8qJ)\xdey\xe7\x1d\xacZ\xb5\n\
\xdb\xb6m\x9b\xf3\x19\xcf\xf3\x08\x06\x83\xf0x<`\x8c\x11\xbf\xdf\xdf\x0c\xe0\
\'\xcb\x01x\xce\xeb\xf5\x9at]\xe7V\xaf^\r\xa7\xd3\xb9\xa8\x91\xae\xae.$\x12\
\x89e\x99\xd74\ro\xbd\xf5\x16:::\x10\x0c\x06\x8b\xe6\xc8\xb2\x8c\xe6\xe6f0\
\xc6\x88\xcf\xe73\x01\xd8[*\x80I\x14\xc5\r<\xcf\x0b&\x93\x89[\xbbv\xed\x92\
\xa3\xcd\xe6\xcd\x9bq\xfc\xf8q\\\xbbv\xad$\xf3\xd3\xd3\xd3\xd8\xbf\x7f?\x9e}\
\xf6Y\xac_\xbf~\xd1<B\x08\xd6\xad[\x07Q\x149\x9e\xe7\x05A\x106\x02X0\x1a\x14\
\x03\xf8\x91,\xcb\x15\x8c1b\xb7\xdbQ__\x0f\xc30`\x18\x06(\xa5\x0bB\x10\x04<\
\xf3\xcc3\x08\x85B\xf8\xe4\x93O\x16<&\x85q\xeb\xd6-\xbc\xf9\xe6\x9bx\xe9\xa5\
\x97\xd0\xd4\xd4T\xb4\xbc\xf4\xc4\x04TE\x01\xa5\x14\r\r\rp8\x1c\xe08\x8e\xb8\
\\./\x805\xa5\x00\xb4;\x1c\x0e^\xd34\xce\xe5rA\x92$(\x8a\x82L&\x03EQ`\x18\
\xc6\x82\xda\x02\x80\xa7\x9ez\n\xb5\xb5\xb5x\xfb\xed\xb7\xa1i\xda\xec\xf5\
\xbcB\xa1\x10\x0e\x1c8\x807\xdex\x03\xd5\xd5\xd5\xb3\xd7\rE\xc1\xcd3gp\xfe\
\x83\x0fp\xfe\xe8Q\xf4]\xbe\x0cU\xd3\xa0(\n$I\x82,\xcb\xd04\x8d+//\xe7\x01\
\xb4\xcd7[\xecEVo2\x99\x88 \x08p\xbb\xdd\xa0\x94\xce\x9a\xa1\x94B\xd7u\xf0<\
\x0fQ\x14A\xc8\\\xfe\xf6\xf6v\xf8|>\xbc\xf6\xdak\xd8\xb7o\x1fjjj\x00\x00_~\
\xf9%\x0e\x1f>\x8c\xf7\xde{\x0fe6\x1bF\xbf\xf8\x02\x83\x9f\x7f\x0e\x85\xe7!x\
\xbd\xa8\r\x06\xb1\xe1\x91G\xc0\x11\x02M\xd3f[\x9c\xe38\xb8\xddn\x10B`6\x9b\
\t\x80\xc6R\x00\xca\x05A \x84\x10\xcej\xb5\x82\xd2\x85S\x11]\xd7\xa1\xeb:\
\x04A\x80(\xce\x1d\xdd\xbc^/\x0e\x1c8\x80\xd7_\x7f\x1d\xdb\xb7oG&\x93A\xf7G\
\x1f\xe1\xd5\xe6f\\{\xf9e\xe8n7\xdc[\xb7\xa2e\xdf>H6\xdb\xed\x960\x0ch\x8a2\
\xa7\xe5\x18c\xb0\xd9l\xe0y\x9e\xe3y\x9e\xe0\xf6\x12uI\x00\x91R\xca\xe5k\x97\
R\x8a\xef\xbe\xfb\x0e\xdd\xdd\xdds\x92\xf6\xee\xdd\x0bJ)TU\x9d\x05\xc9wvQ\
\x14\xf1\xfe\xfb\xef\xe3\xf7;wb\xfd7\xdf\xe0\xb9W^\x81w\xcf\x1e4\xfa\xfds\
\xca\xc8\xb7\xae\xa6i\xf8\xec\xb3\xcf\x10\x89Df?\x93e\x19\xbbw\xef\x9e\x9dj0\
\xc68\x14\xe9\xc4\xc5\x002\x00\x98a\x18\xb3\xcfrKK\x0bZZZ\x16$\xe6kK\xd34h\
\x9a\x06A\x10\xc0\xf3\xfc\xec\xa3\xd5\xf1\xe2\x8bp\xbb\xdd\xa8\xdf\xb4\t\x92\
$-\xa8]\xc30\xa0\xeb:\x18cx\xfc\xf1\xc7\x8b\x96\xaf\xaaj\x1e\x94\x01H\x97\
\x02\x103\x0c\xc3\xd0u\x9dMOOs\xf3;\xe3R\xca\x83\x10B\xc0\xf3<v\xed\xda\x85\
\xc2\x96\xccw\xfa\xfc3^\x8a\x12\x89\x044Mc\xc6\xcc\rc\xa5\x00\\K\xa5R\xccn\
\xb7ctt\x14\x8c\xb1e\xcf:\xf3C\xe2\xbd\x8aR\x8a\xb1\xb110\xc6\x90J\xa5(\x80\
\xeb\xf3s\x8a\r\xa3\xdd\x13\x13\x13Y\x93\xc9D\'\'\'1::z\xcfF\xeeV\xd1h\x14\
\x93\x93\x93\x90$\x89NNNf\x01\\\x9c\x9fS\x0c \x92H$\x065M\xa3\x89D\x02\xe1\
\xf0\xb27\nVL\xe1p\x18SSSPU\x95&\x12\x89\x01\x00\xc3\xf3s\x8a\x010\xc6XW*\
\x95\xd2R\xa9\x14\xeb\xe9\xe9Y\xf0R\xfa_\x881\x86\x9e\x9e\x1e(\x8a\xc2R\xa9\
\x94\xca\x18;\x8d"{G\x8bM\xe6\x0eE"\x91\x0c\xcf\xf3\xf4\xfb\xef\xbfG__\xdf\
\xfdu[D\xbd\xbd\xbd\x08\x85B \x84\xd0\xe1\xe1a\x05\xc0_\x8b\xe5-\x06p5\x16\
\x8b]H&\x93\xfa\xf8\xf88N\x9d:U\xf2\xa8\xb1\x122\x0c\x03\'O\x9eD<\x1eg\xc9dR\
\x8f\xc5b\xe7\x01\\+\x96\xbb\xd4\xf2j4\x9b\xcd\xeev\xbb\xddR2\x99$\xb2,\xa3\
\xae\xae\xee\xfe8\x9e\xa7\xaf\xbe\xfa\n\xa7O\x9fF:\x9d\xa6\xa1P(\x99N\xa7\
\xf7\x01\x18(\x96\xbb\x14@\x7f:\x9d\xde\xe4r\xb9\xeaUU\x15b\xb1\x18\xd7\xd8\
\xd8\x08Y\x96K6\x12\x0e\x87\xa1\xaa*\xca\xca\xcaJ\xbe\xa7\xbf\xbf\x1fG\x8f\
\x1e\xc5\xf0\xf00\x9b\x9e\x9eV\xc2\xe1\xf0I\x00\xef.\x96\x7f\xa7\x05nw<\x1e\
\xffyUUUY2\x99\xe4\xa3\xd1(\xd6\xacY\x03[\xc1\x1cf)}\xfb\xed\xb7 \x84\xa0\
\xa2\xa2\xa2\xa4\xfc\xd1\xd1Q\x1c:t\x08\xa1P\x08\x9a\xa6\xe9\x97.]\xfa\x8fa\
\x18\xbf\xc2\xcc\xd6\xfc]\x01d\x0c\xc3\x98J&\x93\x8f\xca\xb2,MLL\x90\xe1\xe1\
a444\xa0\xbc\xbc\xfc\x8e\x86\x06\x06\x06`\xb3\xd9J\x02\x88D"\xf8\xf0\xc3\x0f\
q\xf9\xf2eh\x9af\\\xbdz5\x91N\xa7\xf7\x038\x8fEv$\x96\x02 \x98\xd9\x0b\xb5\
\x01\x18T\x14\xc5\xa6(J\x8b\xc3\xe1\x10\xc6\xc6\xc6H\x7f\x7f?dYFUU\xd5\x8a\
\x00tww\xe3\xc8\x91#\xb8r\xe5\n4M3n\xdc\xb8\x91\x9e\x98\x98\xf8\x0b\x80#9\
\x8f$\x07\xb1`\x18-\x06@\x00X\x01XrG+\x80k\xe9t\xda\x9eN\xa7\x9beY\x16b\xb1\
\x18\xe9\xeb\xebC<\x1eGuu5\xacV\xeb]\x01\x8c\x8f\x8f\xe3\xe3\x8f?\xc6\xa7\
\x9f~\x8ap8\x0cJ\xa9q\xfd\xfa\xf5t<\x1e?\x06\xe0\x83\\\x1aWp\\\x00Ql\x92c.\
\x00\xb0`.\xcc/,\x16\xcb\x9e\x96\x96\x16\xbb(\x8a\x82\xd9l\xe6jkk\xb1i\xd3&\
\xb4\xb7\xb7\xc3\xef\xf7/X\xe4\xcc\x17\xa5\x14\x83\x83\x83\xe8\xe9\xe9\xc1\
\xd7_\x7f\x8dH$\x02EQ\x98\xaa\xaa\xfa\x95+W\x12\x99L\xe6#\x00\x7f\xc3\xcc\
\xac8\x83\x99\x19h\xe1\xb9r\'\x80\xf9\xa6\xe7\x1f\xd7r\x1c\xf7\xeb\xba\xba:\
\x7fMM\x8dH\x08!\xa2(rN\xa7\x13\x95\x95\x95hjj\x82\xd7\xeb\x85\xc3\xe1\x80\
\xd9l\x06\x80\xfcv!FFF\xd0\xd7\xd7\x87\x91\x91\x11LNNB\xd7uf\x18\x06\xbdy\
\xf3\xa66444\x08\xe0 \x80\xab\x05\xa6\x17;.\t`Z\xa4\xf6-\x05a\x03\xb0\x83\
\xe7\xf9m\xb5\xb5\xb5\x15>\x9f\x8f7\x9b\xcd|6\x9b\x85(\x8a\x9c$I\x10E\x11\
\x82 \xccN\x9f5M\x83\xaa\xaa\xd0u\x9d\x89\xa2\x88l6kD\xa3Q}hhh\xdc0\x8c.\xcc\
\xec\x7f\xa6\nj\xbb\x98\xf1\x0cf~(\\\x12\x00E\x0c\x9b\x8b\x9c\x9br A\x00?\
\xb6\xdb\xed~\x8f\xc7#9\x1c\x0e"\x8a"!\x84\xcc)\x9bR\xca4M\xa3SSSt||<;==}\
\x133#\xcc\xb9\x9c\xc1l\xce\xa0R`\xb6\xf0<\x0f2GKM\xf4\xf3F\xcdE\xc2T\x10\
\x12f\xb6\xfd\xdc\x00\xd6\x01h\x02P\x01\xc0\x8e\xdb\xdb\x81\x1a\x80i\xcc,Hz\
\x01\\\x02\x10\xcb]Ws\xe6\xf3\xa1\x14\x89<\xcc\x02\xddi\xa5B\x161\x9d7\x9e7/\
bfq$`fd\xe3se\xe7\xcbg\xb90r\xa1\xe7B+\x80\x98\x0fR\x08\xb3\xe8\xeah9K-i\x11\
\xe3\x8b\x99\x9f?\x1c\xd1E \xf4\x9c\xf9\xf9 j)\xa6\xee\xf6\x17\n\x923,\xcd3O\
\n\xa2X\x0b\xd0\x82(\x84Ps\xc7e\xafC\xef\xd7_\r\n_>\xc0\xed\x97\x0f\xc3\n\
\xff\xd5\xe0\x07\xfd\xa0{\xd4\x7f\x01~{\x04\xaaE\xe08P\x00\x00\x00\x00IEND\
\xaeB`\x82\x8f\xa2T\x8e'
