/***************************************************************************
 *   Copyright (C) 2005-2006 Nicolas Hadacek <hadacek@kde.org>             *
 *   Copyright (C) 2004 Alain Gibaud <alain.gibaud@free.fr>                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "tools_config_widget.h"

#include <qwidgetstack.h>
#include <qlabel.h>

#include "tool_list.h"
#include "tools/gui/tool_config_widget.h"
#include "tools/gui/tool_group_ui.h"
#include "compile_config.h"
#include "compile_process.h"
#include "libgui/project.h"

//----------------------------------------------------------------------------
HelpDialog::HelpDialog(QWidget *parent)
  : Dialog(parent, "help_dialog", false, i18n("Help Dialog"), Close, Close, false)
{
  QGridLayout *top = new QGridLayout(mainWidget(), 1, 1, 10, 10);
  uint k = 0;
  for (; Compile::ARGUMENT_DATA[k].key; k++) {
    QLabel *label = new QLabel(Compile::ARGUMENT_DATA[k].key, mainWidget());
    top->addWidget(label, k, 0);
    label = new QLabel(i18n(Compile::ARGUMENT_DATA[k].description), mainWidget());
    top->addWidget(label, k, 1);
  }
  top->setColStretch(2, 1);
  top->setRowStretch(k, 1);
}

//----------------------------------------------------------------------------
ToolsConfigWidget::ToolsConfigWidget(Project *project, QWidget *parent)
  : ::ConfigWidget(parent), _project(project), _helpDialog(0)
{
  uint row = 0;

  QLabel *label = new QLabel(i18n("Toolchain:"), this);
  addWidget(label, row,row, 0,0);
  _tool = new KeyComboBox<QString>(this);
  Tool::Lister::ConstIterator it;
  for (it=Tool::lister().begin(); it!=Tool::lister().end(); ++it)
    _tool->appendItem(it.key(), it.data()->label());
  connect(_tool->widget(), SIGNAL(activated(int)), SLOT(toolChanged()));
  addWidget(_tool->widget(), row,row, 1,1);
  label = new QLabel(i18n("Output type:"), this);
  addWidget(label, row,row, 2,2);
  _output = new KeyComboBox<Tool::OutputType>(this);
  FOR_EACH(Tool::OutputType, type) _output->appendItem(type, type.label());
  addWidget(_output->widget(), row,row, 3,3);
  row++;

  _mainStack = new QWidgetStack(this);
  addWidget(_mainStack, row,row, 0,4);
  row++;
  _tabWidget = new QTabWidget(_mainStack);
  _mainStack->addWidget(_tabWidget);
  FOR_EACH(Tool::Category, category) {
    _stacks[category] = new KeyWidgetStack<QString>(_tabWidget);
    _stacks[category]->widget()->setMargin(10);
    Tool::Lister::ConstIterator it;
    for (it=Tool::lister().begin(); it!=Tool::lister().end(); ++it) {
      if ( it.data()->isCustom() ) continue;
      if ( !it.data()->needs(_project, category) ) continue;
      ToolConfigWidget *cw = static_cast<const Tool::GroupUI *>(it.data()->gui())->createConfigWidget(category, project);
      Q_ASSERT(cw);
      _stacks[category]->appendItem(it.key(), cw);
      connect(cw, SIGNAL(displayHelp()), SLOT(displayHelp()));
    }
  }
  _customWidget = new QWidget(_mainStack);
  _mainStack->addWidget(_customWidget);
  QVBoxLayout *vbox = new QVBoxLayout(_customWidget);
  label = new QLabel(i18n("Custom shell commands:"), _customWidget);
  vbox->addWidget(label);
  _commandsEditor = new EditListBox(1, _customWidget, "command_editor", EditListBox::DuplicatesAllowed);
  vbox->addWidget(_commandsEditor);

  setColStretch(4, 1);
}

void ToolsConfigWidget::toolChanged()
{
  QString name = _tool->currentItem();
  bool canMakeLibrary = Tool::lister().group(name)->needs(_project, Tool::Category::Librarian);
  _output->widget()->setEnabled(canMakeLibrary);
  if ( !canMakeLibrary ) _output->setCurrentItem(Tool::OutputType::Executable);
  if ( name==Tool::Group::CUSTOM_NAME ) _mainStack->raiseWidget(_customWidget);
  else {
    _mainStack->raiseWidget(_tabWidget);
    FOR_EACH(Tool::Category, k) {
      _tabWidget->removePage(_stacks[k]->widget());
      _stacks[k]->widget()->hide();
      if ( _stacks[k]->contains(name) ) {
        _stacks[k]->setCurrentItem(name);
        _stacks[k]->widget()->show();
        _tabWidget->addTab(_stacks[k]->widget(), k.label());
      }
    }
    _tabWidget->setCurrentPage(0);
  }
}

void ToolsConfigWidget::loadConfig()
{
  const Tool::Group &group = Compile::Config::toolGroup(_project);
  _tool->setCurrentItem(group.name());
  _output->setCurrentItem(_project->outputType());
  QStringList commands = Compile::Config::customCommands(_project);
  _commandsEditor->setTexts(commands);
  toolChanged();
  FOR_EACH(Tool::Category, k) {
    KeyWidgetStack<QString>::ConstIterator it;
    for (it=_stacks[k]->begin(); it!=_stacks[k]->end(); ++it) {
      if ( it.key()==Tool::Group::CUSTOM_NAME ) continue;
      static_cast<ToolConfigWidget *>(_stacks[k]->item(it.key()))->loadConfig();
    }
  }
}

void ToolsConfigWidget::saveConfig()
{
  QString name = _tool->currentItem();
  Compile::Config::setToolGroup(_project, *Tool::lister().group(name));
  _project->setOutputType(_output->currentItem());
  Compile::Config::setCustomCommands(_project, _commandsEditor->texts());
  FOR_EACH(Tool::Category, k) {
    if ( !_stacks[k]->contains(name) ) continue;
    QWidget *w = _stacks[k]->item(name);
    static_cast<ToolConfigWidget *>(w)->saveConfig();
  }
}

void ToolsConfigWidget::displayHelp()
{
  if ( _helpDialog.isNull() ) _helpDialog = new HelpDialog(this);
  _helpDialog->show();
}
