/***********************************************************************************
* Fancy Tasks: Plasmoid for fancy representing your tasks and launchers.
* Copyright (C) 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "FancyTasksJob.h"

FancyTasksJob::FancyTasksJob(const QString &job, FancyTasksApplet *applet) : QObject(applet),
    m_applet(applet),
    m_job(job),
    m_state(Unknown),
    m_percentage(0),
    m_closeOnFinish(false)
{
    m_applet->dataEngine("applicationjobs")->connectSource(m_job, this, 250, Plasma::NoAlignment);

    dataUpdated("", m_applet->dataEngine("applicationjobs")->query(m_job));
}

void FancyTasksJob::dataUpdated(const QString &source, const Plasma::DataEngine::Data &data)
{
    Q_UNUSED(source)

     FancyTasksApplet::Changes changes = FancyTasksApplet::TextChanged;

    if (m_iconName.isEmpty())
    {
        m_iconName = data["appIconName"].toString();

        changes |= FancyTasksApplet::IconChanged;
    }

    m_title = data["infoMessage"].toString();
    m_application = data["appName"].toString();
    m_percentage = data["percentage"].toInt();
    m_suspendable = data["suspendable"].toBool();
    m_killable = data["killable"].toBool();

    QString state = data["state"].toString();
    JobState previousState = m_state;

    if (!data["error"].toString().isEmpty())
    {
        m_title = data["error"].toString();

        if (m_state != Error)
        {
            emit demandsAttention();
        }

        m_state = Error;
    }
    else if (state == "running" && m_percentage > 0)
    {
        m_state = Running;
    }
    else if (state == "suspended")
    {
        m_state = Suspended;
    }
    else if (state == "stopped")
    {
        m_state = Finished;
    }

    QString description;

    if (m_state == Running || m_state == Suspended || m_state == Unknown)
    {
        m_title = ((m_state == Suspended)?i18n("%1 [Paused]", m_title):m_title);

        if (data["eta"].toUInt() > 0)
        {
            description.append(i18n("%1 (%2 remaining) ", data["speed"].toString(), KGlobal::locale()->prettyFormatDuration(data["eta"].toUInt())) + "<br />");
        }
    }
    else
    {
        setFinished(true);
    }

    m_description.clear();
    m_description.append(description);

    int i = 0;

    while (data.contains(QString("label%1").arg(i)))
    {
        m_description.append(data[QString("labelName%1").arg(i)].toString() + ": <i>" + data[QString("label%1").arg(i)].toString() + "</i><br>");

        ++i;
    }

    m_description = m_description.left(m_description.size() - 4);
    m_information = QString("<b>%1</b> %2").arg(m_percentage?QString("%1 %2%").arg(m_title).arg(m_percentage):m_title).arg(description);

    if (previousState != m_state)
    {
        changes |= FancyTasksApplet::StateChanged;
    }

    emit changed(changes);
}

void FancyTasksJob::setFinished(bool finished)
{
    if (finished)
    {
        m_title = i18n("%1 [Finished]", m_title);
        m_information = QString("<b>%1</b>").arg(m_title);
        m_state = Finished;
    }

    emit demandsAttention();
}

void FancyTasksJob::setCloseOnFinish(bool close)
{
    m_closeOnFinish = close;
}

void FancyTasksJob::suspend()
{
    Plasma::Service *service = m_applet->dataEngine("applicationjobs")->serviceForSource(m_job);
    service->startOperationCall(service->operationDescription("suspend"));
}

void FancyTasksJob::resume()
{
    Plasma::Service *service = m_applet->dataEngine("applicationjobs")->serviceForSource(m_job);
    service->startOperationCall(service->operationDescription("resume"));
}

void FancyTasksJob::stop()
{
    Plasma::Service *service = m_applet->dataEngine("applicationjobs")->serviceForSource(m_job);
    service->startOperationCall(service->operationDescription("stop"));
}

void FancyTasksJob::close()
{
    m_applet->removeJob(m_job, true);
}

void FancyTasksJob::destroy()
{
    emit close(this);

    deleteLater();
}

KMenu* FancyTasksJob::contextMenu()
{
    KMenu *menu = new KMenu;

    if (m_state == Finished || m_state == Error)
    {
        QAction *closeAction = menu->addAction(KIcon("window-close"), i18n("Close Job"));

        connect(closeAction, SIGNAL(triggered()), this, SLOT(close()));
    }
    else
    {
        if (m_suspendable && m_state != Unknown)
        {
            if (m_state == Running)
            {
                QAction *suspendAction = menu->addAction(KIcon("media-playback-pause"), i18n("Pause Job"));

                connect(suspendAction, SIGNAL(triggered()), this, SLOT(suspend()));
            }
            else
            {
                QAction *resumeAction = menu->addAction(KIcon("media-playback-start"), i18n("Resume Job"));

                connect(resumeAction, SIGNAL(triggered()), this, SLOT(resume()));
            }
        }

        if (m_killable)
        {
            if (m_state != Unknown)
            {
                menu->addSeparator();
            }

            QAction *stopAction = menu->addAction(KIcon("media-playback-stop"), i18n("Cancel Job"));
            stopAction->setEnabled(m_state != Finished && m_state != Error);

            connect(stopAction, SIGNAL(triggered()), this, SLOT(stop()));
        }
    }

    menu->addSeparator();

    QAction *closeOnFinishAction = menu->addAction(i18n("Close on Finish"));
    closeOnFinishAction->setCheckable(true);
    closeOnFinishAction->setChecked(m_closeOnFinish && m_state != Error);
    closeOnFinishAction->setEnabled(m_state != Finished && m_state != Error);

    connect(closeOnFinishAction, SIGNAL(toggled(bool)), this, SLOT(setCloseOnFinish(bool)));

    return menu;
}

FancyTasksJob::JobState FancyTasksJob::state() const
{
    return m_state;
}

KIcon FancyTasksJob::icon()
{
    return KIcon(m_iconName);
}

QString FancyTasksJob::title() const
{
    return m_title;
}

QString FancyTasksJob::description() const
{
    return m_description;
}

QString FancyTasksJob::information() const
{
    return m_information;
}

QString FancyTasksJob::application() const
{
    return m_application;
}

int FancyTasksJob::percentage() const
{
    return ((m_percentage >= 0)?m_percentage:0);
}

bool FancyTasksJob::closeOnFinish() const
{
    return m_closeOnFinish;
}
