/*
 * Plasma applet to display indicators from libindicate
 *
 * Copyright 2009 Canonical Ltd.
 *
 * Authors:
 * - Aurélien Gâteau <aurelien.gateau@canonical.com>
 *
 * License: GPL v3
 */
// Self
#include "indicatordisplay.h"

// Qt
#include <QGraphicsLinearLayout>
#include <QLayout>
#include <QRegExp>

// KDE
#include <KGlobalSettings>
#include <KIcon>
#include <Plasma/IconWidget>
#include <Plasma/Theme>
#include <Plasma/ToolTipManager>

// Local
#include "delegate.h"
#include "view.h"

//#define DUMP_MODELS

static const char* NO_NEW_STUFF_ICON = "mail-unread";
static const char* NEW_STUFF_ICON = "mail-unread-new";

K_EXPORT_PLASMA_APPLET(indicatordisplay, IndicatorDisplay)


IndicatorDisplay::IndicatorDisplay(QObject* parent, const QVariantList& args)
: Plasma::PopupApplet(parent, args)
, mListener(0)
, mSourceModel(0)
, mView(new ExpandedTreeView())
, mIconWidget(new Plasma::IconWidget(this))
{
    setBackgroundHints(StandardBackground);
    setAspectRatioMode(Plasma::Square);
}

IndicatorDisplay::~IndicatorDisplay()
{
    removeInterestOnServers();
    delete mView;
}

void IndicatorDisplay::init()
{
    Plasma::ToolTipManager::self()->registerWidget(this);
    mListener = new QIndicate::Listener(this);
    connect(mListener,
            SIGNAL(serverAdded(QIndicate::Listener::Server*, const QString&)),
            SLOT(slotServerAdded(QIndicate::Listener::Server*))
            );

    initSourceModel();
    initView();
    initIcon();
}

void IndicatorDisplay::initIcon()
{
    // Note: we use an icon widget, not the builtin setPopupIcon() because
    // otherwise our view might get embedded if there is enough space, and we do
    // not want that. Therefore we need to implement the icon look and behavior
    // ourself
    QGraphicsLinearLayout* layout = new QGraphicsLinearLayout(this);
    layout->setSpacing(0);
    layout->addItem(mIconWidget);

    connect(mIconWidget, SIGNAL(clicked()), SLOT(togglePopup()));
    updateIcon(false /* no new stuff */);
    updateIconState();
}

void IndicatorDisplay::initSourceModel()
{
    // Reg exp is a hack to avoid regressions while changing app server types
    // from "messaging" to "message.<something>"
    mSourceModel = new ListenerModel(mListener, QRegExp("^messag(e|ing)"));
    connect(mSourceModel, SIGNAL(rowsInserted(const QModelIndex&, int, int)),
            SLOT(slotRowsChanged(const QModelIndex&))
            );
    connect(mSourceModel, SIGNAL(rowsRemoved(const QModelIndex&, int, int)),
            SLOT(slotRowsChanged(const QModelIndex&))
            );
    connect(mSourceModel, SIGNAL(drawAttentionChanged(const QModelIndex&)),
            SLOT(slotDrawAttentionChanged())
            );
    #ifdef DUMP_MODELS
    connect(mSourceModel, SIGNAL(dataChanged(const QModelIndex&, const QModelIndex&)),
            SLOT(dumpModels())
            );
    #endif
}

void IndicatorDisplay::initView()
{
    mView->setModel(mSourceModel);

    mView->setItemDelegate(new Delegate(this));
    mView->setSelectionMode(QAbstractItemView::NoSelection);

    initPalette();
    connect(Plasma::Theme::defaultTheme(), SIGNAL(themeChanged()), SLOT(initPalette()));
    connect(KGlobalSettings::self(), SIGNAL(kdisplayPaletteChanged()), SLOT(initPalette()));
    mView->setFrameStyle(QFrame::NoFrame);
    mView->setRootIsDecorated(false);
    mView->setHeaderHidden(true);
    mView->setIndentation(16);
    mView->setIconSize(QSize(16, 16));

    // Disable scrollbars: we compute the size of the view so that its content
    // fits without scrollbars, but if we do not disable them a vertical
    // scrollbar sometimes appears when the view grows while visible
    mView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    mView->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    mView->setEditTriggers(QAbstractItemView::NoEditTriggers);
    connect(mView, SIGNAL(clicked(const QModelIndex&)),
            SLOT(slotClicked(const QModelIndex&))
            );
    connect(mView, SIGNAL(sizeChanged()),
            SLOT(adjustViewSize())
            );
}

void IndicatorDisplay::initPalette()
{
    QPalette pal = mView->palette();
    pal.setColor(QPalette::Base, Qt::transparent);
    QColor textColor = Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor);
    pal.setColor(QPalette::Text, textColor);
    mView->setPalette(pal);
}

QWidget* IndicatorDisplay::widget()
{
    return mView;
}

void IndicatorDisplay::toolTipAboutToShow()
{
    Plasma::ToolTipContent toolTip;
    toolTip.setImage(mIconWidget->icon());
    toolTip.setMainText(i18n("Indicator Display"));
    int appCount = mSourceModel->rowCount();
    if (appCount == 0) {
        toolTip.setSubText(i18n("No applications running"));
    } else {
        toolTip.setSubText(i18np("One application running", "%1 applications running", appCount));
    }
    Plasma::ToolTipManager::self()->setContent(this, toolTip);
}

void IndicatorDisplay::adjustViewSize()
{
    QSize sh = mView->sizeHint();

    QWidget* dialog = mView->parentWidget();
    int left, top, right, bottom;
    dialog->getContentsMargins(&left, &top, &right, &bottom);

    dialog->setFixedSize(sh.width() + left + right, sh.height() + top + bottom);

    // Hack: Plasma::Dialog only emits dialogResized() if its content is a
    // QGraphicsWidget. Emit it ourself to ensure the dialog is correctly
    // positioned.
    QMetaObject::invokeMethod(dialog, "dialogResized");
}

void IndicatorDisplay::popupEvent(bool show)
{
    if (show) {
        adjustViewSize();
    }
}

void IndicatorDisplay::slotClicked(const QModelIndex& index)
{
    QIndicate::Listener::Server* server = 0;
    QIndicate::Listener::Indicator* indicator = 0;

    mSourceModel->getProxiesForIndex(index, &server, &indicator);
    mListener->display(server, indicator);

    hidePopup();
}

void IndicatorDisplay::slotRowsChanged(const QModelIndex& parent)
{
    if (!parent.isValid()) {
        // A server has been added or removed
        updateIconState();
        return;
    }

    updateIconFromIndicators();

    #ifdef DUMP_MODELS
    dumpModels();
    #endif
}

void IndicatorDisplay::slotDrawAttentionChanged()
{
    updateIconFromIndicators();

    #ifdef DUMP_MODELS
    dumpModels();
    #endif
}

void IndicatorDisplay::updateIcon(bool newStuff)
{
    mIconWidget->setIcon(newStuff ? NEW_STUFF_ICON : NO_NEW_STUFF_ICON);
}

void IndicatorDisplay::updateIconFromIndicators()
{
    // Check if one of the indicators want to draw attention
    QModelIndexList lst = mSourceModel->match(mSourceModel->index(0, 0),
                                              ListenerModel::IndicatorDrawAttentionRole,
                                              QVariant(true),
                                              1 /* hits */,
                                              Qt::MatchExactly | Qt::MatchRecursive);

    updateIcon(!lst.isEmpty());
}

void IndicatorDisplay::updateIconState()
{
    if (mSourceModel->rowCount() == 0) {
        hidePopup();
        mIconWidget->setEnabled(false);
    } else {
        mIconWidget->setEnabled(true);
    }
}

void IndicatorDisplay::slotServerAdded(QIndicate::Listener::Server* server)
{
    mListener->setInterest(server, QIndicate::InterestServerDisplay, true);
    mListener->setInterest(server, QIndicate::InterestServerSignal, true);
}

void IndicatorDisplay::removeInterestOnServers()
{
    for (int row = mSourceModel->rowCount() - 1; row >= 0; --row) {
        QIndicate::Listener::Server* server = 0;
        QIndicate::Listener::Indicator* indicator = 0;
        QModelIndex index = mSourceModel->index(row, 0);
        mSourceModel->getProxiesForIndex(index, &server, &indicator);
        if (server) {
            mListener->setInterest(server, QIndicate::InterestServerDisplay, false);
            mListener->setInterest(server, QIndicate::InterestServerSignal, false);
        } else {
            kWarning() << "No server for row" << row;
        }
    }
}

#ifdef DUMP_MODELS
#include "modeldump.h"

using namespace ModelDump;

void IndicatorDisplay::dumpModels()
{
    RoleDumperHash hash;
    hash.insert(Qt::DisplayRole, new SimpleRoleDumper("display"));
    hash.insert(Qt::DecorationRole, new SimpleRoleDumper("decoration"));
    hash.insert(ListenerModel::ServerTypeRole, new SimpleRoleDumper("type"));
    hash.insert(ListenerModel::CountRole, new SimpleRoleDumper("count"));
    hash.insert(ListenerModel::IndicatorDateTimeRole, new SimpleRoleDumper("time"));
    hash.insert(ListenerModel::IndicatorDrawAttentionRole, new SimpleRoleDumper("attention"));
    hash.insert(Qt::UserRole + 1, new PointerRoleDumper<QIndicate::Listener::Server>("server"));
    hash.insert(Qt::UserRole + 2, new PointerRoleDumper<QIndicate::Listener::Indicator>("indicator"));
    hash.insert(Qt::UserRole + 3, new SimpleRoleDumper("outdated"));
    qDebug();
    qDebug() << "## mSourceModel";
    dump(hash, mSourceModel);
}
#else
void IndicatorDisplay::dumpModels()
{
}
#endif

#include "indicatordisplay.moc"
