/***********************************************************************************
* STasks
* Copyright (C) 2009 Marcin Baszczewski <marcin.baszczewski@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

//OWN
#include "stasks_applet.h"
#include "stasks_item.h"
#include "stasks_task.h"
#include "stasks_frame.h"
#include "stasks_tooltip.h"
//PLASMA
#include <Plasma/Theme>
#include <Plasma/ToolTipManager>
#include <Plasma/Containment>
#include <Plasma/FrameSvg>
//QT
#include <QGraphicsLinearLayout>
#include <QGraphicsSceneMouseEvent>
//KDE
#include <KLocale>
#include <KRun>
#include <KMenu>
#include <KConfigDialog>
#include <KWindowSystem>




STasksApplet::STasksApplet(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args)
{
    setAspectRatioMode(Plasma::IgnoreAspectRatio);
    setHasConfigurationInterface(true);
    setAcceptDrops(true);
    
    if (formFactor() == Plasma::Vertical)
    resize(40, 800);
    else
    resize(800, 40);
}
STasksApplet::~STasksApplet()
{
    delete m_groupManager;
    delete m_frame;
}
void STasksApplet::init()
{
    m_frame = new Plasma::FrameSvg(this);
    m_frame->setImagePath("widgets/tasks");
    m_frame->setCacheAllRenderedFrames(false);
    m_frame->setEnabledBorders(Plasma::FrameSvg::AllBorders);
    m_frame->setElementPrefix("normal");
    m_toolTip = new STasksToolTip( 200, qobject_cast<QWidget*>(this));

    m_groupManager = new TaskManager::GroupManager(this);

    Plasma::Containment* appletContainment = containment();
    if (appletContainment) m_groupManager->setScreen(appletContainment->screen());
    
    connect(this, SIGNAL(settingsChanged()), m_groupManager, SLOT(reconnect()));
    connect(m_groupManager->rootGroup(), SIGNAL(itemAdded(AbstractItemPtr)), this, SLOT(itemAdded(AbstractItemPtr)));
    connect(m_groupManager->rootGroup(), SIGNAL(itemRemoved(AbstractItemPtr)), this, SLOT(itemRemoved(AbstractItemPtr)));
    connect(m_groupManager->rootGroup(), SIGNAL(itemPositionChanged(AbstractItemPtr)), SLOT(itemPositionChanged(AbstractItemPtr)));
    connect(this, SIGNAL(settingsChanged()), this, SLOT(configuration()));

    m_layout = new QGraphicsLinearLayout(this);
    m_layout->setContentsMargins(0, 0, 0, 0);
    m_layout->addStretch();
    setLayout(m_layout);
    emit settingsChanged();
    updateSize();
    setSizePolicy(QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding));
    setMaximumSize(INT_MAX,INT_MAX);
}
void STasksApplet::itemAdded(TaskManager::AbstractItemPtr groupableItem)
{
    STasksItem *item = new STasksItem(groupableItem, this);
    if (m_items.contains(item)) 
    {
        kDebug() << "already exist";
        return;
    }
    item->setSize(m_iconSize);
    item->setParentItem(this);
    connect(this, SIGNAL(setSize(qreal)), item, SLOT(setSize(qreal)));
    connect(item, SIGNAL(itemActive(STasksItem*)), SLOT(updateActiveIconIndex(STasksItem*)));
    connect(TaskManager::TaskManager::self(),SIGNAL(desktopChanged(int)),item,SLOT(updateState()) );
    m_items.append(item);
    m_layout->insertItem(m_layout->count()-1,item);
    m_layout->activate();
    m_tasksHash[groupableItem] = item;

}
void STasksApplet::itemRemoved(TaskManager::AbstractItemPtr groupableItem)
{
    STasksItem* item = m_tasksHash[groupableItem];
    if(item == 0) 
    {
	kDebug() << "trying to remove non-existant task";
	return;
    }
    m_tasksHash.remove(groupableItem);
    m_items.removeAll(item);
    m_layout->removeItem(item);
    m_layout->activate();
    delete item;
}
void STasksApplet::itemPositionChanged(AbstractItemPtr groupableItem)
{
    STasksItem* item = m_tasksHash[groupableItem];
    if (!m_items.contains(item))
    return;
    int index = m_groupManager->rootGroup()->members().indexOf(groupableItem);
    
    if (m_layout->itemAt(index)==item)
    return;
    m_items.move(m_items.indexOf(item),index);
    m_layout->removeItem(item);
    m_layout->insertItem(index,item);
}
void STasksApplet::reload()
{
    m_items.clear();
    m_tasksHash.clear();
    
    foreach(TaskManager::AbstractItemPtr item,m_groupManager->rootGroup()->members() ) 
    {
	itemAdded(item);
    }
}
void STasksApplet::constraintsEvent(Plasma::Constraints constraints)
{
    if (constraints & Plasma::SizeConstraint)
    {
        updateSize();
    }
    if (constraints & Plasma::LocationConstraint)
    {
        m_layout->setOrientation((formFactor() == Plasma::Vertical)?Qt::Vertical:Qt::Horizontal);
    }
}
void STasksApplet::updateSize()
{
    if (formFactor() == Plasma::Vertical)
    {
        m_iconSize = contentsRect().width();
    }
    else
    {
        m_iconSize = contentsRect().height();
    }
    emit setSize(m_iconSize);
}
void STasksApplet::dropEvent(QGraphicsSceneDragDropEvent *event)
{
    KUrl::List urls = KUrl::List::fromMimeData(event->mimeData());
    if (urls.count()>0)
    {
	kDebug() << "shortcut :) ";
	return;
    }
    if ((event->mimeData()->hasFormat("STasksItem")) && (sortingStrategy()==TaskManager::GroupManager::ManualSorting))
    {
	QByteArray data;
	data = event->mimeData()->data("STasksItem");
 	STasksItem *temp;
	memcpy(&temp,data, sizeof(STasksItem*));
	m_layout->removeItem(temp);
	
	int index = -1;
	if (formFactor() == Plasma::Vertical)
	{
	    foreach(STasksItem* item, m_items) 
	    {
		if ( (event->scenePos().y()) >= (temp->scenePos().y()-(m_taskSpacing/2)) && ( (event->scenePos().y()) < (temp->scenePos().y()+temp->boundingRect().height()+(m_taskSpacing/2))) )
		{
		    index = m_items.indexOf(temp);
		    break;
		}
		if ( (event->scenePos().y()) >= (item->scenePos().y()-(m_taskSpacing/2)) && ( (event->scenePos().y()) < (item->scenePos().y()+item->boundingRect().height()+(m_taskSpacing/2))) )
		{
		    index = m_items.indexOf(item);
		    break;
		}
	    }
	}
	else
	{
	    foreach(STasksItem* item, m_items) 
	    {
		if ( (event->scenePos().x()) >= (temp->scenePos().x()-(m_taskSpacing/2)) && ( (event->scenePos().x()) < (temp->scenePos().x()+temp->boundingRect().width()+(m_taskSpacing/2))) )
		{
		    index = m_items.indexOf(temp);
		    break;
		}
		if ( (event->scenePos().x()) >= (item->scenePos().x()-(m_taskSpacing/2)) && ( (event->scenePos().x()) < (item->scenePos().x()+item->boundingRect().width()+(m_taskSpacing/2))) )
		{
		    index = m_items.indexOf(item);
		    break;
		}
	    }
	}
	if (index==-1)
	{
	    index=m_layout->count()-1;
	}

	int change=m_items.indexOf(temp);
	m_items.move(change,index);
	m_layout->insertItem(index,temp);
	m_groupManager->rootGroup()->moveItem(change,index);
    }
}
int STasksApplet::activeIndex()
{
    int index = 0;
    foreach(STasksItem *item, m_items) 
    {
	if (item->task()->type()==STasksTask::Group)
	{
            for (int i=0;i<item->task()->group()->members().count();i++)
	    {
		if ( !item->task()->group()->members().at(i)->isActive() )
		index++;
		else
		return index;
	    }
	}
	else
        {
	    if ( !item->task()->isActive() )
	    index++;
	    else
	    return index;
	}
    }
    return index;
}
void STasksApplet::updateActiveIconIndex(STasksItem *item)
{
    Q_UNUSED(item);
    //m_activeIconIndex = m_items.indexOf(item);
    m_activeIconIndex = activeIndex();
}
int STasksApplet::totalSubTasks()
{
    int count = 0;
    foreach(STasksItem *item, m_items) 
    {
	if (item->task()->type()==STasksTask::Group)
	{
	    count += item->task()->group()->members().count();
	}
	else
	count++;
    }
    return count;
}
AbstractItemPtr STasksApplet::selectSubTask(int index)
{

    foreach(STasksItem *item, m_items) 
    {
	if (item->task()->type()==STasksTask::Group)
	{
	    if (index < item->task()->group()->members().count()) 
	    {
		return item->task()->group()->members().at(index);
		//return m_tasksHash[item->task()->group()->members().at(index)];
	    } 
	    else 
	    {
		index -= item->task()->group()->members().count();
	    }
	} 
	else if (index == 0) 
	{
	    return item->task()->abstractItem();
	} 
	else 
	{
	    --index;
	}
    }
    return NULL;
}
void STasksApplet::wheelEvent(QGraphicsSceneWheelEvent *event)
{
//     event->accept();
// 
//     int steps = (event->delta() / 120);
// 
//     m_activeIconIndex += steps;
// 
//     while (m_activeIconIndex < 0 || m_activeIconIndex >= m_items.count())
//     {
//         if (m_activeIconIndex < 0)
//         {
//             m_activeIconIndex += m_items.count();
//         }
//         else
//         {
//             m_activeIconIndex -= m_items.count();
//         }
//     }
//     if (event->modifiers() & Qt::ShiftModifier)
//     {
//         return;
//     }
//     m_items.at(m_activeIconIndex)->activate();
    int subTasks = totalSubTasks();
    //zero or one tasks don't cycle
    if (subTasks < 1) 
    {
        return;
    }

    //mouse wheel down
    if (event->delta() < 0) {
        m_activeIconIndex++;
        if ( m_activeIconIndex >= subTasks) {
             m_activeIconIndex = 0;
        }
        //mouse wheel up
    } else {
         m_activeIconIndex--;
        if ( m_activeIconIndex < 0) {
             m_activeIconIndex = subTasks - 1; //last item is a spacer
        }
    }

    //kDebug() << "Wheel event m_activeTaskIndex: " << m_activeTaskIndex << " of " << subTasks;
    AbstractItemPtr taskItem = selectSubTask( m_activeIconIndex);
    if (taskItem) 
    {
	if (!taskItem->isGroupItem())
	{
	    TaskManager::TaskItem* task = static_cast<TaskManager::TaskItem*>(taskItem);
	    if (task->task())
	    task->task()->activate();
	}
    }
}
void STasksApplet::expandTasksChanged(int state)
{
    if (state)
    {
	m_configA.expandingSize->setEnabled(true);
	m_configA.keepExpanded->setEnabled(true);
    }
    else
    {
	m_configA.expandingSize->setEnabled(false);
	m_configA.keepExpanded->setEnabled(false);
    }
}
void STasksApplet::lightColorChanged(int state)
{
    if (state)
    {
	m_configA.lightColor->setEnabled(true);
    }
    else
    {
	m_configA.lightColor->setEnabled(false);
    }
}
void STasksApplet::createConfigurationInterface(KConfigDialog *parent)
{
    connect(parent, SIGNAL(applyClicked()), this, SLOT(configAccepted()));
    connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));

    QWidget *widgetG = new QWidget;
    m_configG.setupUi(widgetG);
    parent->addPage(widgetG, i18n("General"), icon());

    QWidget *widgetA = new QWidget;
    m_configA.setupUi(widgetA);
    parent->addPage(widgetA, i18n("Appearance"), "preferences-desktop-theme");

    //general
    if (m_expandTasks)
    m_configA.expandTasks->setCheckState(Qt::Checked);
    else
    m_configA.expandTasks->setCheckState(Qt::Unchecked);
    connect(m_configA.expandTasks, SIGNAL(stateChanged(int)), this, SLOT(expandTasksChanged(int)));
    expandTasksChanged(m_expandTasks);

    //filters
    m_configG.showOnlyCurrentDesktop->setChecked(m_groupManager->showOnlyCurrentDesktop());
    m_configG.showOnlyCurrentScreen->setChecked(m_groupManager->showOnlyCurrentScreen());
    m_configG.showOnlyMinimized->setChecked(m_groupManager->showOnlyMinimized());

    //grouping & sorting
    m_configG.groupingStrategy->addItem(i18n("Do Not Group"),QVariant(TaskManager::GroupManager::NoGrouping));
    //m_configG.groupingStrategy->addItem(i18n("Manually"),QVariant(TaskManager::GroupManager::ManualGrouping));
    m_configG.groupingStrategy->addItem(i18n("By Program Name"),QVariant(TaskManager::GroupManager::ProgramGrouping));
    m_configG.sortingStrategy->addItem(i18n("Do Not Sort"),QVariant(TaskManager::GroupManager::NoSorting));
    m_configG.sortingStrategy->addItem(i18n("Manually"),QVariant(TaskManager::GroupManager::ManualSorting));
    m_configG.sortingStrategy->addItem(i18n("Alphabetically"),QVariant(TaskManager::GroupManager::AlphaSorting));
    m_configG.sortingStrategy->addItem(i18n("By Desktop"),QVariant(TaskManager::GroupManager::DesktopSorting));
    switch (m_groupManager->groupingStrategy()) 
    {
	case TaskManager::GroupManager::NoGrouping:
	    m_configG.groupingStrategy->setCurrentIndex(0);
	    break;
	case TaskManager::GroupManager::ProgramGrouping:
	    m_configG.groupingStrategy->setCurrentIndex(1);
	    break;
	default:
	    m_configG.groupingStrategy->setCurrentIndex(-1);
    }
    switch (m_groupManager->sortingStrategy()) 
    {
	case TaskManager::GroupManager::NoSorting:
	    m_configG.sortingStrategy->setCurrentIndex(0);
	    break;
	case TaskManager::GroupManager::ManualSorting:
	    m_configG.sortingStrategy->setCurrentIndex(1);
	    break;
	case TaskManager::GroupManager::AlphaSorting:
	    m_configG.sortingStrategy->setCurrentIndex(2);
	    break;
	case TaskManager::GroupManager::DesktopSorting:
	    m_configG.sortingStrategy->setCurrentIndex(3);
	    break;
	default:
	    m_configG.sortingStrategy->setCurrentIndex(-1);
    }

    //apearance
    m_configA.shape->addItem(i18n("Rectangle"),QVariant(0));
    m_configA.shape->addItem(i18n("Square"),QVariant(1));
    m_configA.taskSpacing->setValue(m_taskSpacing);
    m_configA.iconScale->setValue(m_iconScale);
    m_configA.lightColor->setColor(m_lightColor);
    connect(m_configA.lights, SIGNAL(stateChanged(int)), this, SLOT(lightColorChanged(int)));
    lightColorChanged(m_lights);
    m_configA.expandingSize->setValue(m_expandingSize);
    m_configA.animationDuration->setValue(m_animationDuration);    
    m_configA.shape->setCurrentIndex(m_shape);
    if (m_lights)
    m_configA.lights->setCheckState(Qt::Checked);
    else
    m_configA.lights->setCheckState(Qt::Unchecked);
    m_configA.keepExpanded->addItem(i18n("None"),0);
    m_configA.keepExpanded->addItem(i18n("Active"),1);
    m_configA.keepExpanded->addItem(i18n("From the current desktop"),2);
    m_configA.keepExpanded->addItem(i18n("All"),3);
    m_configA.keepExpanded->setCurrentIndex(m_keepExpanded);
}
void STasksApplet::configAccepted()
{
    bool changed = false;

    //genral
    if ( m_configA.expandTasks->checkState() != m_expandTasks ) 
    {
	KConfigGroup cg = config();
	
	if (m_configA.expandTasks->checkState()==Qt::Unchecked)
	cg.writeEntry("expandTasks", false);
	else
	cg.writeEntry("expandTasks", true);
        changed = true;
    }

    //grouping & strategy
    if (m_groupManager->sortingStrategy() != (m_configG.sortingStrategy->currentIndex())) 
    {
        m_groupManager->setSortingStrategy(static_cast<TaskManager::GroupManager::TaskSortingStrategy>(m_configG.sortingStrategy->itemData(m_configG.sortingStrategy->currentIndex()).toInt()));
        KConfigGroup cg = config();
        cg.writeEntry("sortingStrategy", static_cast<int>(m_groupManager->sortingStrategy()));
        changed = true;
    }
    if (m_groupManager->groupingStrategy() != (m_configG.groupingStrategy->currentIndex())) 
    {
        m_groupManager->setGroupingStrategy(static_cast<TaskManager::GroupManager::TaskGroupingStrategy>(m_configG.groupingStrategy->itemData(m_configG.groupingStrategy->currentIndex()).toInt()));
        KConfigGroup cg = config();
        cg.writeEntry("groupingStrategy", static_cast<int>(m_groupManager->groupingStrategy()));
        changed = true;
    }
    //apearance
    if ( m_configA.taskSpacing->value() != m_taskSpacing ) 
    {
	KConfigGroup cg = config();
        cg.writeEntry("taskSpacing", m_configA.taskSpacing->value());
        changed = true;
    }

    if ( m_configA.keepExpanded->currentIndex() != m_keepExpanded ) 
    {
	KConfigGroup cg = config();
	
	cg.writeEntry("keepExpanded", m_configA.keepExpanded->currentIndex());
        changed = true;
    }

    if ( m_configA.expandingSize->value() != m_expandingSize) 
    {
	KConfigGroup cg = config();
        cg.writeEntry("expandingSize", m_configA.expandingSize->value());
        changed = true;
    }

    if ( m_configA.iconScale->value() != m_iconScale ) 
    {
	KConfigGroup cg = config();
        cg.writeEntry("iconScale", m_configA.iconScale->value());
        changed = true;
    }
    if ( m_configA.lightColor->color() != m_lightColor ) 
    {
	KConfigGroup cg = config();
        cg.writeEntry("lightColor", m_configA.lightColor->color());
        changed = true;
    }
    if ( m_configA.animationDuration->value() != m_animationDuration ) 
    {
	KConfigGroup cg = config();
        cg.writeEntry("animationDuration", m_configA.animationDuration->value());
        changed = true;
    }

    if ( m_configA.lights->checkState() != m_lights ) 
    {
	KConfigGroup cg = config();
	
	if (m_configA.lights->checkState()==Qt::Unchecked)
	cg.writeEntry("lights", false);
	else
	cg.writeEntry("lights", true);
        changed = true;
    }

    if ( m_configA.shape->currentIndex() != m_shape ) 
    {
	KConfigGroup cg = config();
	cg.writeEntry("shape", m_configA.shape->currentIndex());
        changed = true;
    }
    
    //filters 
     if (m_groupManager->showOnlyCurrentDesktop() != (m_configG.showOnlyCurrentDesktop->isChecked())) {
        m_groupManager->setShowOnlyCurrentDesktop(!m_groupManager->showOnlyCurrentDesktop());
        KConfigGroup cg = config();
        cg.writeEntry("showOnlyCurrentDesktop", m_groupManager->showOnlyCurrentDesktop());
        changed = true;
    }
    if (m_groupManager->showOnlyCurrentScreen() != (m_configG.showOnlyCurrentScreen->isChecked())) {
        m_groupManager->setShowOnlyCurrentScreen(!m_groupManager->showOnlyCurrentScreen());
        KConfigGroup cg = config();
        cg.writeEntry("showOnlyCurrentScreen", m_groupManager->showOnlyCurrentScreen());
        changed = true;
    }
    if (m_groupManager->showOnlyMinimized() != (m_configG.showOnlyMinimized->isChecked())) {
        m_groupManager->setShowOnlyMinimized(!m_groupManager->showOnlyMinimized());
        KConfigGroup cg = config();
        cg.writeEntry("showOnlyMinimized", m_groupManager->showOnlyMinimized());
        changed = true;
    }
 
    if (changed) 
    {
        emit settingsChanged();
        emit configNeedsSaving();
    }
}
void STasksApplet::configuration()
{	
    KConfigGroup cg = config();

    m_taskSpacing = cg.readEntry("taskSpacing", 5);
    m_layout->setSpacing(m_taskSpacing);
    m_iconScale = cg.readEntry("iconScale", 80);
    m_lights = cg.readEntry("lights", true);
    m_expandTasks = cg.readEntry("expandTasks", false);
    m_keepExpanded = cg.readEntry("keepExpanded", 0);
    m_shape = cg.readEntry("shape", 0);
    m_expandingSize = cg.readEntry("expandingSize", 175);
    m_lightColor = cg.readEntry("lightColor", QColor(78, 196, 249, 200));

    m_animationDuration = cg.readEntry("animationDuration", 175);
    m_sortingStrategy = static_cast<TaskManager::GroupManager::TaskSortingStrategy>(cg.readEntry("sortingStrategy", static_cast<int>(TaskManager::GroupManager::AlphaSorting)));
    m_groupingStrategy = static_cast<TaskManager::GroupManager::TaskGroupingStrategy>(cg.readEntry("groupingStrategy", static_cast<int>(TaskManager::GroupManager::ProgramGrouping)));

    m_groupManager->setOnlyGroupWhenFull(false);
    m_groupManager->setSortingStrategy(m_sortingStrategy);
    m_groupManager->setGroupingStrategy(m_groupingStrategy);
    
    m_groupManager->setShowOnlyCurrentDesktop(cg.readEntry("showOnlyCurrentDesktop", false));
    m_groupManager->setShowOnlyCurrentScreen(cg.readEntry("showOnlyCurrentScreen", false));
    m_groupManager->setShowOnlyMinimized(cg.readEntry("showOnlyMinimized", false));
    m_groupManager->reconnect();

    foreach(STasksItem* item, m_items) 
    {
	item->setSize(m_iconSize);
    }
}

K_EXPORT_PLASMA_APPLET(stasks, STasksApplet)
#include "stasks_applet.moc"