/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file linear_expression.cpp
 * \brief Implementation of the bear::expr::linear_expression class.
 * \author Julien Jorge.
 */
#include "expr/linear_expression.hpp"

#include "expr/linear_constant.hpp"
#include "expr/linear_equality.hpp"
#include "expr/logical_not.hpp"

#include <algorithm>

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 */
bear::expr::linear_expression::linear_expression()
  : m_expr( new linear_constant(0) )
{

} // linear_expression::linear_expression()

/*----------------------------------------------------------------------------*/
/**
 * \brief Contructor.
 * \param e An expression to init from.
 */
bear::expr::linear_expression::linear_expression
( const base_linear_expression& e )
  : m_expr(e.clone())
{

} // linear_expression::linear_expression()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy contructor.
 * \param that The instance to copy from.
 */
bear::expr::linear_expression::linear_expression
( const linear_expression& that )
  : m_expr( that.m_expr->clone() )
{

} // linear_expression::linear_expression()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::expr::linear_expression::~linear_expression()
{
  delete m_expr;
} // linear_expression::~linear_expression()

/*----------------------------------------------------------------------------*/
/**
 * \brief Evaluate the expression.
 */
double bear::expr::linear_expression::evaluate() const
{
  return m_expr->evaluate();
} // linear_expression::evaluate()

/*----------------------------------------------------------------------------*/
/**
 * \brief Evaluate the expression.
 */
bear::expr::linear_expression::operator double() const
{
  return evaluate();
} // linear_expression::operator double()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assignment.
 * \param that The instance to copy from.
 */
bear::expr::linear_expression&
bear::expr::linear_expression::operator=( const linear_expression& that )
{
  linear_expression tmp(that);

  std::swap(tmp.m_expr, m_expr);

  return *this;
} // linear_expression::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Equality.
 * \param that The expression to compare to.
 */
bear::expr::boolean_expression bear::expr::linear_expression::operator==
( const linear_expression& that ) const
{
  return linear_equality( *this, that );
} // linear_expression::operator==()

/*----------------------------------------------------------------------------*/
/**
 * \brief Disequality.
 * \param that The expression to compare to.
 */
bear::expr::boolean_expression bear::expr::linear_expression::operator!=
( const linear_expression& that ) const
{
  return logical_not( linear_equality( *this, that ) );
} // linear_expression::operator!=()
