/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file string_to_arg.tpp
 * \brief Implementation of the bear::text_interface::string_to_arg class.
 * \author Julien Jorge.
 */

#include <sstream>
#include <stdexcept>

#include <claw/logger.hpp>
#include <claw/string_algorithm.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string to the equivalent value to pass as an argument for a
 *        character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
template<typename Context, typename T>
typename bear::text_interface::string_to_arg<Context, T>::result_type
bear::text_interface::string_to_arg<Context, T>::convert
( const Context& c, const std::string& arg )
{
  result_type result;
  std::istringstream iss(arg);
  iss >> result;

  if ( !iss || (iss.rdbuf()->in_avail() != 0) )
    {
      claw::logger << claw::log_warning << "Can't convert '" << arg << "'"
                   << std::endl;
      throw std::invalid_argument("Can't convert '" + arg + "'");
    }

  return result;
} // string_to_arg::convert()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string to the equivalent value to pass as an argument for a
 *        character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
template<typename Context>
typename bear::text_interface::string_to_arg<Context, std::string>::result_type
bear::text_interface::string_to_arg<Context, std::string>::convert
( const Context& c, const std::string& arg )
{
  return arg;
} // string_to_arg::convert()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string to the equivalent value to pass as an argument for a
 *        character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
template<typename Context>
typename bear::text_interface::string_to_arg
<Context, const std::string>::result_type
bear::text_interface::string_to_arg<Context, const std::string>::convert
( const Context& c, const std::string& arg )
{
  return arg;
} // string_to_arg::convert()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string to the equivalent value to pass as an argument for a
 *        character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
template<typename Context>
typename bear::text_interface::string_to_arg
<Context, const std::string&>::result_type
bear::text_interface::string_to_arg<Context, const std::string&>::convert
( const Context& c, const std::string& arg )
{
  return arg;
} // string_to_arg::convert()




/*----------------------------------------------------------------------------*/
/**
 * \brief Convert a string to the equivalent sequence of value to pass as an
 *        argument for a character method.
 * \param c The context on which the conversion is done.
 * \param arg The string representation of the value.
 */
template<typename Context, typename Sequence>
typename
bear::text_interface::string_to_sequence_arg<Context, Sequence>::result_type
bear::text_interface::string_to_sequence_arg<Context, Sequence>::convert
( const Context& c, const std::string& arg )
{
  result_type result;

  if ( !arg.empty() )
    claw::text::split( result, arg.begin()+1, arg.end(), arg[0] );

  return result;
} // string_to_sequence_arg::convert()
