/*
    Bear Engine - Level editor

    Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/layer.cpp
 * \brief Implementation of the bf::layer class.
 * \author Julien Jorge
 */
#include "bf/layer.hpp"

#include "level_code_value.hpp"

#include "bf/item_class_pool.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param width The width of the layer.
 * \param height The height of the layer.
 * \param layer_type The type of the layer.
 */
bf::layer::layer
( unsigned int width, unsigned int height, const std::string& layer_type )
  : m_width(width), m_height(height), m_layer_type(layer_type)
{
  CLAW_PRECOND( width > 0 );
  CLAW_PRECOND( height > 0 );
} // layer::layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The layer to copy from.
 */
bf::layer::layer( const layer& that )
{
  assign(that);
} // layer::layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bf::layer::~layer()
{
  clear();
} // layer::~layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the size of the layer.
 * \param width The new width of the layer.
 * \param height The new height of the layer.
 */
void bf::layer::resize( unsigned int width, unsigned int height )
{
  CLAW_PRECOND( width > 0 );
  CLAW_PRECOND( height > 0 );

  m_width = width;
  m_height = height;
} // layer::resize()

/*----------------------------------------------------------------------------*/
/**
 * \brief Change the name of the class of the layer.
 * \param class_name The new name of the class.
 */
void bf::layer::set_class_name( const std::string& class_name )
{
  CLAW_PRECOND( !class_name.empty() );

  m_layer_type = class_name;
} // layer::set_class_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add an item in the layer.
 * \param item The item to add.
 */
void bf::layer::add_item( item_instance* item )
{
  m_item.insert(item);

  if ( evaluate_filters_on_item(*item) )
    m_filtered_item.insert(item);
} // layer::add_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove an item from the layer.
 * \param item The item to remove.
 * \remark \a item won't be <tt>delete</tt>'d.
 */
void bf::layer::remove_item( item_instance* item )
{
  CLAW_PRECOND( m_item.find(item) != m_item.end() );

  if ( evaluate_filters_on_item(*item) )
    m_filtered_item.erase(item);

  m_item.erase(item);
} // layer::remove_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the type of the layer.
 */
const std::string& bf::layer::get_class_name() const
{
  return m_layer_type;
} // layer::get_class_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the layer.
 */
unsigned int bf::layer::get_width() const
{
  return m_width;
} // layer::get_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the layer.
 */
unsigned int bf::layer::get_height() const
{
  return m_height;
} // layer::get_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the number of items.
 */
std::size_t bf::layer::get_items_count() const
{
  return m_item.size();
} // layer::get_items_count()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if there is no item in the layer.
 */
bool bf::layer::empty() const
{
  return m_item.empty();
} // layer::empty()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the layer contains a given item.
 * \param item The item to look for.
 */
bool bf::layer::contains( const item_instance& item ) const
{
  return m_item.find(const_cast<item_instance*>(&item)) != m_item.end();
} // layer::contains()

/*----------------------------------------------------------------------------*/
/**
 * \brief Test if the layer is valid.
 * Return the bad items.
 */
std::list<bf::item_instance*> bf::layer::check
( const std::map<std::string,item_instance*>& map_id )
{
  std::list<item_instance*> items;
  item_set_type::const_iterator it;

  for (it=m_item.begin(); it!=m_item.end(); ++it)
    if ( !(*it)->check(map_id) )
      items.push_back(*it);

  return items;
} // layer::check()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the layer.
 * \param f The file in which we compile.
 */
void bf::layer::compile( compiled_file& f ) const
{
  f << m_layer_type << m_width << m_height;

  std::map<std::string, unsigned int> id_to_int;
  std::list<item_instance*> referenced, not_referenced;
  std::list<item_instance*>::const_iterator iti;
  item_set_type::const_iterator it;

  for (it=m_item.begin(); it!=m_item.end(); ++it)
    if ( (*it)->get_id().empty() )
      not_referenced.push_front(*it);
    else
      referenced.push_back(*it);

  sort_and_identify(referenced, id_to_int);

  // declaration of referenced items
  if ( !referenced.empty() )
    {
      f << bear::level_code_value::item_declaration << referenced.size();

      for (iti=referenced.begin(); iti!=referenced.end(); ++iti)
        f << (*iti)->get_class().get_class_name();

      // definition of referenced items
      for (iti=referenced.begin(); iti!=referenced.end(); ++iti)
        {
          f << bear::level_code_value::item_definition;
          (*iti)->compile( f, id_to_int );
        }
    }

  // not referenced items
  for (iti=not_referenced.begin(); iti!=not_referenced.end(); ++iti)
    {
      f << bear::level_code_value::base_item
        << (*iti)->get_class().get_class_name();
      (*iti)->compile( f, id_to_int );
    }
} // layer::compile()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the first filter.
 */
bf::layer::const_filter_iterator bf::layer::filter_begin() const
{
  return m_filters.begin();
} // layer::filter_begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator just past the last filter.
 */
bf::layer::const_filter_iterator bf::layer::filter_end() const
{
  return m_filters.end();
} // layer::filter_end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the set of items.
 * \remark Iterators are invalidated when inserting a filter if there was no
 *         filter before and when removing a filter if the pointed item does not
 *         verify any of the remaining filters or if there is no more filters.
 */
bf::layer::item_iterator bf::layer::item_begin()
{
  if ( m_filters.empty() )
    return m_item.begin();
  else
    return m_filtered_item.begin();
} // layer::item_begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator past the end of the set of items.
 */
bf::layer::item_iterator bf::layer::item_end()
{
  if ( m_filters.empty() )
    return m_item.end();
  else
    return m_filtered_item.end();
} // layer::item_end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the set of items.
 * \remark Iterators are invalidated when inserting a filter if there was no
 *         filter before and when removing a filter if the pointed item does not
 *         verify any of the remaining filters or if there is no more filters.
 */
bf::layer::const_item_iterator bf::layer::item_begin() const
{
  if ( m_filters.empty() )
    return m_item.begin();
  else
    return m_filtered_item.begin();
} // layer::item_begin()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator past the end of the set of items.
 */
bf::layer::const_item_iterator bf::layer::item_end() const
{
  if ( m_filters.empty() )
    return m_item.end();
  else
    return m_filtered_item.end();
} // layer::item_end()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the set of items.
 * \remark These iterators iterates over all items, whatever the filters are.
 */
bf::layer::item_iterator bf::layer::item_begin_no_filter()
{
  return m_item.begin();
} // layer::item_begin_no_filter()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator past the end of the set of items.
 */
bf::layer::item_iterator bf::layer::item_end_no_filter()
{
  return m_item.end();
} // layer::item_end_no_filter()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator on the set of items.
 * \remark These iterators iterates over all items, whatever the filters are.
 */
bf::layer::const_item_iterator bf::layer::item_begin_no_filter() const
{
  return m_item.begin();
} // layer::item_begin_no_filter()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get an iterator past the end of the set of items.
 */
bf::layer::const_item_iterator bf::layer::item_end_no_filter() const
{
  return m_item.end();
} // layer::item_end_no_filter()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assignment operator.
 * \param that The instance to copy from.
 */
bf::layer& bf::layer::operator=( const layer& that )
{
  if ( &that != this )
    {
      clear();
      assign( that );
    }

  return *this;
} // layer::operator=()

/*----------------------------------------------------------------------------*/
/**
 * \brief Add a filter on the items of the layer.
 * \param filter The filter to apply on the items.
 */
void bf::layer::add_filter( const item_filter& filter )
{
  m_filters.push_back(filter);

  item_set_type::iterator it;

  for (it=m_item.begin(); it!=m_item.end(); ++it)
    if ( filter.evaluate(**it) )
      m_filtered_item.insert(*it);
} // layer::add_filter()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove a filter on the items of the layer.
 * \param filter The filter to remove.
 */
void bf::layer::remove_filter( const item_filter& filter )
{
  const filter_list::iterator itf =
    std::find(m_filters.begin(), m_filters.end(), filter);

  if ( itf!=m_filters.end() )
    m_filters.erase( itf );

  item_set_type::iterator it(m_filtered_item.begin());

  while ( it!=m_filtered_item.end() )
    if ( !evaluate_filters_on_item(**it) )
      {
        item_set_type::iterator tmp(it);
        ++it;
        m_filtered_item.erase(tmp);
      }
    else
      ++it;
} // layer::remove_filter()

/*----------------------------------------------------------------------------*/
/**
 * \brief Remove all items.
 */
void bf::layer::clear()
{
  item_set_type::iterator it;

  for (it=m_item.begin(); it!=m_item.end(); ++it)
    delete *it;

  m_item.clear();
} // layer::clear()

/*----------------------------------------------------------------------------*/
/**
 * \brief Assign a layer to this one.
 * \param that The instance to copy from.
 */
void bf::layer::assign( const layer& that )
{
  CLAW_PRECOND( m_item.empty() );
  CLAW_PRECOND( m_filters.empty() );
  CLAW_PRECOND( &that != this );

  m_width = that.m_width;
  m_height = that.m_height;
  m_layer_type = that. m_layer_type;

  for ( filter_list::const_iterator it=that.m_filters.begin();
        it!=that.m_filters.end(); ++it )
    m_filters.push_back( it->copy() );

  for ( item_set_type::const_iterator it=that.m_item.begin();
        it!=that.m_item.end(); ++it )
    m_item.insert( new item_instance( **it ) );
} // layer::assign()

/*----------------------------------------------------------------------------*/
/**
 * \brief Sort referenced items such that the referenced is before the
 *        referencer and give an identifier to each item.
 * \param referenced (in/out) The referenced items.
 * \param id_to_int (out) A table giving, for each identifier of item, the
 *        position of the corresponding item in referenced.
 */
void bf::layer::sort_and_identify
( std::list<item_instance*>& referenced,
  std::map<std::string, unsigned int>& id_to_int ) const
{
  std::map<std::string, item_instance*> items;
  std::list<item_instance*>::const_iterator it;

  for (it=referenced.begin(); it!=referenced.end(); ++it)
    items[(*it)->get_id()] = *it;

  referenced.clear();

  while ( !items.empty() )
    sort_by_dependency( items.begin()->second, items, referenced );

  unsigned int index = 0;

  for (it=referenced.begin(); it!=referenced.end(); ++it, ++index)
    id_to_int[(*it)->get_id()] = index;
} // layer::sort_and_identify()

/*----------------------------------------------------------------------------*/
/**
 * \brief Sort referenced items such that the referenced is before the
 *        referencer.
 * \param p Pointer on the item that references an other item.
 * \param items The set of referenced items associated with their name.
 * \param referenced (out) The ordered referenced items.
 */
void bf::layer::sort_by_dependency
( item_instance* p, std::map<std::string, item_instance*>& items,
  std::list<item_instance*>& referenced ) const
{
  const item_class& the_class(p->get_class());

  items.erase(p->get_id());

  std::list<std::string> f;
  the_class.get_field_names_in_hierarchy(f);
  std::list<std::string>::const_iterator it;

  for (it=f.begin(); it!=f.end(); ++it)
    {
      const bf::type_field& field(the_class.get_field(*it));

      if ( field.get_field_type() == type_field::item_reference_field_type )
        if ( p->has_value(field) )
          {
            if ( field.is_list() )
              {
                std::list<item_reference_type> v;
                p->get_value(field.get_name(), v);

                for( ; !v.empty(); v.pop_front() )
                  if ( items.find( v.front().get_value() ) != items.end() )
                    sort_by_dependency
                      (items[v.front().get_value()], items, referenced);
              }
            else
              {
                item_reference_type v;
                p->get_value(field.get_name(), v);
                if ( items.find(v.get_value()) != items.end() )
                  sort_by_dependency(items[v.get_value()], items, referenced);
              }
          }
    }

  referenced.push_back(p);
} // layer::sort_by_dependency()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if an item validates at least one filter.
 * \param item The item on which the filters are evaluated.
 */
bool bf::layer::evaluate_filters_on_item( const item_instance& item ) const
{
  bool result = false;
  filter_list::const_iterator it;

  for ( it=m_filters.begin(); !result && (it!=m_filters.end()); ++it )
    result = it->evaluate(item);

  return result;
} // layer::evaluate_filters_on_item()
